/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

#include <string.h>

#include "command_pv.h"
#include <termui/console.h>

static void command_next_arg(char **line, char quote)
{
  char		*end = *line;

  while (1)
    {
      switch (*end)
	{
	case (' '):
	case ('\t'):
	case ('\n'):
	  if (quote)
	    break;
	  *end = '\0';
	  *line = end + 1;
	  return;

	case ('\0'):
	  *line = end;
	  return;

	case ('\''):
	case ('"'):
	  if (quote != *end)
	    break;
	  *end = '\0';
	  *line = end + 1;
	  return;
	}

      end++;
    }
}

int command_split_args(char *line, char *argv[])
{
  int		argc = 0;

  while (*line && argc < COMMAND_MAX_ARGS)
    {
      char	*arg;

      line += strspn(line, " \t\n");

      switch (*line)
	{
	case ('\0'):
	  return argc;

	case ('\''):
	case ('"'):
	  line++;
	  arg = line;
	  command_next_arg(&line, line[-1]);
	  break;

	default:
	  arg = line;
	  command_next_arg(&line, 0);
	}

      argv[argc++] = arg;
    }

  return argc;
}

int
command_args_parse(int acl, const struct command_entry_s *e,
		   void *argctx, int argc, char **argv,
		   char **paramv, unsigned int *mask,
		   struct console_ctx_s *con)
{
  struct command_args_s	*desc = e->args_desc;
  int			args_ok = 1;
  int			paramc = 0;
  unsigned int		x, used = 0;
  unsigned int		i;

  while (argc)
    {
      /* look for possible option in args list */

      if (args_ok)
	{
	  if (!strcmp(*argv, "--"))
	    {
	      args_ok = 0;
	      argc--, argv++;
	      continue;
	    }

	  for (i = 0; desc[i].str_short; i++)
	    if (**argv && (desc[i].acl & acl) &&
		!((1 << i) & e->args_disabled) &&
		(!strcmp(desc[i].str_short, *argv)
		 || !strcmp(desc[i].str_long, *argv)))
	      break;

	  if (desc[i].str_short)
	    {	  
	      int	res;

	      /* test dependency mask */
	      if (~used & desc[i].depend & ~e->args_disabled)
		{
		  int	j;

		  /* display list of excluded options */
		  console_printf(con, "error: Use of `%1A%s%A' needs: %1A", *argv);

		  for (j = 0; desc[j].str_short; j++)
		    if ((1 << j) & desc[i].depend & ~e->args_disabled)
		      console_printf(con, "%s %s ",
			      desc[j].str_short, desc[j].str_long);
		  console_printf(con, "%A\n");

		  return -1;
		}

	      /* test exclusion mask */
	      if (used & desc[i].exclude)
		{
		  int	j;

		  /* display list of excluded options */
		  console_printf(con, "error: Use of `%1A%s%A' not allowed with: %1A", *argv);

		  for (j = 0; desc[j].str_short; j++)
		    if ((1 << j) & desc[i].exclude & ~e->args_disabled)
		      console_printf(con, "%s %s ",
			      desc[j].str_short, desc[j].str_long);
		  console_printf(con, "%A\n");

		  return -1;
		}

	      /* set entry exclusion bits */
	      used |= (1 << i);

	      /* determine action for this arg */
	      res = desc[i].func(con, desc + i, argctx, argc - 1, argv + 1);

	      if (res < 0)
		{
		  console_printf(con, "error: Invalid parameter for `%1A%s%A' option\n", *argv);
		  return -1;
		}

	      /* skip to next arg */
	      argc -= res + 1;
	      argv += res + 1;

	      continue;
	    }

	  if (**argv == '-')
	    {
	      /* error, invalid option */
	      console_printf(con, "error: Invalid option: `%1A%s%A'\n", *argv);
	      return -1;
	    }
	}

      /* add to param list */
      if (paramc >= e->args_max)
	{
	  console_printf(con, "error: Too many extra arguments `%1A%s%A ...'\n", *argv);
	  return -1;
	}

      paramv[paramc++] = *argv;
      argc--, argv++;
    }

  if (paramc < e->args_min)
    {
      console_printf(con, "error: %u extra argument%s expected, found %u\n",
		     e->args_min, e->args_min > 1 ? "s" : "", argc);
      return -1;
    }

  *mask = used;

  while ((x = ~used & e->args_mandatory))
    {
      int	j;

      x ^= x & (x - 1);

      /* display list of excluded options */
      console_printf(con, "error: Missing option: %1A");

      for (j = 0; desc[j].str_short; j++)
	if ((1 << j) & x & ~e->args_disabled)
	  console_printf(con, "%s %s ", desc[j].str_short, desc[j].str_long);
      console_printf(con, "%A\n");

      used |= x;

      paramc = -1;
    }

  return (paramc);
}

