/*
    This file is part of libtermui.

    libtermui is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    libtermui is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with libtermui.  If not, see <http://www.gnu.org/licenses/>.

    Copyright 2006, Alexandre Becoulet <alexandre.becoulet@free.fr>

*/

/*
 * This example show how to use libtermui console and builtin command
 * parsing feature over a TTY.
 */

#include <termui/console.h>
#include <termui/command.h>

#include <stdlib.h>
#include <errno.h>

/************************************************************************/

COMMAND_ARGS_CONTEXT(test)
{
  int		a;
};

COMMAND_ARGS_PARSE_FUNC(test_a)
{
  COMMAND_ARGS_CONTEXT(test)	*ctx = argctx;

  if (argc >= 1)
    {
      ctx->a = atoi(argv[0]);
      return 1;
    }

  return 0;
}

COMMAND_ARGS_PARSE_FUNC(test_b)
{
  return 0;
}

COMMAND_ARGS_PARSE_FUNC(test_c)
{
  return 0;
}

COMMAND_ARGS_CLEANUP_FUNC(test)
{
}

COMMAND_ARGS_DECL(test) =
{
  /* arg 0x01. The "a" arg exclude itself (0x01), it cant be used more than once */
  COMMAND_ARGS_ENTRY(test_a, "--aaa", "-a", 0x01, 0x00, COMMAND_ACL_ALL, "The a option"),

  /* arg 0x02. The "b" arg exclude itself (0x02) and "a" (0x01) */
  COMMAND_ARGS_ENTRY(test_b, "--bbb", "-b", 0x03, 0x00, COMMAND_ACL_ALL, "The b option"),

  /* arg 0x04. The "c" arg depends on arg "a" (0x01) */
  COMMAND_ARGS_ENTRY(test_c, "--ccc", "-c", 0x00, 0x01, COMMAND_ACL_ALL, "The c option"),

  COMMAND_LIST_END
};

static COMMAND_CMD_FUNC(test)
{
  COMMAND_ARGS_CONTEXT(test)	*ctx = argctx;
  unsigned int	i;

  console_printf(con,   "Option value for `a': %i\n", ctx->a);

  for (i = 0; i < argc; i++)
    console_printf(con, "Extra parameter     : %s\n", argv[i]);

  return 0;
}

/************************************************************************/

static COMMAND_GROUP_DECL(sub1) =
{
  COMMAND_CMD_ENTRYA(test,	"test", COMMAND_ACL_ALL,   "Test command", NULL, 0, 10, test, 0, 0),
  COMMAND_LIST_END
};

static COMMAND_GROUP_DECL(sub2) =
{
  COMMAND_GROUP_ENTRY(sub1, "group", COMMAND_ACL_ALL),
  COMMAND_LIST_END
};

/************************************************************************/

static COMMAND_CMD_FUNC(add)
{
  console_register(con, con_group_sub2);

  return 0;
}

/************************************************************************/

static COMMAND_CMD_FUNC(del)
{
  command_unregister(con_group_sub2);

  return 0;
}

/************************************************************************/

static COMMAND_GROUP_DECL(root) =
{
  /* use builtin commands entris defined in libtermui */
  COMMAND_CMD_ENTRY (console_builtin_help, "help", COMMAND_ACL_ALL, "Display help about commands", NULL, 0, 1),
  COMMAND_CMD_ENTRY (console_builtin_list, "list", COMMAND_ACL_ALL, "List available commands", NULL, 0, 0),
  COMMAND_CMD_ENTRY (console_builtin_quit, "quit", COMMAND_ACL_ALL, "Leave console", NULL, 0, 0),

  /* add 4 commands _with_ arguments parsing functions */
  COMMAND_CMD_ENTRYA(test,		"test", COMMAND_ACL_ALL,   "Test command", NULL, 0, 10, test, 0, 0),
  COMMAND_CMD_ENTRYA(test,		"testa", COMMAND_ACL_ALL,  "Test command", NULL, 1, 10, test, 0, 0),
  COMMAND_CMD_ENTRYA(test,		"testaa", COMMAND_ACL_ALL, "Test command", NULL, 1, 1,  test, 0, 0),
  COMMAND_CMD_ENTRYA(test,		"testb", COMMAND_ACL_ALL,  "Test command", NULL, 0, -1, test, 0, 0),

  /* add 2 commands */
  COMMAND_CMD_ENTRY (add,		"add", COMMAND_ACL_ALL,    "Register more commands", NULL, 0, 0),
  COMMAND_CMD_ENTRY (del,		"del", COMMAND_ACL_ALL,    "Unregister commands", NULL, 0, 0),

  COMMAND_LIST_END
};

/************************************************************************/

int main()
{
  struct console_ctx_s	*con;
  int res;

  COMMAND_EMPTY_ROOT(root);

  /* open console with stdin/stdout */
  if (!(con = console_alloc(&root, 0, 1, getenv("TERM"))))
    return -1;

  /* set console prompt */
  console_set_prompt(con, "[%31Aconsole%A] ");

  /* show welcome message */
  console_printf(con,
		 "\n"
		 "  Welcome to libtermui example." "\n"
		 "  Copyright Alexandre Becoulet (c) 2006" "\n"
		 "\n"
		 "  You may type `%1Ahelp%A'\n"
		 "\n");

  /* register root command list */
  console_register(con, con_group_root);

  /* process user commands */
  do {
    res = console_process(con);
  } while (res != -EIO && res != -ECANCELED);

  console_printf(con, "Bye.\n");

  /* free allocated resources */
  console_free(con);

  return 0;
}

