/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for transport of the Ogg format
 *
 * Copyright (C) 2006, 2007, 2008 Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Ogg::Transport implementation                                             */
/*                                                                           */
/*===========================================================================*/
/********************************************************************
 * Certain sections of this file, commented as "xiph code",         *
 * are copied almost verbatim from the libogg SOURCE CODE.          *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING-XIPH'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.  *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2002             *
 * by the Xiph.Org Foundation http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************/
#ifdef __GNUG__
#pragma implementation "Transport.H"
#pragma implementation "TransportToLogical.H"
#pragma implementation "TransportImpl.H"
#endif

#include	<TransportImpl.H>
#include	<Ogg/Debug.H>

namespace Ogg
{

  //--checksum------------------------------------------------------------------------------
  // xiph code
  // {
  const
  unsigned int PageImpl::crc_lookup[256] = 
    {
      0x00000000,0x04c11db7,0x09823b6e,0x0d4326d9,
      0x130476dc,0x17c56b6b,0x1a864db2,0x1e475005,
      0x2608edb8,0x22c9f00f,0x2f8ad6d6,0x2b4bcb61,
      0x350c9b64,0x31cd86d3,0x3c8ea00a,0x384fbdbd,
      0x4c11db70,0x48d0c6c7,0x4593e01e,0x4152fda9,
      0x5f15adac,0x5bd4b01b,0x569796c2,0x52568b75,
      0x6a1936c8,0x6ed82b7f,0x639b0da6,0x675a1011,
      0x791d4014,0x7ddc5da3,0x709f7b7a,0x745e66cd,
      0x9823b6e0,0x9ce2ab57,0x91a18d8e,0x95609039,
      0x8b27c03c,0x8fe6dd8b,0x82a5fb52,0x8664e6e5,
      0xbe2b5b58,0xbaea46ef,0xb7a96036,0xb3687d81,
      0xad2f2d84,0xa9ee3033,0xa4ad16ea,0xa06c0b5d,
      0xd4326d90,0xd0f37027,0xddb056fe,0xd9714b49,
      0xc7361b4c,0xc3f706fb,0xceb42022,0xca753d95,
      0xf23a8028,0xf6fb9d9f,0xfbb8bb46,0xff79a6f1,
      0xe13ef6f4,0xe5ffeb43,0xe8bccd9a,0xec7dd02d,
      0x34867077,0x30476dc0,0x3d044b19,0x39c556ae,
      0x278206ab,0x23431b1c,0x2e003dc5,0x2ac12072,
      0x128e9dcf,0x164f8078,0x1b0ca6a1,0x1fcdbb16,
      0x018aeb13,0x054bf6a4,0x0808d07d,0x0cc9cdca,
      0x7897ab07,0x7c56b6b0,0x71159069,0x75d48dde,
      0x6b93dddb,0x6f52c06c,0x6211e6b5,0x66d0fb02,
      0x5e9f46bf,0x5a5e5b08,0x571d7dd1,0x53dc6066,
      0x4d9b3063,0x495a2dd4,0x44190b0d,0x40d816ba,
      0xaca5c697,0xa864db20,0xa527fdf9,0xa1e6e04e,
      0xbfa1b04b,0xbb60adfc,0xb6238b25,0xb2e29692,
      0x8aad2b2f,0x8e6c3698,0x832f1041,0x87ee0df6,
      0x99a95df3,0x9d684044,0x902b669d,0x94ea7b2a,
      0xe0b41de7,0xe4750050,0xe9362689,0xedf73b3e,
      0xf3b06b3b,0xf771768c,0xfa325055,0xfef34de2,
      0xc6bcf05f,0xc27dede8,0xcf3ecb31,0xcbffd686,
      0xd5b88683,0xd1799b34,0xdc3abded,0xd8fba05a,
      0x690ce0ee,0x6dcdfd59,0x608edb80,0x644fc637,
      0x7a089632,0x7ec98b85,0x738aad5c,0x774bb0eb,
      0x4f040d56,0x4bc510e1,0x46863638,0x42472b8f,
      0x5c007b8a,0x58c1663d,0x558240e4,0x51435d53,
      0x251d3b9e,0x21dc2629,0x2c9f00f0,0x285e1d47,
      0x36194d42,0x32d850f5,0x3f9b762c,0x3b5a6b9b,
      0x0315d626,0x07d4cb91,0x0a97ed48,0x0e56f0ff,
      0x1011a0fa,0x14d0bd4d,0x19939b94,0x1d528623,
      0xf12f560e,0xf5ee4bb9,0xf8ad6d60,0xfc6c70d7,
      0xe22b20d2,0xe6ea3d65,0xeba91bbc,0xef68060b,
      0xd727bbb6,0xd3e6a601,0xdea580d8,0xda649d6f,
      0xc423cd6a,0xc0e2d0dd,0xcda1f604,0xc960ebb3,
      0xbd3e8d7e,0xb9ff90c9,0xb4bcb610,0xb07daba7,
      0xae3afba2,0xaafbe615,0xa7b8c0cc,0xa379dd7b,
      0x9b3660c6,0x9ff77d71,0x92b45ba8,0x9675461f,
      0x8832161a,0x8cf30bad,0x81b02d74,0x857130c3,
      0x5d8a9099,0x594b8d2e,0x5408abf7,0x50c9b640,
      0x4e8ee645,0x4a4ffbf2,0x470cdd2b,0x43cdc09c,
      0x7b827d21,0x7f436096,0x7200464f,0x76c15bf8,
      0x68860bfd,0x6c47164a,0x61043093,0x65c52d24,
      0x119b4be9,0x155a565e,0x18197087,0x1cd86d30,
      0x029f3d35,0x065e2082,0x0b1d065b,0x0fdc1bec,
      0x3793a651,0x3352bbe6,0x3e119d3f,0x3ad08088,
      0x2497d08d,0x2056cd3a,0x2d15ebe3,0x29d4f654,
      0xc5a92679,0xc1683bce,0xcc2b1d17,0xc8ea00a0,
      0xd6ad50a5,0xd26c4d12,0xdf2f6bcb,0xdbee767c,
      0xe3a1cbc1,0xe760d676,0xea23f0af,0xeee2ed18,
      0xf0a5bd1d,0xf464a0aa,0xf9278673,0xfde69bc4,
      0x89b8fd09,0x8d79e0be,0x803ac667,0x84fbdbd0,
      0x9abc8bd5,0x9e7d9662,0x933eb0bb,0x97ffad0c,
      0xafb010b1,0xab710d06,0xa6322bdf,0xa2f33668,
      0xbcb4666d,0xb8757bda,0xb5365d03,0xb1f740b4
    };
  // }
  // xiph code
  
  unsigned int
  PageImpl::checksum()
  {
    /* zero for (re)computing checksum */
    header_[22]=0;
    header_[23]=0;
    header_[24]=0;
    header_[25]=0;
  
    // xiph code
    // {
    unsigned int crc_reg=0;
    size_t i;

    for(i=0;i<headerSize_;i++)
      crc_reg=(crc_reg<<8)^crc_lookup[((crc_reg >> 24)&0xff)^header_[i]];
    for(i=0;i<bodySize_;i++)
      crc_reg=(crc_reg<<8)^crc_lookup[((crc_reg >> 24)&0xff)^body_[i]];
    // }
    // xiph code


    return(crc_reg);
  }


  bool
  PageEncapser::gotLastPacket()
  {
    pthread_mutex_lock(&lockLast_);
    {
      while( !gotLastPacket_ )
	pthread_cond_wait(&condLast_, &lockLast_);
    }
    pthread_mutex_unlock(&lockLast_);

    return(true);
  }

  void
  PageEncapser::gotLastPacket(
			      bool
			      )
  {
    pthread_mutex_lock(&lockLast_);
    {
      gotLastPacket_ = true;
      pthread_cond_signal(&condLast_);
    }
    pthread_mutex_unlock(&lockLast_);
  }

  void
  PageEncapser::setChecksum()
  {
    unsigned int crc_reg = PageImpl::checksum();

    header_[22]=(unsigned char)(crc_reg&0xff);
    header_[23]=(unsigned char)((crc_reg>>8)&0xff);
    header_[24]=(unsigned char)((crc_reg>>16)&0xff);
    header_[25]=(unsigned char)((crc_reg>>24)&0xff);
  }

  bool
  PageDecapser::checksumOK()
  {
    unsigned int cksum = (
			  header_[22] |
			  (header_[23]<<8) |
			  (header_[24]<<16) |
			  (header_[25]<<24)
			  );

    if ( cksum == PageImpl::checksum() )
      return true;

    header_[22]=(unsigned char)(cksum&0xff);
    header_[23]=(unsigned char)((cksum>>8)&0xff);
    header_[24]=(unsigned char)((cksum>>16)&0xff);
    header_[25]=(unsigned char)((cksum>>24)&0xff);
    
    return false;
  }

  //--Transport-----------------------------------------------------------------------------
  bool
  Transport::laterThan (
			const Page &		p1
			,const Page &		p2
			)
  {
    if ( p1.isHeader() != p2.isHeader() )
      return( !p1.isHeader() );

    if ( p1.isHeader() )
      {
	if ( p1.pageNo() != p2.pageNo() )
	  return( p1.pageNo() >= p2.pageNo() );
	else
	  return( p1.serialNo() >= p2.serialNo() );
      }

    return( p1.granulePosition() >= p2.granulePosition() );
  }

  bool
  PageImpl::operator < (const PageImpl & p2) const
  {
    Transport & transport = logical_->intf().transport();

    return(transport.laterThan(*this, p2));
  }

  //.......................................................

  Transport::Transport(
		       bool encapsulating
		       )
  {
    if ( encapsulating )
      impl_ = new TransportEncapser();
    else
      impl_ = new TransportDecapser();

    TransportImpl * impl = static_cast<TransportImpl *>(impl_);
    impl->intf_ = this;
  }

  TransportImpl::TransportImpl(
			       bool	encapsulating
			       )
    : encapsulating_(encapsulating)
      , terminate_(false)
      , gotAllStreams_(false)
  {
    pthread_mutex_init(&lockPages_, 0);
    pthread_cond_init(&condWork_, 0);
    pthread_cond_init(&condAll_, 0);
  }

  Transport::~Transport()
  {
    throw Transport::NoDeletion();
  }

  //.......................................................

  Logical *
  Transport::logical(
		     long	serialNo
		     )
  {
    return((static_cast<TransportImpl *>(impl_))->logical(serialNo));
  }

  Logical *
  TransportImpl::logical(
			 long	serialNo
			 )
  {
    Logical * logical = 0;

    pthread_mutex_lock(&lockPages_);
    {
      iPages i = pages_.find(serialNo);

      if ( i != pages_.end() )
	{
	  PageImpl & pg = *(i->second);

	  logical = &pg.logical_->intf();
	}
    }
    pthread_mutex_unlock(&lockPages_);

    
    return(logical);
  }

  //.......................................................

  void
  Transport::terminate()
  {
    (static_cast<TransportImpl *>(impl_))->terminate_ = true;
  }

  //.......................................................

  void
  Transport::loop()
  {
    TransportImpl * impl = static_cast<TransportImpl *>(impl_);

    while ( !impl->terminate_ )
      {
	impl->loop();
      }
  }

  void
  TransportEncapser::loop()
  {
    // Waiting for Logicals to register
    while ( logicalCount() < nToMux_ )
      ;

    // First round is first headers
    encapPages();

    // We should have full complement of first headers.
    // Write them all out.
    // This is strictly not necessary, but more efficient
    // and is mandated by the Ogg protocal.
    writePages(orderedPages_.size());

    while ( logicalCount() > 0 )
      {
	encapPages();
	writePages(1);
      }
    
    debug(true,
	  "reached end of loop"
	  );

    reset();
    return;
  }

  void
  TransportDecapser::loop()
  {
    PageDecapser * pg;

    while ( !gotAllStreams_ )
      {
	// insist on a second page from some stream
	pg = readPage();
	if ( pg )
	  decapPage(*pg);
      }

    while ( logicalCount() > 0 )
      {
	pg = readPage();
	if ( pg )
	  decapPage(*pg);
      }
    
    reset();
  }

  //.......................................................

  Transport &
  TransportToLogical::intf()
  {
    TransportImpl * impl = static_cast<TransportImpl *>(this);

    return(*impl->intf_);
  }

  //.......................................................

  bool
  TransportToLogical::encapsulating()
  {
    TransportImpl * impl = static_cast<TransportImpl *>(this);

    return(impl->encapsulating_);
  }


  //.......................................................
  
  // If this returns, it has succeeded
  void
  TransportToLogical::add(
			  LogicalToTransport &	logical
			  )
  {
    TransportImpl * impl = static_cast<TransportImpl *>(this);
    

    bool added = false;
    pthread_mutex_lock(&impl->lockPages_);
    try
      {
	TransportImpl::iPages i = impl->pages_.find(logical.intf().serialNo());
	if ( i == impl->pages_.end() )
	  {
	    // It doesn't get here if decapsulating.
	    // Because this is called by Logical() within logical()
	    // after we have already found the stream with the serialNo
	    // It is incumbant upon the Logical threads to synchronize
	    // So that this doesn't happen.
	    // This doesn't need a mutex because the timing is forgiving.
	    if ( impl->gotAllStreams_ )
	      throw Logical::MuxTooLate(logical.intf().serialNo());

	    // This here waits on workPage_ to be instantiated.
	    // In the meantime, logicalCount is being called by Transport.
	    // logicalCount checks to see whether there is any workPage_.
	    // If not, make one, and signal condPage_. There is a fight
	    // among the threads that are waiting for this. The winner
	    // takes the workPage_, and claims it for its Logical.
	    // All other threads keep waiting.
	    // This is a sync mechanism. Due to timing, you could
	    // end up with an unused allocated PageImpl. Which is OK.
	  
	    while ( !impl->workPage_ )
	      pthread_cond_wait(&impl->condWork_, &impl->lockPages_);

	    impl->workPage_->logical_ = &logical;
	    impl->workPage_->serialNo_ = logical.intf().serialNo();
	    impl->pages_[impl->workPage_->serialNo_] = impl->workPage_;

	    impl->workPage_ = 0;
	    added = true;
	    debug(true,
		  "Serial No. ", logical.intf().serialNo()
		  ," stream added"
		  );
	  }
      }
    catch( ... )
      {
	pthread_mutex_unlock(&impl->lockPages_);
	throw;
      }
    pthread_mutex_unlock(&impl->lockPages_);
      
    if ( encapsulating() && !added )
      throw Logical::SerialNoAlreadyUsed(logical.intf().serialNo());

    return;
  }

  //.......................................................

  long
  Transport::serialNo(long	serialno)
  {
    return((static_cast<TransportImpl *>(impl_))->serialNo(serialno));
  }

  long
  TransportEncapser::serialNo(long	serialno)
  {
    iPages i;

    pthread_mutex_lock(&lockPages_);
    {
      for ( i = pages_.begin()
	      ;
	    i != pages_.end()
	      ;
	    ++i
	    )
	{
	  PageImpl & pg = *(i->second);
	  if ( serialno == pg.serialNo_ )
	    serialno = pg.serialNo_ + 1;
	}
    }
    pthread_mutex_unlock(&lockPages_);
    
    return(serialno);
  }
   
  long
  TransportDecapser::serialNo(long serialno)
  {
    pthread_mutex_lock(&lockPages_);
    {
      iPages i;

      // all pages scanned, but there could be new ones
      while ( !gotAllStreams_ )
	pthread_cond_wait(&condAll_, &lockPages_);

      for ( i = pages_.begin()
	      ;
	    i != pages_.end()
	      ;
	    ++i
	    )
	{
	  PageImpl & pg = *(i->second);
	  if (
	      (pg.serialNo_ > serialno)
	      &&
	      !pg.logical_ )
	    {
	      // found an unclaimed page above serialno
	      pthread_mutex_unlock(&lockPages_);
	      return(pg.serialNo_);
	    }
	}
    }
    pthread_mutex_unlock(&lockPages_);
    
    return(-1);
  }

  //.......................................................

  void
  TransportToLogical::enterMux()
  {
    TransportEncapser * impl = static_cast<TransportEncapser *>(this);

    pthread_mutex_lock(&impl->lockPages_);
    {
      impl->nToMux_++;
    }
    pthread_mutex_unlock(&impl->lockPages_);
  }

  void
  TransportToLogical::exitMux()
  {
    TransportEncapser * impl = static_cast<TransportEncapser *>(this);

    pthread_mutex_lock(&impl->lockPages_);
    {
      impl->nToMux_--;
    }
    pthread_mutex_unlock(&impl->lockPages_);
  }

  void
  TransportToLogical::enterSelect()
  {
    TransportDecapser * impl = static_cast<TransportDecapser *>(this);

    pthread_mutex_lock(&impl->lockPages_);
    {
      impl->nToSelect_++;
      debug(true,
	    "transport nToSelect incremented to ", impl->nToSelect_
	    );
    }
    pthread_mutex_unlock(&impl->lockPages_);
  }

  void
  TransportToLogical::exitSelect()
  {
    TransportDecapser * impl = static_cast<TransportDecapser *>(this);

    pthread_mutex_lock(&impl->lockPages_);
    {
      impl->nToSelect_--;
      debug(true,
	    "transport nToSelect decremented to ", impl->nToSelect_
	    );
    }
    pthread_mutex_unlock(&impl->lockPages_);
  }

  void
  TransportToLogical::doneSelect()
  {
    TransportDecapser * impl = static_cast<TransportDecapser *>(this);

    pthread_mutex_lock(&impl->lockPages_);
    {
      impl->nDoneSelect_++;
      debug(true,
	    "transport nDoneSelect incremented to ", impl->nDoneSelect_
	    );
      pthread_cond_broadcast(&impl->condSelect_);
    }
    pthread_mutex_unlock(&impl->lockPages_);
  }

  //.......................................................

  const
  Packet *
  TransportToLogical::firstPacket(
				  long	serialNo
				  )
  {
    if ( encapsulating() )
      return(0);

    TransportImpl * impl = static_cast<TransportImpl *>(this);
    const Packet * pkt = 0;

    pthread_mutex_lock(&impl->lockPages_);
    {
      TransportImpl::iPages i = impl->pages_.find(serialNo);

      if ( i != impl->pages_.end() )
	{
	  PageDecapser * pg = static_cast<PageDecapser*>(i->second);

	  if ( !pg->logical_ )
	    {
	      // not yet claimed
	      // wait for first packet
	      pg->gotFirstPacket();

	      pkt = static_cast<const Packet *>(pg->lastPacket_);
	    }
	}
    }
    pthread_mutex_unlock(&impl->lockPages_);

    return(pkt);
  }

  //.......................................................

  void
  TransportToLogical::claim(
			    LogicalToTransport &	logical
			    )
  {
    if ( encapsulating() )
      // Treacherously, no exception thrown for implementer error
      return;

    TransportImpl * impl = static_cast<TransportImpl *>(this);

    pthread_mutex_lock(&impl->lockPages_);
    {
      TransportImpl::iPages i = impl->pages_.find(logical.intf().serialNo());

      if ( i != impl->pages_.end() )
	{
	  PageImpl & pg = *(i->second);

	  if ( !pg.logical_ )
	    {
	      pg.logical_ = &logical;
	    }
	  else
	    {
	      pthread_mutex_unlock(&impl->lockPages_);
	      throw Logical::StreamAlreadyClaimed(logical.intf().serialNo());
	    }
	}
      else
	{
	  pthread_mutex_unlock(&impl->lockPages_);
	  throw Logical::NoSuchStream(logical.intf().serialNo());
	}
    }
    pthread_mutex_unlock(&impl->lockPages_);
  }

  //.......................................................

  Packet *
  TransportToLogical::seek(
			   LogicalToTransport &	logical
			   ,PacketNo		packetNo
			   ,Position		granulePosition
			   )
  {
    TransportImpl * impl = static_cast<TransportImpl *>(this);

    PageDecapser * pg = 0;

    pthread_mutex_lock(&impl->lockPages_);
    {
      TransportImpl::iPages i = impl->pages_.find(logical.intf().serialNo());
      if ( i != impl->pages_.end() )
	{
	  pg = static_cast<PageDecapser *>(i->second);
	}
    }
    pthread_mutex_unlock(&impl->lockPages_);

    if ( !pg )
      throw Logical::SeekInternalError(logical.intf().serialNo());

    pg->seekNo_ = packetNo;
    pg->seekPosition_ = granulePosition;
    debug(true,
	  "Serial No. ", pg->serialNo_
	  ," at packet ", pg->packetNo_, "\n"
	  ,", seeking packet ", pg->seekNo_, ", position ", pg->seekPosition_
	  );
    pg->seeking(true);

    logical.wait();
    return(logical.deq());
  }

  //.......................................................

  void
  TransportToLogical::lastPacket(
				 long	serialNo
				 )
  {
    if ( !encapsulating() )
      return;

    TransportImpl * impl = static_cast<TransportImpl *>(this);
    PageEncapser * pg = 0;

    pthread_mutex_lock(&impl->lockPages_);
    {
      TransportImpl::iPages i = impl->pages_.find(serialNo);

      if ( i != impl->pages_.end() )
	{
	  pg = static_cast<PageEncapser*>(i->second);
	}
    }
    pthread_mutex_unlock(&impl->lockPages_);

    if ( pg )
      // wait for last packet to be deleted
      // so that Logical thread doesn't exit before its packet is deleted
      // which crashes the Transport thread
      pg->gotLastPacket();
  }

  //.......................................................

  void
  TransportImpl::reset()
  {
    pthread_mutex_lock(&lockPages_);
    {
      while ( !orderedPages_.empty() )
	{
	  orderedPages_.pop();
	}

      if ( !pages_.empty() )
	{
	  for ( iPages i = pages_.begin()
		  ;
		i != pages_.end()
		  ;
		++i
		)
	    {
	      delete i->second;
	    }
	    
	  pages_.clear();
	}
    }
    pthread_mutex_unlock(&lockPages_);

    gotAllStreams_ = false;
  }

  void
  TransportDecapser::reset()
  {
    // nToSelect_ remains the same.
    // This allows the same Logical streams to be selected
    // from the following bit stream.
    TransportImpl::reset();
    nDoneSelect_ = 0;
  }

  //.......................................................

  size_t
  TransportEncapser::logicalCount()
  {
    long	count = TransportImpl::logicalCount();

    debug(false,
	  "logical count = ", count
	  );
    
    pthread_mutex_lock(&lockPages_);
    {
      if ( !workPage_ )
	{
	  workPage_ = new PageEncapser();
	  pthread_cond_broadcast(&condWork_);
	}
    }
    pthread_mutex_unlock(&lockPages_);

    return(count);
  }
    
  //.......................................................

  void
  TransportEncapser::encapPages()
  {
    while (
	   (orderedPages_.size() < logicalCount())
	   )
      {
	pthread_mutex_lock(&lockPages_);
	try
	{
	  for ( iPages i = pages_.begin()
		  ;
		i != pages_.end()
		  ;
		++i
		)
	    {
	      PageEncapser & pg = *(static_cast<PageEncapser *>(i->second));

	      pthread_mutex_unlock(&lockPages_);
	      {
		if (! pg.ended()
		    &&
		    ! pg.ready_
		    )
		  {
		    pg.page();
		    if ( pg.ready_ )
		      {
			orderedPages_.push(&pg);
			debug(true,
			      "Serial No. ",pg.serialNo_
			      ," page No. ",pg.pageNo_
			      ," page pushed"
			      );
		      }
		  }
	      }
	      pthread_mutex_lock(&lockPages_);
	    }
	}
	catch (...)
	  {
	    // unlock the mutex before re-throwing
	    pthread_mutex_unlock(&lockPages_);
	    throw;
	  }

	pthread_mutex_unlock(&lockPages_);
      }

    if ( !gotAllStreams_ )
      gotAllStreams_ = true;

    return;
  }

  //.......................................................

  void
  TransportEncapser::writePages(
				long	npages
				)
  {
    while ( (npages > 0)
	    &&
	    !orderedPages_.empty() )
      {
	PageEncapser & pg = *(static_cast<const PageEncapser *>(orderedPages_.top()));
	if ( 0 < pg.numSegments_ )
	  {
	    intf().sendCallback(
				pg
				);
	    orderedPages_.pop();
	    npages--;
	    
	    pg.clear();
	  }
      }
  }

  //.......................................................

  size_t
  TransportImpl::logicalCount()
  {
    long	count = 0;

    pthread_mutex_lock(&lockPages_);
    {
      // delete any ended page
      for ( iPages i = pages_.begin()
	      ;
	    i != pages_.end()
	      ;
	    ++i
	    )
	{
	  PageImpl & pg = *(static_cast<PageImpl *>(i->second));

	  if ( pg.ended() )
	    {
	      debug(true,
		    "Serial No. ",pg.serialNo()
		    ," ended at page No. ",pg.pageNo()
		    ," deleting."
		    );
	      pages_.erase(i);
	      break;
	    }
	}

      for ( iPages i = pages_.begin()
	      ;
	    i != pages_.end()
	      ;
	    ++i
	    )
	{
	  PageImpl & pg = *(static_cast<PageImpl *>(i->second));

	  if ( !pg.ended() )
	    count++;
	}
    }
    pthread_mutex_unlock(&lockPages_);

    return(count);
  }
    
  //.......................................................

  PageDecapser *
  TransportDecapser::readPage()
  {
    // workPage_.header_ is at head of receiving area
    long long		sizeRead = (
				    workPage_->body_
				    - workPage_->header_
				    );
    PageDecapser *	pg = 0;

    try
      {
	while (true)
	  {
	    if ( sizeRead <= 0 )
	      {
		// do we have enough space?
		if ( (PageImpl::workSize_ -
		      // what has been used
		      (workPage_->header_
		       - workPage_->data_
		       )
		      )
		     < workPage_->size()	// fixed at maxPageSize_
		     )
		  {
		    // not enough, restart from beginning
		    debug(false,
			  "TransportDecapser::readPage, not enough space"
			  );
		    workPage_->header_ = workPage_->data_;
		    workPage_->body_ = workPage_->header_;

		    if ( pg )
		      {
			// save the partial temp page
			memcpy(workPage_->data_
			       ,pg->header_
			       ,pg->size()
			       );
			workPage_->header_ += pg->size();
			workPage_->body_ = workPage_->header_;
		
			pg->header_ = workPage_->data_;
			pg->body_ = pg->header_ + pg->headerSize_;
		      }
		  }

		sizeRead = intf().recvCallback(*workPage_);
		if ( sizeRead <= 0 )
		  {
		    // premature end of this Ogg stream
		    throw Transport::PrematureEndOfOggStream();
		  }
		// workPage_.body_ is set to beyond end of receiving area
		workPage_->body_ += sizeRead;
	      }
	    else
	      {
		// Try consuming all that has been read with the current temp page
		if ( !pg )
		  {
		    // instantiate a temp page
		    // for segmenting out an Ogg page
		    pg = new PageDecapser(*(static_cast<PageDecapser *>(workPage_)));
		  }
		// whether or not we have a header
		// we increase the page size by increasing the bodySize_
		pg->bodySize_ += sizeRead;
		sizeRead = 0;
		workPage_->header_ = workPage_->body_;

		if ( pg->headerSize_ <= 0 )
		  // Try to decode the header
		  pg->header();

		if ( pg->headerSize_ )
		  {
		    // header() decoded headerSize_ and actual pageSize_
		    if ( (sizeRead = ((sizeRead=(pg->headerSize_ + pg->bodySize_))
				      - (pg->pageSize_))) >= 0
			 )
		      {
			// set the workPage_ pointers for re-entry into while loop
			workPage_->header_ = workPage_->body_ - sizeRead;

			// we have a full page and maybe some left over
			// set actual body size
			pg->bodySize_ = pg->pageSize_ - pg->headerSize_;
		

			if ( pg->checksumOK() )
			  {
			    break;
			  }
			else
			  {
			    // checksum failed, skip forward
			    debug(true,
				  "Serial No. ", pg->serialNo_
				  ," checksum failed\n"
				  ,"sizeRead=",sizeRead," page size=",pg->size()
				  );
			    pg->skip();
			  }
		      }
		    else
		      sizeRead = 0;
		  }
	      }
	  }
      }
    catch(Transport::PrematureEndOfOggStream)
      {
	/* For all non-ended streams, send out a partial
	   packet if there is one. If not, fake a zero-data end of
	   stream packet.

	   Then we throw this to the transport thread.
	*/
	handlePrematureEnd();
	throw;
      }

    // you only get here if a full Ogg page broke
    // See whether we already have this serialNo
    
    debug(false,
	  "Serial No. ", pg->serialNo_
	  ," *** Got page ",pg->pageNo()
	  );

    pthread_mutex_lock(&lockPages_);
    {
      iPages i = pages_.find(pg->serialNo_);
      if ( i == pages_.end() )
	{
	  if ( !gotAllStreams_ )
	    {
	      // Very first page of this logical stream
	      // Create a new PageDecapser only if
	      // we have not already gotten a second page.
	      pages_[pg->serialNo_] = pg;

	      debug(true && 
		    ( pg->pagesSize_ = pg->size() )
		    &&
		    ( pg->headersSize_ = pg->headerSize_ )
		    ,"Serial No. ",pg->serialNo_
		    ," new stream"
		    );

	      // Check version once per stream
	      if ( pg->version() > Transport::CurrentVersion )
		{
		  // If this is indeed a version problem,
		  // there would most likely be a crash.
		  pg->transportError(Error::BeyondCurrentVersion);
		}
	    }
	  else
	    {
	      // Otherwise, we inform the other streams there has been an error
	      // To be passed into the next packets
	      for ( iPages i = pages_.begin()
		      ;
		    i != pages_.end()
		      ;
		    ++i
		    )
		{
		  PageDecapser & p = *(static_cast<PageDecapser*>(i->second));
		  // only if page has been claimed
		  if ( p.logical_ )
		    p.transportError(Error::SomeStreamsIgnored);
		}
	      // We throw this page away
	      pg = 0;
	    }
	}
      else
	{
	  if ( !gotAllStreams_ )
	    {
	      // Second page of this stream!
	      gotAllStreams_ = true;
	      pthread_cond_broadcast(&condAll_);
	    }
	  

	  PageDecapser & p = *(static_cast<PageDecapser*>(i->second));

	  // existing page takes over temp page info
	  p = *pg;

	  // debug
	  debug(true &&
		( p.pagesSize_ += p.size() )
		&&
		( p.headersSize_ += p.headerSize_ )
		);

	  // get rid of temp page
	  // but not the data area
	  pg->data_ = 0;
	  delete pg;

	  pg = &p;
	}
    }
    pthread_mutex_unlock(&lockPages_);

    return(pg);
  }

  //.......................................................

  void
  TransportDecapser::decapPage(
			       PageDecapser & pg
			       )
  {
    /* Initially, the page will decapsulate into a first packet.
       If the page is unclaimed, this first packet is held till
       a second packet for the stream comes along. At this point
       the call to pg.page() will return null.
    */
    bool triedAgain = false;
    while ( !pg.page() )
      {
	pthread_mutex_lock(&lockPages_);
	{
	  if ( nDoneSelect_ < nToSelect_ )
	    {
	      // This allows a co-ordinating stream's header pages
	      // which may contain info about the other streams
	      // to be decapsulated before the first header pages
	      // for these other streams have been claimed,
	      // as long as the co-ordinating stream is claimed
	      // first, and its header pages precede the subsequent
	      // header pages of the other streams.
	      debug(true,
		    "transport stalled on serial ", pg.serialNo_
		    );
	      pthread_cond_wait(&condSelect_, &lockPages_);
	    }
	  else
	    {
	      if ( triedAgain ) {
		// Since the last selection has occured
		// there is no chance we can be claimed
		// we despair and delete it
		debug(true,
		      "transport deleting page for serial ", pg.serialNo_
		      );
		iPages i = pages_.find(pg.serialNo_);
		if ( i != pages_.end() )
		  {
		    pages_.erase(i);
		  }
		// but not the data area
		pg.data_ = 0;
		delete &pg;
		pthread_mutex_unlock(&lockPages_);
		break;
	      }
	      else {
		debug(true,
		      "transport trying again with ", nDoneSelect_,
		      " claimed out of ", nToSelect_
		      );
		triedAgain = true;
	      }
	    }
	}
	pthread_mutex_unlock(&lockPages_);
      }
  }

  //.......................................................

  void
  TransportDecapser::handlePrematureEnd()
  {
    for ( iPages i = pages_.begin()
	    ;
	  i != pages_.end()
	    ;
	  ++i
	  )
      {
	PageDecapser & p = *(static_cast<PageDecapser*>(i->second));
	p.handlePrematureEnd();
      }
  }

  //--Page----------------------------------------------------------------------------------

  PageImpl::PageImpl()
    : bodySize_(0)
    , headerSize_(0)
    , numSegments_(0)
    , granulePosition_(0)
    , pageSize_(defaultPageSize_)
    , streamBegun_(false)
    , streamEnded_(false)
    , isHeader_(true)
    , logical_(0)
    , pageNo_(0)
    , lastPacket_(0)
    , packetNo_(0)
  {}

  PageEncapser::PageEncapser()
    : PageImpl()
    , ready_(false)
    , gotLastPacket_(false)
  {
    pthread_mutex_init(&lockLast_, 0);
    pthread_cond_init(&condLast_, 0);

    data_ = new unsigned char[maxPageSize_];

    body_ = data_ + allocedHeaderSize_;
    header_ = body_;
  }

  PageDecapser::PageDecapser()
    : PageImpl()
  {
    data_ = new unsigned char[workSize_];

    header_ = data_;
    headerSize_ = 0;
    body_ = header_;
    bodySize_ = maxPageSize_;

  }

  PageDecapser::PageDecapser(
			     const PageDecapser & pg
			     )
    : PageImpl()
    , seeking_(false)
  {
    data_ = pg.data_;
    pageSize_ = 0;

    header_ = pg.header_;
    headerSize_ = 0;
    body_ = header_;
    bodySize_ = 0;

    pthread_mutex_init(&lockSeek_, 0);
    pthread_cond_init(&condSeek_, 0);
    pthread_mutex_init(&lockFirst_, 0);
    pthread_cond_init(&condFirst_, 0);

    packetNo_ = 0;
  }

  //.......................................................

  PageImpl::~PageImpl()
  {
    if ( lastPacket_ )
      delete lastPacket_;
  }

  PageEncapser::~PageEncapser()
  {
    if ( data_ )
      delete data_;
  }

  // PageDecapser instances share workPage_'s data_

  //.......................................................

  PageEncapser *
  PageEncapser::page()
  {
    bool firstPacket = false;

    while ( (isHeader_		// each logical header in its own page
	     &&
	     (bodySize_ <= 0)	// when last header has been written out
	     )
	    ||
	    (!isHeader_		// have data
	     &&
	     !streamEnded_	// stream has not ended
	     &&			// and have not exceeded segment limit
	     (numSegments_ < 255)
	     &&			// and have not exceeded page size
	     (bodySize_ < pageSize_)
	     )
	    )
      {
	if ( bodySize_ == 0 )
	  {
	    // fresh page
	    granulePosition_ = -1;
	  }

	if ( !lastPacket_ )
	  {
	    lastPacket_ = static_cast<PacketToTransport*>(logical_->deq());
	    if ( !lastPacket_ )
	      {
		// yield to other pages
		debug(false,
		      "Serial No. ",serialNo_
		      ," no packet at numSegments ", numSegments_
		      ," body size ", bodySize_
		      );
		return(0);
	      }

	    if ( numSegments_ <= 0 )
	      {
		// First packet of fresh page
		firstPacket = true;
	      }
	    else 
	      {
		// a page is in progress
		// flush it if requested
		if ( lastPacket_->flush() )
		  break;
	      }
	  }

	if ( numSegments_ <= 0 )
	  {
	    // Fresh page
	    if ( lastPacket_->pageSize() > 0 )
	      // packet pageSize_ == 0 means default or don't change
	      pageSize_ = lastPacket_->pageSize();
	    isHeader_ = lastPacket_->isHeader();
	    headerNo_ = lastPacket_->headerNo();
	  }
	
	long toCopy = lastPacket_->toDo();
	long copySize = 0;
	long remainderSize = pageSize_ - bodySize_;
	for (
	     ;
	     (numSegments_ < 255)
	       ;
	     numSegments_++
	     )
	  {
	    segmentSizes_[numSegments_] = ( toCopy < 255
					    ? toCopy
					    : 255
					    );
	    toCopy -= segmentSizes_[numSegments_];

	    copySize += segmentSizes_[numSegments_];
	    if ( copySize > remainderSize )
	      {
		numSegments_++;
		break;
	      }

	    if ( segmentSizes_[numSegments_] < 255 )
	      {
		granulePosition_ = lastPacket_->granulePosition();
		numSegments_++;
		break;
	      }

	  }
	
	if ( firstPacket )
	  {
	    segmentSizes_[0] |= 0x100;
	    firstPacket = false;
	  }

	if ( copySize > 0 )
	  {
	    memcpy(body_ + bodySize_
		   ,lastPacket_->copyLoc()
		   ,copySize
		   );
	    lastPacket_->done(copySize);
	    bodySize_ += copySize;
	  }
	
	if ( lastPacket_->toDo() <= 0 )
	  {
	    if ( lastPacket_->streamEnding() )
	      {
		// indicating last packet
		streamEnded_ = true;
		debug(true,
		      "Serial No. ",serialNo_
		      ," got last packet No. ",lastPacket_->Packet::packetNo()
		      ," at page No. ",pageNo_
		      );
	      }
	    
	    if ( (segmentSizes_[numSegments_-1] & 0xff)
		 < 255
		 )
	      {
		// this packet has ended in this page
		delete lastPacket_;
		lastPacket_ = 0;

		if ( streamEnded_ )
		  gotLastPacket(true);
	      }
	    //else
	    //we still need to write a zero segment size segment
	  }
      }

    header();
    ready_ = true;

    debug(true,
	  "Serial No. ",serialNo_
	  ," encapsulated page ",pageNo_
	  ," num segments ", numSegments_
	  ," bodySize ", bodySize_
	  );

    return(this);
  }

  //.......................................................

  void
  PageEncapser::header()
  {
    // calculate beginning of header
    headerSize_ = numSegments_ + 27;
    header_ = body_ - headerSize_;
	    

    // construct the header
    memcpy(header_,"OggS",4);
  
    /* stream structure version */
    header_[4]=Transport::CurrentVersion;
  
    /* continued packet flag? */
    header_[5]=0x00;
    if((segmentSizes_[0]&0x100)==0)
      {
	// page does not begin with fresh packet
	header_[5]|=0x01;
      }

    /* first page flag? */
    if( !streamBegun_ )
      header_[5]|=0x02;
    /* last page flag? */
    if( streamEnded_ )
      header_[5]|=0x04;
    streamBegun_= true;

    unsigned int i;

    /* 64 bits of PCM position */
    Position		position = granulePosition_;
    for(i=6;i<14;i++){
      header_[i]=(unsigned char)(position&0xff);
      position>>=8;
    }

    /* 32 bits of stream serial number */
    
    long serialno = logical_->intf().serialNo();
    for(i=14;i<18;i++){
      header_[i]=(unsigned char)(serialno&0xff);
      serialno>>=8;
    }


    /* 32 bits of page counter (we have both counter and page header
       because this val can roll over) */
    long pageno = pageNo_;
    for(i=18;i<22;i++){
      header_[i]=(unsigned char)(pageno&0xff);
      pageno>>=8;
    }
  
    /* segment table */
    header_[26]=(unsigned char)(numSegments_&0xff);
    for(i=0;i<numSegments_;i++)
      header_[i+27]=(unsigned char)(segmentSizes_[i]&0xff);
  
    /* calculate the checksum and set it into the header*/
  
    setChecksum();

  }
  
  //.......................................................
  void
  PageDecapser::skip()
  {
    body_ = static_cast<unsigned char*>(memchr(header_+1
					       ,'O'
					       ,size()-1
					       )
					);
    if ( body_ )
      {
	bodySize_ = size() - (
			      body_ - header_	// skipped
			      );
	debug(true,
	      "skipped ", (body_ - header_), "bytes"
	      );
      }
    else
      {
	// skip over whole thing
	body_ = header_ + size();
	debug(true,
	      "skipped ", size(), "bytes"
	      );
	bodySize_ = 0;
      }

    header_ = body_;
    headerSize_ = 0;
  }
  
  //.......................................................

  void
  PageDecapser::header()
  {
    if ( headerSize_ <= 0 )
      {
	// don't have a header yet
	// bodySize_ has size of all data read so far
	if ( bodySize_ < 27 )
	  {
	    // still don't
	    debug(false,
		  bodySize_," not enough for an initial header"
		  );
	    return;
	  }

	if ( memcmp(header_
		    ,"OggS"
		    ,4)
	     != 0
	     )
	  {
	    // failed to detect page beginning
	    debug(true,
		  "failed to detect page beginning"
		  );
	    skip();
	    return;
	  }

	numSegments_ = header_[26];
	size_t headerSize = numSegments_ + 27;
	if ( bodySize_ < headerSize )
	  {
	    // don't have anough for a header
	    debug(false,
		  bodySize_," not enough to get the header size", headerSize
		  );
	    return;
	  }

	headerSize_ = headerSize;
	body_ = header_ + headerSize_;
	// now set bodySize_ to size of data beyond header
	bodySize_ -= headerSize;
  
	// decode the serialNo
	serialNo_ = (
		     header_[14] |
		     (header_[15]<<8) |
		     (header_[16]<<16) |
		     (header_[17]<<24)
		     );
	
	// set pageSize_ to expected bodySize
	pageSize_ = headerSize;
	unsigned int i = 27;
	unsigned int j = 0;
	int k = -1;
	for ( ;
	      i < headerSize_
		;
	      ++i, ++j
	      )
	  {
	    pageSize_ += segmentSizes_[j] = header_[i];
	    if ( segmentSizes_[j] != 255 )
	      k = j;
	  }
	if ( k >= 0 )
	  segmentSizes_[k] |= 0x100;

      }

    return;

  }
  
  //.......................................................

  PageDecapser &
  PageDecapser::operator =(
			   const PageDecapser & p
			   )
  {
    // At this point, data transport has been validated for incoming page.
    // Here we do some assignment of incoming page properties to existing page,
    header_ = p.header_;
    headerSize_ = p.headerSize_;
    body_ = p.body_;
    bodySize_ = p.bodySize_;

    numSegments_ = p.numSegments_;
    for ( size_t i = 0
	    ;
	  i < numSegments_
	    ;
	  ++i
	  )
      {
	segmentSizes_[i] = p.segmentSizes_[i];
      }
    pageSize_ = p.pageSize_;


    return(*this);
  }

  //.......................................................

  bool
  PageDecapser::seeking()
  {
    bool	yes;

    debug(false,
	  "Serial No. ", serialNo_
	  ," at page ", pageNo_,", testing for seeking"
	  );
    pthread_mutex_lock(&lockSeek_);
    {
      yes = seeking_;
    }
    pthread_mutex_unlock(&lockSeek_);

    return(yes);
  }

  void
  PageDecapser::seeking(
			bool	yes
			)
  {
    if ( yes )
      {
	pthread_mutex_lock(&lockSeek_);
	{
	  // called by Logical thread
	  // wait till seek has completed
	  // seeking_ is true only while waiting
	  // false all other times
	  seeking_ = yes;
	  /*
	  while ( seeking_ )
	    pthread_cond_wait(&condSeek_
			      ,&lockSeek_
			      );
	  */
	}
	pthread_mutex_unlock(&lockSeek_);
      }
    else
      {
	// called by Transport
	// after seek has found or stream ended
	pthread_mutex_lock(&lockSeek_);
	{
	  seeking_ = yes;
	  pthread_cond_signal(&condSeek_);
	}
	pthread_mutex_unlock(&lockSeek_);
      }
  }

  //.......................................................

  PageDecapser *
  PageDecapser::page()
  {
    if ( lastPacket_ 
	 && !logical_
	 )
      return(0);

    void * body = body_;
    size_t i = 0;

    if ( seeking() )
      {
	body = seek(
		    &i
		    );
	if ( seeking() )
	  return(this);
      }
    else
      {
	if ( lastPacket_ )
	  {
	    if ( !packetContinued() )
	      {
		// Unless lastPacket_ was first packet,
		// which was retained even though it was complete,
		// lastPacket_ has not ended, and yet
		// the current page does not continue it - 
		// set transportError_, and enque it.
		// We leaves it up to Logical to decide what to do.
		if ( lastPacket_->Packet::packetNo() > 0 )
		  lastPacket_->transportError(Error::PacketNonContinuation);
		
		logical_->enq(
			      *lastPacket_
			      );
		lastPacket_ = 0;
	      }
	  }
      }
    
    while ( i < numSegments_ )
      {
	size_t	packetSize = 0;
	for ( ;
	      i < numSegments_
		;
	      ++i
	      )
	  {
	    packetSize += segmentSizes_[i]&0xff;
	    if (segmentSizes_[i] != 255)
	      {
		++i;
		break;
	      }
	  }

	formPacket(body
		   ,packetSize
		   );
	
	if ( segmentSizes_[i-1] != 255 )
	  {
	    // Packet ends in this page
	    if ( segmentSizes_[i-1] & 0x100 )
	      {
		// last packet ending in this page
		lastPacket_->granulePosition(granulePosition());
		if ( streamEnding() )
		  // last packet of last page
		  lastPacket_->endStream();
	      }
	    else
	      {
		// pass on the last known position
		lastPacket_->granulePosition(granulePosition_);
	      }
	    
	    if ( packetNo_ == 1 )
	      {
		gotFirstPacket(true);
	      }

	    if ( logical_ )
	      {
		// This stream has been claimed
		logical_->enq(
			      *lastPacket_
			      );
		lastPacket_ = 0;
	      }
	  }
	
	body_ += packetSize;
	body = body_;
      }

    if ( !streamEnded_ && streamEnding() )
      {
	debug(true,
	      "Serial No. ", serialNo_
	      ," end of stream page ", pageNo(),"\n"
	      ," total size = ", pagesSize_
	      ," headers size = ", headersSize_
	      );
	streamEnded_ = streamEnding();
      }

    if ( pageNo_ == pageNo() )
      ++pageNo_;
    
    clear();

    return(this);
  }

  void *
  PageDecapser::seek(
		     size_t * whichSegment
		     )
  {
    // seeking forward simply skips copying the data
    // except for partial packets
    // therefore you can seek on multiple Logical streams
    // without interference - you do have to wait though
    unsigned char * body = body_;
    size_t i = 0;
    size_t last_i = 0;

    while ( i < numSegments_ )
      {
	if (
	    (seekPosition_ <= granulePosition())
	    &&
	    (seekNo_ <= packetNo_ )
	    )
	  {
	    // arrived
	    seeking(false);
	    *whichSegment = i;
	    return(body);
	  }

	last_i = i;
	size_t packetSize = 0;
	for ( ;
	      i < numSegments_
		;
	      ++i
	      )
	  {
	    packetSize += segmentSizes_[i]&0xFF;
	    if (segmentSizes_[i] != 255)
	      {
		++i;
		break;
	      }
	  }
	
	if ( 
	    (segmentSizes_[i-1] != 255)	// packet ended in page
	    )
	  {
	    if ( lastPacket_ )
	      {
		// throw away any previous packet
		delete lastPacket_;
		lastPacket_ = 0;
	      }
	    else
	      {
		++packetNo_;
	      }

	    Position position = granulePosition();
	    if ( granulePosition_ < position )
	      granulePosition_ = position;
	    
	    if ( segmentSizes_[i-1] & 0x100 )
	      {
		// last packet ending in this page
		if ( streamEnding() )
		  {
		    // last packet in streamEnding page
		    // hand it back to page() to enq
		    --packetNo_;
		    seeking(false);
		    *whichSegment = last_i;
		    return(body);
		  }
	      }

	    if ( packetNo_ == 1 )
	      {
		gotFirstPacket(true);
	      }

	    if ( !streamBegun_ )
	      streamBegun_ = true;

	  }
	else
	  {
	    // form partial packet
	    // just in case it is the seek target
	    // we won't know till the next page
	    formPacket(
		       body
		       ,packetSize
		       );
	  }

	body_ += packetSize;
	body = body_;
	
      }

    if ( pageNo_ == pageNo() )
      ++pageNo_;

    clear();

    *whichSegment = i;
    return(body);
  }

  void
  PageDecapser::formPacket(
			   void *	body
			   ,size_t	packetSize
			   )
  {
    if ( !lastPacket_ )
      {
	lastPacket_ = PacketToTransport::createPacket(
						      logical_
						      ,packetSize
						      );
	lastPacket_->packetNo(packetNo_);
	++packetNo_;

	if ( packetContinued() )
	  {
	    // Previous packet had ended.
	    // So this packet now has a packet continuation error
	    lastPacket_->transportError(Error::PacketContinuation);
	    // Only one continuity error per page
	    packetContinued(false);
	  }
      }
    else
      {
	lastPacket_->resize(
			   lastPacket_->size()
			   + packetSize
			   );
      }

    if ( lastPacket_->Packet::packetNo() == 0 )
      {
	// first packet error only
	lastPacket_->transportError(transportError());
      }


    if (pageNo_ != pageNo())
      // Detect transport continuity problem
      lastPacket_->transportError(Error::NonContinuity);

    Position position = granulePosition();
    if ( granulePosition_ > position )
      lastPacket_->transportError(Error::BadGranulePosition);
    else if ( granulePosition_ < position )
      {
	granulePosition_ = position;
	debug(false,
	      "Serial No. ", serialNo_
	      ," granule position changed to ", granulePosition_
	      );
      }
    
    if ( streamEnded_ )
      // Already ended, why this packet?
      lastPacket_->transportError(Error::StreamEnd);

    if ( !streamBegun_ )
      {
	if ( !streamBeginning()
	     ||
	     streamEnding() )
	  lastPacket_->transportError(Error::StreamBegin);
	// By setting the true condition,
	// we will continue to report on this error.
	streamBegun_ = true;
      }
    else
      {
	if ( streamBeginning() )
	  lastPacket_->transportError(Error::StreamBegin);
      }

    if ( packetSize )
      {
	// copy segments into packet
	memcpy(lastPacket_->copyLoc()
	       ,body
	       ,packetSize
	       );
	lastPacket_->done(packetSize);
      }
  }

  //.......................................................
  void
  PageDecapser::handlePrematureEnd()
  {
    streamEnded_ = true;
    numSegments_ = 0;

    if ( !logical_ )
      {
	// ended before the stream can be claimed!

	return;
      }

    if ( !lastPacket_ )
      {
	// make up a packet if we don't have one
	lastPacket_ = PacketToTransport::createPacket(logical_,0);
	lastPacket_->packetNo(packetNo_);
	++packetNo_;
      }

    lastPacket_->transportError(Error::PrematureEnd);
    lastPacket_->endStream();

    logical_->enq(
		  *lastPacket_
		  );
    lastPacket_ = 0;

    return;
  }

  //.......................................................

  bool
  PageDecapser::gotFirstPacket()
  {
    pthread_mutex_lock(&lockFirst_);
    {
      while( !gotFirstPacket_ )
	{
	  pthread_cond_wait(&condFirst_, &lockFirst_);
	}
    }
    pthread_mutex_unlock(&lockFirst_);

    return(true);
  }

  void
  PageDecapser::gotFirstPacket(
			       bool
			       )
  {
    pthread_mutex_lock(&lockFirst_);
    {
      gotFirstPacket_ = true;
      pthread_cond_signal(&condFirst_);
    }
    pthread_mutex_unlock(&lockFirst_);
  }

  //.......................................................
  
  void
  PageDecapser::transportError(
			       Error::ErrorNo errorNo
			       )
  {
    debug(true,
	  "Serial No. ", serialNo_
	  , " error : ", ErrorImpl::descriptions_[errorNo]
	  );
    transportError_ |= errorNo;
  }

  Error
  PageDecapser::transportError()
  {
    return(transportError_);
  }

  //.......................................................
  long
  Page::pageNo() const
  {
    const PageImpl * p = static_cast<const PageImpl *>(this);

    return(p->pageNo());
  }

  long
  PageEncapser::pageNo() const
  {
    return(pageNo_);
  }

  long
  PageDecapser::pageNo() const
  {
    return(header_[18] |
	   (header_[19]<<8) |
	   (header_[20]<<16) |
	   (header_[21]<<24)
	   );
  }

  //.......................................................

  unsigned char
  PageDecapser::version()
  {
    return(header_[4]);
  }

  //.......................................................

  bool
  PageDecapser::packetContinued()
  {
    return(header_[5]&0x01);
  }

  //.......................................................

  void
  PageDecapser::packetContinued(
				bool	contd
				)
  {
    if ( contd )
      header_[5] |= 0x01;
    else
      header_[5] &= ~(0x01);
  }

  //.......................................................

  bool
  PageDecapser::streamBeginning()
  {
    return(header_[5]&0x02);
  }

  //.......................................................

  bool
  PageDecapser::streamEnding()
  {
    return(header_[5]&0x04);
  }

  //.......................................................

  Position
  PageDecapser::granulePosition()
  {
    Position position = header_[13];
    for ( short i = 12
	    ;
	  i > 5
	    ;
	  --i
	  )
      {
	position = (position<<8) | header_[i];
      }

    return(position);
  }

  //.......................................................

  void
  PageEncapser::clear()
  {
    pageNo_++;
    header_ = body_;
    bodySize_ =0;
    headerSize_ =0;
    numSegments_ =0;
    granulePosition_ = -1;
    ready_ = false;
  }


  void
  PageDecapser::clear()
  {
    pageSize_ = 0;

    headerSize_ = 0;
    body_ = header_;
    bodySize_ = 0;
    numSegments_ = 0;
  }
  //.......................................................

  void *
  Page::data()
  {
    PageImpl * p = static_cast<PageImpl *>(this);

    if ( p->size() )
      return(p->header_);
    else
      return(0);
  }

  void
  Page::data(
	     void *	received
	     ,size_t	siz
	     )
  {
    if ( size() < siz )
      siz = size();

    memcpy(data()
	   ,received
	   ,siz
	   );
  }

  size_t
  Page::size() const
  {
    const PageImpl * p = static_cast<const PageImpl *>(this);

    return( p->headerSize_ + p->bodySize_ );
  }

  bool
  Page::ending() const
  {
    const PageImpl * p = static_cast<const PageImpl *>(this);

    return( p->streamEnded_ );
  }

  bool
  PageImpl::ended() const
  {
    return( streamEnded_
	    &&
	    (numSegments_ <= 0)
	    );
  }

  Position
  Page::granulePosition() const
  {
    const PageImpl * p = static_cast<const PageImpl *>(this);

    return(p->granulePosition_);
  }
     
  bool
  Page::isHeader() const
  {
    const PageImpl * p = static_cast<const PageImpl *>(this);

    return(p->isHeader_);
  }
     
  long
  Page::serialNo() const
  {
    const PageImpl * p = static_cast<const PageImpl *>(this);

    return(p->serialNo_);
  }

  //--Callbacks----------------------------------------------------------------------------

  void
  Transport::sendCallback(
			  Page &	pg
			  )
  {
    std::cout.write(static_cast<const char *>(pg.data())
		    ,pg.size()
		    );
  }

  size_t
  Transport::recvCallback(
			  Page &	pg
			  )
  {
    // At eof, this does not terminate Transport
    if ( std::cin.read(static_cast<char *>(pg.data())
		       , pg.size()
		       )
	 ||
	 std::cin.eof()
	 )
      {
	size_t count = std::cin.gcount();
	debug(false,
	      "Serial No. ", pg.serialNo()
	      ," received ",count," bytes at ",pg.data()
	      );

	return(count);
      }
    else
      {
	return(0);
      }
  }

};
