/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for transport of the Ogg format
 *
 * Copyright (C) 2006, 2007, 2008 Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Ogg::Logical implementation                                               */
/*                                                                           */
/*===========================================================================*/
#ifdef __GNUG__
#pragma implementation "Logical.H"
#pragma implementation "LogicalToTransport.H"
#pragma implementation "LogicalImpl.H"
#endif

#include	<LogicalImpl.H>
#include	<Ogg/Debug.H>

extern "C" 
{
# include	<string.h>
}

namespace Ogg
{
  //--Logical-------------------------------------------------------------------------------
  Logical::Logical(
		   Transport &		transport
		   ,long		serialNo
		   )
  {
    if ( serialNo <= 0 )
      throw Logical::SerialNotPositive(serialNo);

    LogicalImpl * impl = new LogicalImpl(
					 transport
					 ,serialNo
					 );

    impl->intf_ = this;
    impl_ = impl;

    impl->transport_->enterMux();
  }

  Logical::Logical(
		   Transport & transport
		   )
  {
    LogicalImpl * impl = new LogicalImpl(
					 transport
					 ,0
					 );
    impl->intf_ = this;
    impl_ = impl;

    impl->transport_->enterSelect();
  }

  //.......................................................

  LogicalImpl::LogicalImpl(
			   Transport &	transport
			   ,long	serialNo
			   )
    : transport_(static_cast<TransportToLogical *>(transport.impl()))
    , serialNo_(serialNo)
    , writer_(0)
    , reader_(0)
  {
    pthread_mutex_init(&lockPackets_, 0);
    pthread_cond_init(&condEmpty_, 0);
    pthread_cond_init(&condFull_, 0);
  }

  //.......................................................

  Logical::~Logical()
  {
    if ( impl_ )
      {
	LogicalImpl * impl = static_cast<LogicalImpl *>(impl_);
	delete impl;
	impl_ = 0;
      }

  }
  
  LogicalImpl::~LogicalImpl()
  {
    if ( writer_ )
      {
	delete writer_;
	writer_ = 0;
	transport_->exitMux();
      }

    if ( reader_ )
      {
	delete reader_;
	reader_ = 0;
	transport_->exitSelect();
      }

    // delete all packets_, needed only for decapsulation
    pthread_mutex_lock(&lockPackets_);
    {
      while ( !packets_.empty() )
	{
	  packets_.pop();
	}
    }
    pthread_mutex_unlock(&lockPackets_);
  }

  //.......................................................

  long
  Logical::serialNo() const
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(impl_);

    return (impl->serialNo_);
  }

  //.......................................................

  Transport &
  Logical::transport() const
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(impl_);

    return (impl->transport_->intf());
  }

  //.......................................................

  Logical::Writer &
  Logical::writer()
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(impl_);

    if ( impl->transport_->encapsulating() )
      {
	if ( !impl->writer_ )
	  {
	    impl->writer_ = new LogicalImpl::WriterImpl(*impl);
	    return(*impl->writer_);
	  }
	else
	  throw Logical::WriterAlreadyExists(impl->serialNo_);
      }
    else
      throw Logical::NotForWriting(impl->serialNo_);
  }

  Logical::Reader &
  Logical::reader()
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(impl_);

    if ( !impl->transport_->encapsulating() )
      {
	if ( !impl->reader_ )
	  {
	    impl->reader_ = new LogicalImpl::ReaderImpl(*impl);
	    return( *impl->reader_ );
	  }
	else
	  throw Logical::ReaderAlreadyExists(impl->serialNo_);
      }
    else
      throw Logical::NotForReading(impl->serialNo_);
  }

  //.......................................................

  Logical &
  LogicalToTransport::intf()
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(this);

    return(*impl->intf_);
  }

  //.......................................................

  bool
  LogicalToTransport::empty()
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(this);
    bool none = false;

    pthread_mutex_lock(&impl->lockPackets_);
    {
      none = impl->packets_.empty();
    }
    pthread_mutex_unlock(&impl->lockPackets_);

    return(none);
  }

  //.......................................................

  bool
  LogicalToTransport::full()
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(this);
    bool none = false;

    pthread_mutex_lock(&impl->lockPackets_);
    {
      none = ( impl->packets_.size() >= LogicalImpl::defaultQueueSize_ );
    }
    pthread_mutex_unlock(&impl->lockPackets_);

    return(none);
  }

  //.......................................................
  
  void
  LogicalToTransport::enq(
			  Packet & pkt
			  )
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(this);

    pthread_mutex_lock(&impl->lockPackets_);
    {
      debug(true &&
	    ( impl->packets_.size() >= (LogicalImpl::defaultQueueSize_- 2) ),
	    "Serial No. ", impl->serialNo_
	    ," enq time packets size = ", impl->packets_.size()
	    );

      while ( impl->packets_.size() >= LogicalImpl::defaultQueueSize_ )
	pthread_cond_wait(&impl->condFull_
			  ,&impl->lockPackets_
			  );
      impl->packets_.push(static_cast<PacketImpl*>(&pkt));
      pthread_cond_signal(&impl->condEmpty_);
    }
    pthread_mutex_unlock(&impl->lockPackets_);
  }
  
  //.......................................................

  void
  LogicalToTransport::wait()
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(this);

    pthread_mutex_lock(&impl->lockPackets_);
    {
      while ( impl->packets_.empty() )
	pthread_cond_wait(&impl->condEmpty_
			  ,&impl->lockPackets_
			  );
    }
    pthread_mutex_unlock(&impl->lockPackets_);

    return;
  }

  //.......................................................
  
  Packet *
  LogicalToTransport::deq()
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(this);
    Packet * pkt = 0;

    if ( !empty() )
      {
	pthread_mutex_lock(&impl->lockPackets_);
	{
	  pkt = impl->packets_.front();
	  impl->packets_.pop();
	  pthread_cond_signal(&impl->condFull_);
	}
	pthread_mutex_unlock(&impl->lockPackets_);
      }

    return(pkt);
  }
  
  //--Packet-------------------------------------------------------------------------------
  PacketImpl::PacketImpl(
			 LogicalImpl *	logical
			 ,size_t	size
			 )
    : logical_(logical)
      , size_(size)
      , done_(0)
      , pageSize_(0)	// defaults to what is set in Transport
      , flush_(false)
      , isHeader_(true)
      , headerNo_(0)
      , packetNo_(0)
      , granulePosition_(0)
      , streamEnding_(false)
      , transportError_()
  {
    if ( size <= 0 )
      // sometimes we need to create a packet of data size 0
      size = 1;

    data_ = new unsigned char[size];
  }

  //.......................................................

  PacketImpl::PacketImpl(
			 PacketImpl &	pkt
			 )	// constructs the next packet in the sequence
    : logical_(pkt.logical_)
      , size_(pkt.size_)
      , done_(0)
      , pageSize_(pkt.pageSize())
      , flush_(false)
      , isHeader_(pkt.isHeader())
      , headerNo_(pkt.headerNo())
      , packetNo_(pkt.packetNo_ + 1)
      , granulePosition_(pkt.granulePosition())
      , streamEnding_(false)
      , transportError_()
  {
    if ( isHeader_ )
      // one more header packet
      ++headerNo_;
      
    data_ = new unsigned char[pkt.size_];
  }

  //.......................................................

  PacketImpl::~PacketImpl()
  {
    ::operator delete(data_);
  }
  //.......................................................
  
  void *
  Packet::data()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->data_);
  }

  void
  Packet::data(
	       void *	received
	       ,size_t	siz
	       )
  {
    if ( size() < siz )
      siz = size();

    memcpy(data()
	   ,received
	   ,siz
	   );
  }

  //.......................................................
  
  size_t
  Packet::size()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->size_);
  }

  //.......................................................
  
  PacketToTransport *
  PacketToTransport::createPacket(
				  LogicalToTransport *	logical
				  ,size_t	size
				  )
  {
    LogicalImpl * impl = static_cast<LogicalImpl *>(logical);

    return(new PacketImpl(impl
			  ,size
			  )
	   );
  }

  //.......................................................
  
  void
  PacketToTransport::endStream()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    debug(true
	  ,"Serial No. ", impl->logical_->serialNo_
	  ," ending stream at ", impl->packetNo_, "th packet"
	  );
    impl->streamEnding_ = true;
  }

  //.......................................................
  
  bool
  PacketToTransport::streamEnding()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->streamEnding_);
  }

  //.......................................................
  
  long
  PacketToTransport::pageSize()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->pageSize_);
  }

  //.......................................................
  
  bool
  PacketToTransport::flush()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->flush_);
  }

  //.......................................................
  
  bool
  PacketToTransport::isHeader()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->isHeader_);
  }

  //.......................................................
  
  long
  PacketToTransport::headerNo()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->headerNo_);
  }

  //.......................................................
  
  PacketNo
  Packet::packetNo()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->packetNo_);
  }

  void
  PacketToTransport::packetNo(PacketNo	pktno)
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    impl->packetNo_ = pktno;
  }

  //.......................................................
  
  Error
  PacketToTransport::transportError()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->transportError_);
  }

  void
  PacketToTransport::transportError(
				    Error::ErrorNo	errorNo
				    )
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    impl->transportError_ |= errorNo;
  }

  void
  PacketToTransport::transportError(
				    Error	error
				    )
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    impl->transportError_ |= error;
  }

  //.......................................................
  
  long
  PacketToTransport::done()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->done_);
  }

  void
  PacketToTransport::done(long	did)
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    impl->done_ += did;
  }

  //.......................................................
  
  long
  PacketToTransport::toDo()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->size() - impl->done());
  }

  //.......................................................
  
  void *
  PacketToTransport::copyLoc()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);
    
    return(impl->data_ + impl->done());
  }

  //.......................................................
  
  Position
  PacketToTransport::granulePosition()
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    return(impl->granulePosition_);
  }

  void
  PacketToTransport::granulePosition(Position	position)
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    impl->granulePosition_ = position;
    debug(false && (impl->granulePosition_ >= 0),
	  "packetNo ", impl->packetNo_
	  ," granule position changed to ", impl->granulePosition_
	  );
  }

  //.......................................................
  
  PacketToTransport *
  PacketToTransport::resize(
			    size_t	size
			    )
  {
    PacketImpl * impl = static_cast<PacketImpl *>(this);

    if ( size > impl->size_ )
      {
	unsigned char * data = impl->data_;
	impl->data_ = new unsigned char[size];
	memcpy(impl->data_
	       ,data
	       ,impl->size_
	       );
	delete data;
      }
    impl->size_ = size;

    return(this);
  }

  //--Writer-------------------------------------------------------------------------------

  LogicalImpl::WriterImpl::WriterImpl(
				      LogicalImpl &	logical
				      )
    : logical_(logical)
      , packet_(0)
      , dataSize_(defaultDataSize_)
  {
    logical_.transport_->add(logical_);

    packet_ = new PacketImpl(
			     &logical_
			     ,dataSize_
			     );
  }

  //.......................................................
  
  Logical::Writer::~Writer()
  {
    LogicalImpl::WriterImpl * w
      = static_cast<LogicalImpl::WriterImpl *>(this);

    if ( w->packet_ )
      {
	// try enquing current packet as last packet
	w->packet_->endStream();
	w->logical_.enq(*w->packet_);
      }

    // wait till Transport has deleted last packet
    w->logical_.transport_->lastPacket(w->logical_.serialNo_);

    w->logical_.writer_ = 0;
  }

  //.......................................................
  
  size_t
  Logical::Writer::dataSize() const
  {
    const LogicalImpl::WriterImpl * w
      = static_cast<const LogicalImpl::WriterImpl *>(this);

    return(w->dataSize_);
  }

  //.......................................................
  
  void
  Logical::Writer::dataSize(
			    size_t	size
			    )
  {
    LogicalImpl::WriterImpl * w
      = static_cast<LogicalImpl::WriterImpl *>(this);

    w->packet_->resize(size);

    w->dataSize_ = size;
  }

  //.......................................................
  
  void
  Logical::Writer::flush()
  {
    LogicalImpl::WriterImpl * w
      = static_cast<LogicalImpl::WriterImpl *>(this);

    w->packet_->flush_ = true;
  }

  //.......................................................
  
  void
  Logical::Writer::beginData()
  {
    LogicalImpl::WriterImpl * w
      = static_cast<LogicalImpl::WriterImpl *>(this);

    w->packet_->isHeader_ = false;
  }

  //.......................................................
  
  void
  Logical::Writer::granulePosition(
				   Position	position
				   )
  {
    LogicalImpl::WriterImpl * w
      = static_cast<LogicalImpl::WriterImpl *>(this);

    w->packet_->granulePosition_ = position;
  }

  //.......................................................
  
  Position
  Logical::Writer::granulePosition() const
  {
    const LogicalImpl::WriterImpl * w
      = static_cast<const LogicalImpl::WriterImpl *>(this);

    return(w->packet_->granulePosition());
  }

  //.......................................................
  
  size_t
  Logical::Writer::pageSize() const
  {
    const LogicalImpl::WriterImpl * w
      = static_cast<const LogicalImpl::WriterImpl *>(this);

    return(w->packet_->pageSize());  }

  //.......................................................
  
  void
  Logical::Writer::pageSize(
			    size_t	size
			    )
  {
    LogicalImpl::WriterImpl * w
      = static_cast<LogicalImpl::WriterImpl *>(this);

    w->packet_->pageSize_ = size;
  }

  //.......................................................
  
  Logical::Writer &
  Logical::Writer::operator++()
  {
    LogicalImpl::WriterImpl * w
      = static_cast<LogicalImpl::WriterImpl *>(this);

    PacketImpl * pkt = w->packet_;
    w->packet_ = new PacketImpl(*pkt);
    w->packet_->resize(w->dataSize_);

    // enq the last packet
    w->logical_.enq(*pkt);

    return(*this);
  }
  
  //.......................................................
  
  Packet *
  Logical::Writer::operator->() const throw()
  {
    const LogicalImpl::WriterImpl * w
      = static_cast<const LogicalImpl::WriterImpl *>(this);

    return(w->packet_);
  }

  //--Reader-------------------------------------------------------------------------------

  LogicalImpl::ReaderImpl::ReaderImpl(
				      LogicalImpl &	logical
				      )
    : logical_(logical)
      , granulePosition_(-1)
      , packet_(0)
  {
    long serialNo = 0;
    const Packet * pkt = 0;
    while ( (serialNo = logical.transport_->intf().serialNo(serialNo)) > 0 )
    {
      pkt = logical.transport_->firstPacket(serialNo);
      if ( pkt && logical.intf().selectCallback(*(const_cast<Packet *>(pkt))) )
	{
	  // claim the stream
	  logical.serialNo_ = serialNo;
	  logical.transport_->claim(logical);
	  break;
	}
      pkt = 0;
    }
    logical.transport_->doneSelect();


    if ( !pkt )
      throw Logical::FailedToSelectStream(logical.intf().serialNo());

    Reader::operator++();
  }

  //.......................................................
  
  Logical::Reader::~Reader()
  {
    LogicalImpl::ReaderImpl * r
      = static_cast<LogicalImpl::ReaderImpl *>(this);

    if ( r->packet_ )
      delete r->packet_;

    // Best not to wipe the serialNo_
    // to allow it to be read even after the Reader is gone
    // r->logical_.serialNo_ = 0;
    r->logical_.reader_ = 0;

  }

  //.......................................................
  
  Logical::Reader &
  Logical::Reader::operator++()
  {
    LogicalImpl::ReaderImpl * r
      = static_cast<LogicalImpl::ReaderImpl *>(this);

    if ( r->packet_ )
      delete r->packet_;

    r->logical_.wait();
    r->packet_ = static_cast<PacketImpl*>(r->logical_.deq());
    if ( r->packet_->granulePosition() > r->granulePosition_  )
      // update last known granule position, could be bad
      // -1 granuleposition means we don't know
      r->granulePosition_ = r->packet_->granulePosition();

    return(*this);
  }

  //.......................................................
  
  Logical::Reader &
  Logical::Reader::operator+=(
			      PacketNo	 packetNo
			      )
  {
    LogicalImpl::ReaderImpl * r
      = static_cast<LogicalImpl::ReaderImpl *>(this);

    if ( packetNo > 0 )
      {
	operator++();
	if ( packetNo < 2 )
	  return(*this);
	operator++();
	if ( packetNo < 3 )
	  return(*this);
	  
	r->seek(
		r->packet_->packetNo_ + packetNo - 2
		, r->packet_->granulePosition_
		);

      }

    return(*this);
  }

  //.......................................................
  
  Logical::Reader &
  Logical::Reader::operator=(
			     Position granulePosition
			     )
  {
    LogicalImpl::ReaderImpl * r
      = static_cast<LogicalImpl::ReaderImpl *>(this);

    if ( granulePosition > r->packet_->granulePosition_ )
      {
	r->seek(
		r->packet_->packetNo_
		,granulePosition
		);
      }

    return(*this);
  }

  //.......................................................
  
  bool
  Logical::Reader::ending() const
  {
    const LogicalImpl::ReaderImpl * r
      = static_cast<const LogicalImpl::ReaderImpl *>(this);

    return(r->packet_->streamEnding());
  }

  //.......................................................
  
  Error
  Logical::Reader::transportError() const
  {
    const LogicalImpl::ReaderImpl * r
      = static_cast<const LogicalImpl::ReaderImpl *>(this);

    return(r->packet_->transportError());
  }

  //.......................................................
  
  Position
  Logical::Reader::granulePosition() const
  {
    const LogicalImpl::ReaderImpl * r
      = static_cast<const LogicalImpl::ReaderImpl *>(this);

    return(r->granulePosition_);
  }

  //.......................................................
  
  Packet *
  Logical::Reader::operator->() const throw()
  {
    const LogicalImpl::ReaderImpl * r
      = static_cast<const LogicalImpl::ReaderImpl *>(this);

    return(r->packet_);
  }

  //.......................................................
  
  void
  LogicalImpl::ReaderImpl::seek(
				PacketNo	packetNo
				,Position	granulePosition
				)
  {
    delete packet_;
    packet_ = 0;

    debug(true &&
	  ( logical_.packets_.size() >= (LogicalImpl::defaultQueueSize_- 2) ),
	  "Serial No. ", logical_.serialNo_
	  ," seek time packets size", logical_.packets_.size()
	  );

    while ( !logical_.empty() )
      {
	// examine packets queued so far
	// this makes sure Transport thread is not stalled with a full queue
	// within a lockPages_ which leads to deadlock
	packet_ = static_cast<PacketImpl*>(logical_.deq());
	if ( packet_->granulePosition() > granulePosition_ )
	  {
	    // update last known granule position, could be bad
	    // -1 granuleposition means we don't know
	    granulePosition_ = packet_->granulePosition();
	    debug(true,
		  "Serial No. ", logical_.serialNo_
		  ," before seek: packet ", packet_->Packet::packetNo()
		  ," granule position changed to ", granulePosition_
		  );
	  }
	if ( ((packet_->Packet::packetNo() >= packetNo)
	      &&
	      (granulePosition_ >= granulePosition)
	      )
	     ||
	     ending()
	     )
	  {
	    break;
	  }
	else
	  {
	    packet_ = 0;
	  }
      }

    if ( !packet_ )
      {
	packet_ = static_cast<PacketImpl*>(logical_.transport_->seek(logical_
								     ,packetNo
								     ,granulePosition
								     )
					   );
	while (true)
	  {
	    if ( packet_->granulePosition() > granulePosition_ )
	      {
		// update last known granule position, could be bad
		// -1 granuleposition means we don't know
		granulePosition_ = packet_->granulePosition();
		debug(true,
		      "Serial No. ", logical_.serialNo_
		      ," after seek: packet ", packet_->Packet::packetNo()
		      ," granule position changed to ", granulePosition_
		      );
	      }
	    if ( ((packet_->Packet::packetNo() < packetNo)
		  ||
		  (granulePosition_ < granulePosition)
		  )
		 &&
		 !ending()
		 )
	      {
		delete packet_;
		logical_.wait();
		packet_ = static_cast<PacketImpl*>(logical_.deq());
	      }
	    else
	      break;
	  }
      }
    else
      debug(true,
	    "Serial No. ", logical_.serialNo_
	    ," emptied queue at packet", packet_->Packet::packetNo()
	    );


  }
};
