/* Copyright (C) 2004 W.P. van Paassen - peter@paassen.tmfweb.nl

   This file is part of libmd5model
   
   libmd5model will attempt to parse ID's Doom3 mesh and animation model formats
   
   libmd5model is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your
   option) any later version.

   libmd5model is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with libmd5model; see the file COPYING.  If not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  */

#include "libmd5model.h"
#include "stdlib.h"
#include <stdio.h>
#include <string.h>

extern FILE* MD5MESH_in;
extern FILE* MD5ANIM_in;
extern int MD5MESH_parse();
extern int MD5ANIM_parse();
extern int md5_perror;
extern MD5MESH_JOINT* mesh_joints;
extern MD5_MESH* meshes;
extern MD5ANIM_JOINT* baseframe;
extern MD5_BOUND* bounds;
extern MD5MESH_HEADER mesh_header;
extern MD5ANIM_HEADER anim_header;

int md5_parsed = MD5_PARSED_FAIL;

int MD5_parse(const char* md5_filename)
{
	if (strstr(md5_filename, ".md5mesh")) 
	{
		md5_parsed = MD5_PARSED_MESH;
		if ((MD5MESH_in=fopen(md5_filename, "r")) == NULL)
			return MD5_PARSED_FAIL;
	}
	else if (strstr(md5_filename, ".md5anim"))
	{	
		md5_parsed = MD5_PARSED_ANIM;
		if ((MD5ANIM_in=fopen(md5_filename, "r")) == NULL)
			return MD5_PARSED_FAIL;
	}
	else
	{
		return MD5_PARSED_FAIL;
	}

	/*free data of earlier read model */
	MD5_free();

	/*now parse the model data */	
	if (md5_parsed == MD5_PARSED_MESH)
		MD5MESH_parse();
	else if (md5_parsed == MD5_PARSED_ANIM)
		MD5ANIM_parse();

	if (md5_perror == 1)
		return MD5_PARSED_FAIL;
	else	
		return md5_parsed;
}

void MD5_free()
{
	if (md5_parsed == MD5_PARSED_MESH)
	{
		free((char*)mesh_header.commandline);
		mesh_header.commandline = (char*)NULL;

		free((MD5MESH_JOINT*)mesh_joints);
		mesh_joints = (MD5MESH_JOINT*)NULL;
			
		free((MD5_MESH*)meshes);
		meshes = (MD5_MESH*)NULL;
	}
	else if (md5_parsed == MD5_PARSED_ANIM)
	{
		int i;

		free((char*)anim_header.commandline);
		anim_header.commandline = (char*)NULL;

		free((MD5_BOUND*)bounds);
		bounds = (MD5_BOUND*)NULL;
		
		for (i=0; i < anim_header.num_joints; ++i)
		{
			free((MD5_TRANSFORM*)baseframe[i].frames);
			baseframe[i].frames = (MD5_TRANSFORM*)NULL;
		}

		free((MD5ANIM_JOINT*)baseframe);
		baseframe = (MD5ANIM_JOINT*)NULL;
	}
}

MD5MESH_HEADER* MD5Mesh_get_header()
{
	return &mesh_header;
}

MD5MESH_JOINT* MD5Mesh_get_joint(int idx)
{
	return mesh_joints + idx;
}

MD5MESH_JOINT** MD5Mesh_get_joints()
{
	return &mesh_joints;
}

MD5_MESH* MD5Mesh_get_mesh(int idx)
{
	return meshes + idx;
}

MD5_MESH** MD5Mesh_get_meshes()
{
	return &meshes;
}

MD5_VERTEX* MD5Mesh_get_vertex(int mesh_id, int vertex_id)
{
	return (MD5_VERTEX*)(((MD5_MESH*)(meshes + mesh_id))->verts + vertex_id);
}

MD5_TRIANGLE* MD5Mesh_get_triangle(int mesh_id, int triangle_id)
{
	return (MD5_TRIANGLE*)(((MD5_MESH*)(meshes + mesh_id))->tris + triangle_id);
}

MD5_WEIGHT* MD5Mesh_get_weight(int mesh_id, int weight_id)
{
	return (MD5_WEIGHT*)(((MD5_MESH*)(meshes + mesh_id))->weights + weight_id);
}

MD5_VERTEX** MD5Mesh_get_vertices(int mesh_id)
{
	return &((MD5_MESH*)(meshes + mesh_id))->verts;
}

MD5_TRIANGLE** MD5Mesh_get_triangles(int mesh_id)
{
	return &((MD5_MESH*)(meshes + mesh_id))->tris;
}

MD5_WEIGHT** MD5Mesh_get_weights(int mesh_id)
{
	return &((MD5_MESH*)(meshes + mesh_id))->weights;
}

MD5ANIM_HEADER* MD5Anim_get_header()
{
	return &anim_header;
}

MD5ANIM_JOINT* MD5Anim_get_joint(int idx)
{
	return (MD5ANIM_JOINT*)(baseframe + idx);
}

MD5ANIM_JOINT** MD5Anim_get_joints()
{
	return &baseframe;

}

MD5_BOUND* MD5Anim_get_bound(int idx)
{
	return (MD5_BOUND*)(bounds + idx);
}

MD5_BOUND** MD5Anim_get_bounds()
{
	return &bounds;
}

MD5_TRANSFORM* MD5Anim_get_joint_frame(int joint, int idx)
{
	if (baseframe[joint].frames == (MD5_TRANSFORM*)NULL)
		return (MD5_TRANSFORM*)NULL;
	else
		return (MD5_TRANSFORM*)(baseframe[joint].frames + idx);
}

MD5_TRANSFORM** MD5Anim_get_joint_frames(int joint)
{
	return &baseframe[joint].frames;
}
