/* Copyright (C) 2004 W.P. van Paassen - peter@paassen.tmfweb.nl

   This file is part of libmd5model
   
   libmd5model will attempt to parse ID's Doom3 mesh and animation model formats
   
   libmd5model is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your
   option) any later version.

   libmd5model is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with libmd5model; see the file COPYING.  If not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  */

#include "md5.h"
#include "stdlib.h"
#include <stdio.h>
#include <errno.h>

extern FILE* yyin;
extern int yyparse();

int MD5_parse(const char* md5_filename, int compute_indices)
{
	if ((yyin=fopen(md5_filename, "r")) == NULL)
			return errno;

	/*free data of earlier read model */
	MD5_free();

	/*now parse the model data */	
	yyparse();

	if (compute_indices)
	{
		/* all integer indices to joints, weights or pointers in the struct's unions are replaced by pointers to structs */

		int i,j;

		for (i = 0; i < header.num_joints; ++i)
		{
			if (joints[i].parent.idx == -1)
				joints[i].parent.parent = (MD5_JOINT*)NULL;
			else
				joints[i].parent.parent = (MD5_JOINT*)(joints + joints[i].parent.idx);
		}

		for (i = 0; i < header.num_meshes; ++i)
		{
			for (j = 0; j < meshes[i].num_verts; ++j)
			{
				meshes[i].verts[j].weight.weight = (MD5_WEIGHT*)(meshes[i].weights + meshes[i].verts[j].weight.idx); 
			}

			for (j = 0; j < meshes[i].num_tris; ++j)
			{
				meshes[i].tris[j].vertex1.vertex = (MD5_VERTEX*)(meshes[i].verts + meshes[i].tris[j].vertex1.idx); 
				meshes[i].tris[j].vertex2.vertex = (MD5_VERTEX*)(meshes[i].verts + meshes[i].tris[j].vertex2.idx); 
				meshes[i].tris[j].vertex3.vertex = (MD5_VERTEX*)(meshes[i].verts + meshes[i].tris[j].vertex3.idx); 
			}

			for (j = 0; j < meshes[i].num_weights; ++j)
			{
				meshes[i].weights[j].joint.joint = (MD5_JOINT*)(joints + meshes[i].weights[j].joint.idx); 
			}
		}	
	}

	return 0;
}

void MD5_free()
{
	if (header.commandline != (char*)NULL)
	{
		free((char*)header.commandline);
		header.commandline = (char*)NULL;
	}

	if (joints != (MD5_JOINT*)NULL)
	{
		free((MD5_JOINT*)joints);
		joints = (MD5_JOINT*)NULL;
	}

	if (meshes != (MD5_MESH*)NULL)
	{
		free((MD5_MESH*)meshes);
		meshes = (MD5_MESH*)NULL;
	}
}

MD5_MESH_HEADER* MD5Mesh_get_header()
{
	return &header;
}

MD5_JOINT* MD5Mesh_get_joint(int idx)
{
	return joints + idx;
}
MD5_MESH* MD5Mesh_get_mesh(int idx)
{
	return meshes + idx;
}
MD5_JOINT** MD5Mesh_get_joints()
{
	return &joints;
}

MD5_MESH** MD5Mesh_get_meshes()
{
	return &meshes;
}

MD5_VERTEX* MD5Mesh_get_vertex(int mesh_id, int vertex_id)
{
	return (MD5_VERTEX*)(((MD5_MESH*)(meshes + mesh_id))->verts + vertex_id);
}

MD5_TRIANGLE* MD5Mesh_get_triangle(int mesh_id, int triangle_id)
{
	return (MD5_TRIANGLE*)(((MD5_MESH*)(meshes + mesh_id))->tris + triangle_id);
}

MD5_WEIGHT* MD5Mesh_get_weight(int mesh_id, int weight_id)
{
	return (MD5_WEIGHT*)(((MD5_MESH*)(meshes + mesh_id))->weights + weight_id);
}

MD5_VERTEX** MD5Mesh_get_vertices(int mesh_id)
{
	return &((MD5_MESH*)(meshes + mesh_id))->verts;
}

MD5_TRIANGLE** MD5Mesh_get_triangles(int mesh_id)
{
	return &((MD5_MESH*)(meshes + mesh_id))->tris;
}

MD5_WEIGHT** MD5Mesh_get_weights(int mesh_id)
{
	return &((MD5_MESH*)(meshes + mesh_id))->weights;
}
