/* Copyright (C) 2004 W.P. van Paassen - peter@paassen.tmfweb.nl

   This file is part of libmd5model
   
   libmd5model will attempt to parse ID's Doom3 mesh and animation model formats
   
   libmd5model is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your
   option) any later version.

   libmd5model is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with libmd5model; see the file COPYING.  If not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  */
%{
#include "md5.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
MD5_MESH_HEADER header;
MD5_JOINT* joints = (MD5_JOINT*)NULL;
MD5_MESH* meshes = (MD5_MESH*)NULL;
int count;
int mesh_count;

extern float sqrtf(float x);
extern char* strdup(const char* s);
extern int yylex(void);

void yyerror(char const* s)
{
	fprintf(stderr, "%s\n", s);
}

%}
%error-verbose

%union{
	int itype;
	float ftype;
	const char* ctype;
}

%token NUMBER
%token <itype> VERSIONID
%token <itype> CMDLINEID
%token <ctype> STRING
%token <itype> NUMJOINTSID
%token <itype> NUMMESHESID
%token <itype> JOINTSID
%token <itype> MESHID
%token <itype> LCURLY
%token <itype> RCURLY
%token <itype> LBRACKET
%token <itype> RBRACKET
%token <itype> SHADERID
%token <itype> WEIGHTID
%token <itype> TRIID
%token <itype> VERTID
%token <itype> NUMVERTSID
%token <itype> NUMTRISID
%token <itype> NUMWEIGHTSID

%%

md5_file:	md5_header md5_jointspart md5_meshpart
		;

md5_header:	md5_version md5_cmdline md5_numjoints md5_nummeshes
		;

md5_version:	VERSIONID NUMBER
		{
			header.version = $<itype>2;
		}
		;

md5_cmdline:	CMDLINEID STRING
		{
			header.commandline = (const char*)strdup($2);
		}
		;

md5_numjoints:	NUMJOINTSID NUMBER
		{
			header.num_joints = $<itype>2;
			joints = (MD5_JOINT*)malloc(header.num_joints * sizeof(MD5_JOINT));
			count = 0;
		}
		;

md5_nummeshes:	NUMMESHESID NUMBER
		{
			header.num_meshes = $<itype>2;
			meshes = (MD5_MESH*)malloc(header.num_meshes * sizeof(MD5_MESH));
			mesh_count = 0;
		}
		;

md5_jointspart:	JOINTSID LCURLY md5_joints RCURLY 
		{
			if (count != header.num_joints)
				yyerror("Error: Incorrect number of joints");	
		}
		;

md5_joints:	md5_joint
		| md5_joints md5_joint
		;

md5_joint:	STRING NUMBER LBRACKET NUMBER NUMBER NUMBER RBRACKET LBRACKET NUMBER NUMBER NUMBER RBRACKET
		{
			if (count < header.num_joints)
			{
				float tmp,q1,q2,q3;

				joints[count].name = (const char*)strdup($1);
				joints[count].parent.idx = $<itype>2;

				joints[count].position.x = $<ftype>4;
				joints[count].position.y = $<ftype>5;
				joints[count].position.z = $<ftype>6;

				q1 = $<ftype>9;
				q2 = $<ftype>10;
				q3 = $<ftype>11;
				
				joints[count].orientation.x = q1;
				joints[count].orientation.y = q2;
				joints[count].orientation.z = q3;
	
				tmp = 1.0f - ( q1 * q1 + q2 * q2 + q3 * q3 );
				if(tmp > 0.0f)
					joints[count].orientation.w = -sqrtf( tmp );
				else
					joints[count].orientation.w = 0.0f;
				count++;
			}
		}
		;

md5_meshpart:	md5_mesh
		| md5_meshpart md5_mesh

md5_mesh:	MESHID LCURLY md5_shader md5_numverts md5_verts md5_numtris md5_tris md5_numweights md5_weights RCURLY
		{
			if (count != meshes[mesh_count].num_weights)
				yyerror("Error: Incorrect number of weights in mesh");
			mesh_count++;
			count = 0;
		}
		;

md5_shader:	SHADERID STRING
		{
			meshes[mesh_count].shader = (const char*)strdup($2);	
		}
		;

md5_numverts:	NUMVERTSID NUMBER
		{
			meshes[mesh_count].num_verts = $<itype>2;	
			meshes[mesh_count].verts = (MD5_VERTEX*)malloc($<itype>2 * sizeof(MD5_VERTEX));
			count = 0;
		}
		;

md5_verts:	md5_vert
		| md5_verts md5_vert
		;

md5_vert:	VERTID NUMBER LBRACKET NUMBER NUMBER RBRACKET NUMBER NUMBER
		{
			if (count < meshes[mesh_count].num_verts)
			{
				/*meshes[mesh_count].verts[count].id = $<itype>2;*/	
				meshes[mesh_count].verts[count].texture_u = $<ftype>4;	
				meshes[mesh_count].verts[count].texture_v = $<ftype>5;	
				meshes[mesh_count].verts[count].weight.idx = $<itype>7;	
				meshes[mesh_count].verts[count].num_weights = $<itype>8;	
				count++;
			}
		}
		;

md5_numtris:	NUMTRISID NUMBER 
		{
			if (count != meshes[mesh_count].num_verts)
				yyerror("Error: Incorrect number of vertices in mesh");
			meshes[mesh_count].num_tris = $<itype>2;	
			meshes[mesh_count].tris = (MD5_TRIANGLE*)malloc($<itype>2 * sizeof(MD5_TRIANGLE));
			count = 0;
		}
		;

md5_tris:	md5_tri
		| md5_tris md5_tri
		;

md5_tri:	TRIID NUMBER NUMBER NUMBER NUMBER
		{
			if (count < meshes[mesh_count].num_tris)
			{
				/*meshes[mesh_count].tris[count].id = $<itype>2;*/	
				meshes[mesh_count].tris[count].vertex1.idx = $<itype>3;	
				meshes[mesh_count].tris[count].vertex2.idx = $<itype>4;	
				meshes[mesh_count].tris[count].vertex3.idx = $<itype>5;	
				count++;
			}
		}
		;

md5_numweights:	NUMWEIGHTSID NUMBER
		{
			if (count != meshes[mesh_count].num_tris)
				yyerror("Error: Incorrect number of triangles in mesh");
			meshes[mesh_count].num_weights = $<itype>2;	
			meshes[mesh_count].weights = (MD5_WEIGHT*)malloc($<itype>2 * sizeof(MD5_WEIGHT));
			count = 0;
		}
		;

md5_weights:	md5_weight
		| md5_weights md5_weight
		;

md5_weight:	WEIGHTID NUMBER NUMBER NUMBER LBRACKET NUMBER NUMBER NUMBER RBRACKET
		{
			if (count < meshes[mesh_count].num_weights)
			{
				/*meshes[mesh_count].weights[count].id = $<itype>2;*/	
				meshes[mesh_count].weights[count].joint.idx = $<itype>3;	
				meshes[mesh_count].weights[count].weight = $<ftype>4;	
				meshes[mesh_count].weights[count].offset_x = $<ftype>6;	
				meshes[mesh_count].weights[count].offset_y = $<ftype>7;	
				meshes[mesh_count].weights[count].offset_z = $<ftype>8;	
				count++;
			}
		}
		; 
%%


