/********************************************************************************
	eddml toolset - parses eddml files to extract error information
	Copyright (C) 2004 by Pete Rowley
	pete@openrowley.com

    This file is part of the eddml toolset.

    The eddml toolset is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The eddml toolset is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Foobar; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*********************************************************************************/

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"

#include "eddml.h"

#define MAX_ARG_MULTIPLE 50

#define OPT_VERSION 1
#define OPT_HELP 2
#define OPT_NODEFAULTS 3
#define OPT_NODEFAULT_FILE 4
#define OPT_NODEFAULT_SET 5

#define EDT_ENV_ERRORSETS "ERRORSETS"

struct _edt_opts
{
	int list;
	int brief;
	int verbose;
	int file_index;
	char *files[MAX_ARG_MULTIPLE+1];
	int path_index;
	char *paths[MAX_ARG_MULTIPLE+1];
	int search_index;
	char *searches[MAX_ARG_MULTIPLE+1];
	int nodefaultset;
	int nodefaultfile;
	char *error_text;
};

typedef struct _edt_opts edt_options;


void
help_exit()
{
	printf("etd [OPTION]... errorcode\n");
	printf("Look up textual description and diagnostic information for error codes\n\n");
	printf("  -l, --list            display descriptions in a terse single line format\n");
	printf("  -v, --verbose         display descriptions with additional information\n");
	printf("  -s, --search=NAME     search only error sets beginning with NAME\n");
	printf("  -f, --file=FILE       include FILE when resolving errors\n");
	printf("  -p, --path=PATH       include PATH when resolving errors\n");
	printf("      --no-default-set  do not use any default error set configuration\n");
	printf("      --no-default-file do not use any default file or path configuration\n");
	printf("      --no-defaults     do not use any default configuration\n");
	printf("      --help            display this help and exit\n");
	printf("      --version         display version information and exit\n");
	exit(0);
}


void
version_exit()
{
	printf("etd (eddml toolset) %s\n", VERSION);
	printf("Written by Pete Rowley\n\n");
	printf("Copyright (C) 2004/2005 Pete Rowley\n");
	printf("This is free software; see the source for copying conditions.  There is NO\n");
	printf("warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
	exit(0);
}


void
parse_options(int argc, char * const argv[], edt_options *edt_opts)
{
	int ret = 0;
	char *optstring = "lvs:f:p:";
	int flag;
	struct option opts[] =
	{
		/* name, hasarg, flag, val */
		{"list", 0, 0, 'l'},
		{"verbose", 0, 0, 'v'},
		{"search", 1, 0, 's'},
		{"file", 1, 0, 'f'},
		{"path", 1, 0, 'p'},
		{"no-defaults", 0, &flag, OPT_NODEFAULTS},
		{"no-default-set", 0, &flag, OPT_NODEFAULT_SET},
		{"no-default-file", 0, &flag, OPT_NODEFAULT_FILE},
		{"help", 0, &flag, OPT_HELP},
		{"version", 0, &flag, OPT_VERSION},
		{0,0,0,0}
	};

	memset(edt_opts, 0, sizeof(edt_options));

	while(1)
	{
		ret = getopt_long(argc, argv, optstring, opts, 0);

		if(ret == -1)
			break;

		switch(ret)
		{
			case 0:
			{
				if(flag == OPT_HELP)
				{
					help_exit();
				}
				else if(flag == OPT_VERSION)
				{
					version_exit();
				}
				else if(flag == OPT_NODEFAULTS)
				{
					edt_opts->nodefaultset = 1;
					edt_opts->nodefaultfile = 1;
				}
				else if(flag == OPT_NODEFAULT_SET)
				{
					edt_opts->nodefaultset = 1;
				}
				else if(flag == OPT_NODEFAULT_FILE)
				{
					edt_opts->nodefaultfile = 1;
				}
				break;
			}
			case 'l':
			{
				/* list */
				edt_opts->list = 1;
				break;
			}
			case 'v':
			{
				/* verbose */
				edt_opts->verbose = 1;
				break;
			}
			case 's':
			{
				/* search */
				if(edt_opts->search_index < MAX_ARG_MULTIPLE)
				{
					edt_opts->searches[edt_opts->search_index] = strdup(optarg);
					edt_opts->search_index++;
				}
				else
				{
					fprintf(stderr, "Error: too many search arguments.\n");
					exit(1);
				}
				break;
			}
			case 'f':
			{
				/* file */
				if(edt_opts->file_index < MAX_ARG_MULTIPLE)
				{
					edt_opts->files[edt_opts->file_index] = optarg;
					edt_opts->file_index++;
				}
				else
				{
					fprintf(stderr, "Error: too many file arguments.\n");
					exit(1);
				}
				break;
			}
			case 'p':
			{
				/* path */
				if(edt_opts->path_index < MAX_ARG_MULTIPLE)
				{
					edt_opts->paths[edt_opts->path_index] = optarg;
					edt_opts->path_index++;
				}
				else
				{
					fprintf(stderr, "Error: too many path arguments.\n");
					exit(1);
				}
				break;
			}
			default:
				exit(1);
		}
	}

	if(edt_opts->list && edt_opts->verbose)
	{
		fprintf(stderr, "Error: the --list and --verbose options are mutally exclusive.\n");
		exit(1);
	}

	/* if we have no search sets parse the environment search sets */
	if(0 == edt_opts->searches[0] && 0 == edt_opts->nodefaultset)
	{
		char *env = getenv(EDT_ENV_ERRORSETS);
		if(env && *env)
		{
			/* parse paths */
			char *pos;
			char *start = env;;

			pos = strchr(start, ':');
			while(start)
			{
				int end = pos? (int)(pos - start):0;

				if(pos == 0)
				{
					/* last path */
					if(edt_opts->search_index < MAX_ARG_MULTIPLE)
					{
						edt_opts->searches[edt_opts->search_index] = strdup(start);
						edt_opts->search_index++;
					}
					else
					{
						fprintf(stderr, "Error: too many search arguments.\n");
						exit(1);
					}
					break;
				}
				else
				{
					char *buf =(char*)malloc(end+2);
					strncpy(buf, start, end);
					buf[end+1] = 0;

					if(edt_opts->search_index < MAX_ARG_MULTIPLE)
					{
						edt_opts->searches[edt_opts->search_index] = buf;
						edt_opts->search_index++;
					}
					else
					{
						fprintf(stderr, "Error: too many search arguments.\n");
						exit(1);
					}
				}

				start = (&start[end] + 1);
				if(*start)
				{
					pos = strchr(start, ':');
				}
			}
		}
	}

	if(0 == edt_opts->list + edt_opts->verbose)
	{
		/* default to brief */
		edt_opts->brief = 1;
	}

	if (optind < argc) {
		/* get the requested error code */
		edt_opts->error_text = argv[optind];
	}
}


int
display_errors(edt_options *edt_opts)
{
	int ret;
	eddml_translator *t;
	char *bad_files[MAX_ARG_MULTIPLE+1];
	int i;

	/* get set up for translation */
	ret = eddml_init();
	if(ret)
	{
		fprintf(stderr, "Error: could not initialize libeddml.so.\n");
		goto end;
	}

	ret = eddml_get_translator(&t);
	if(ret)
	{
		fprintf(stderr, "Error: could not get a translator.\n");
		goto cleanup;
	}

	/* process initialization file options */
	if(edt_opts->files[0])
	{
		bad_files[0] = 0;

		ret = eddml_use_translation_files(t, (const char**)edt_opts->files, bad_files);
		if(ret)
		{
			fprintf(stderr, "Error: setting translation files.\n");
			goto cleanup;
		}

		if(bad_files[0])
		{
			for(i = 0; bad_files[i]; i++)
			{
				fprintf(stderr, "Warning: %s is not a valid file and will be ignored.\n", bad_files[i]);
			}
		}
	}

	/* process initialization path options */
	if(edt_opts->paths[0])
	{
		bad_files[0] = 0;

		ret = eddml_use_translation_paths(t, (const char**)edt_opts->paths, bad_files);
		if(ret)
		{
			fprintf(stderr, "Error: setting translation paths.\n");
			goto cleanup;
		}

		if(bad_files[0])
		{
			for(i = 0; bad_files[i]; i++)
			{
				fprintf(stderr, "Warning: %s is not a valid path and will be ignored.\n", bad_files[i]);
			}
		}
	}

	/* use env set paths */
	if(!edt_opts->nodefaultfile)
	{
		ret = eddml_use_translation_defaults(t);
		if(ret)
		{
			fprintf(stderr, "Error: setting default paths.\n");
			goto cleanup;
		}
	}

	/* get errors */
	ret = eddml_translate( t, edt_opts->error_text, 0, (const char**)edt_opts->searches);
	if(ret == 0)
	{
		ret = eddml_first_error(t);
		if(ret != 0)
		{
			printf("The query matched nothing\n");
		}

		while(ret == 0)
		{
			char *set, *ver, *auth, *doc;
			char *code, *desc, *sev;
			char **titles, **bodies, **stitles, **sbodies;
			int i;

			/* it costs little to get everything up front - the work is already done */
			eddml_get_name(t, &set);
			eddml_get_version(t, &ver);
			eddml_get_authority(t, &auth);
			eddml_get_docs(t, &doc);

			eddml_get_error_code(t, &code);
			eddml_get_error_desc(t, &desc);
			eddml_get_error_severity(t, &sev);

			eddml_get_error_note_titles(t, &titles);
			eddml_get_error_note_bodies(t, &bodies);

			eddml_get_error_symbol_languages(t, &stitles);
			eddml_get_error_symbol_bodies(t, &sbodies);

			/* display as requested */
			if(edt_opts->list)
			{
				/* list format is compact single line */
				printf("%s (%s): %s\n", set, code, desc);

			}
			else if(edt_opts->brief || edt_opts->verbose)
			{
				/* brief provides all diagnostic information */
				printf("%s (%s)\n", set, ver);

				if(edt_opts->verbose)
				{
					/* verbose gives errorset information
					   in addition to the brief information
					*/
					printf("Authority: %s\nAuthoritive Documents: %s\n", auth, doc);
				}

				printf("Code: %s\nSeverity: %s\nDescription: %s\n", code, sev, desc);

				if(titles)
				{
					for(i=0; titles[i]; i++)
					{
						printf("\n%s: %s\n", titles[i], bodies[i]);
					}
				}

				if(stitles)
				{
					printf("\n");

					for(i=0; stitles[i]; i++)
					{
						printf("%s symbol: %s\n", stitles[i], sbodies[i]);
					}
				}
			}

			ret = eddml_next_error(t);
			if(ret == 0 && edt_opts->list == 0)
			{
				printf("\n=================\n");
			}
		}
	}
	else
	{
		fprintf(stderr, "Error: setting up translation.\n");
	}

cleanup:
	eddml_translation_free(&t);
	eddml_done();

end:
	return ret;
}

void
free_options(edt_options *edt_opts)
{
	int i = 0;

	while(edt_opts->searches[i])
	{
		free(edt_opts->searches[i]);
		i++;
	}
}

int
main(int argc, char * const argv[])
{
	int ret = 0;
	edt_options edt_opts;

	parse_options(argc, argv, &edt_opts);
	ret = display_errors(&edt_opts);
	free_options(&edt_opts);

	return ret;
}
