#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation.
 */

// This file contains definitions of static functions
// No exported functions or symbols should be in this file.


static int open_timeout = 0;
static pthread_mutex_t new_mutex = PTHREAD_MUTEX_INITIALIZER;

static int sdbh_writeheader (DBHashTable * dbh, int flush);
static int sdbh_updateBranch (DBHashTable *, FILE_POINTER);
static unsigned char sdbh_readBranches (DBHashTable *, FILE_POINTER);

static FILE_POINTER *sdbh_locate (DBHashTable *, FILE_POINTER *);
static FILE_POINTER *sdbh_locateFind (DBHashTable *, FILE_POINTER *, int);
static void sdbh_cuenta (unsigned char *, unsigned char, FILE_POINTER);
static FILE_POINTER sdbh_write (char, DBHashTable *, unsigned char);
static int sdbh_read (char, DBHashTable *, unsigned char);
static int sdbh_readheader (DBHashTable *);

#ifdef TURN
static FILE_POINTER sdbh_turnaround (FILE_POINTER);
#endif
static FILE_POINTER sdbh_z (unsigned char, unsigned char);
static unsigned char sdbh_cuantumR (FILE_POINTER, unsigned char, FILE_POINTER *);
static unsigned char sdbh_cuantum (FILE_POINTER, unsigned char, FILE_POINTER *);
static void sdbh_cifra (unsigned char *, unsigned char, unsigned char, FILE_POINTER *);

static int sdbh_reversebarre (DBHashTable *, FILE_POINTER, int);
static int sdbh_barre (DBHashTable *, FILE_POINTER, int);

static void sdbh_transfer (DBHashTable *);
static DBHashTable *sdbh_open_S (const char *, int, int);

static int sdbh_size (DBHashTable *, FILE_POINTER);

#ifdef PARALLEL_SAFE
static char *lock_name(const char *archivo);
#endif

/*************************************************************************/
static int my_open(const char *filename, int flags){
#ifndef HAVE_WINDOWS_H
    if (flags & O_CREAT) return open(filename, flags, 0640);
    return open(filename, flags);
#else
    if (flags & O_CREAT) return open(filename, flags|_O_BINARY, 0640);
    return open(filename, flags|_O_BINARY);
#endif
}

#if 0
// Faster without this.
static int
sync_fd (int fd) {
    /*if (fsync(fd) < 0) {
       // this should not happen, really.
       ERR("fsync(): %s\n", strerror(errno));
       return 0;
       } */
    return 1;
}
#endif

static int
advance_fp (DBHashTable * dbh, FILE_POINTER offset) {
    //if (!sync_fd(dbh->fd)) return 0;
    if(lseek (dbh->fd, (off_t) offset, SEEK_CUR) < 0) {
        errno = EBADF;
        ERR( "Error: advance_fp failed to advance fp %lld\n", (long long)offset);
        return 0;
    }
    return 1;
}

static int
place_fp_at (DBHashTable * dbh, off_t seek) {
    //if (!sync_fd(dbh->fd)) return 0;
    if(lseek (dbh->fd, (off_t) seek, SEEK_SET) != seek) {
        // this should not happen...
        errno = EBADF;
	ERR( "Error: sdbh_locate() cannot place file pointer at bof=0x%llu\n", (long long unsigned)seek);
        return 0;
    }
    return 1;
}

static FILE_POINTER
place_eof (DBHashTable * dbh) {
    //if (!sync_fd(dbh->fd)) return 0;
    FILE_POINTER eof;
    eof = (FILE_POINTER) lseek (dbh->fd, (off_t) 0L, SEEK_END);
    if(eof < sizeof (dbh_header_t)) {
        errno = EBADF;
        ERR( "Error: tell_eof() <  sizeof (dbh_header_t)\n");
        eof = -1;
    }
    if(eof < 0) {
        // this should not happen...
        errno = EBADF;
        ERR( "Error: tell_eof() cannot place file pointer at eof\n");
    }
    return eof;
}

  /* Funciones nivel 0: */
static void
sdbh_operate (DBHashTable * dbh) {
    if(dbh == NULL) {
        ERR ("Must define function operate(place*) before doing a sweep\n");
    }
}

static char *
sdbh_randomfilename (DBHashTable * dbh, char code) {
    time_t segundos;
    FILE_POINTER divisor;
    char *archivo;
    const char *tmpdir;
    if(!dbh) return NULL;
    // determine best tmp directory
    if(dbh->head_info->user_tmpdir == 0) {
        struct stat dbh_st, tmp_st;
        if (stat (dbh->path, &dbh_st) < 0){
            ERR("Unable to stat: %s\n", dbh->path);
            return NULL;
        }
#ifdef HAVE_WINDOWS_H
        const char * tmp = getenv("TEMP");
	if (!tmp || access(tmp,W_OK) < 0) tmp ="c:\\";
#else
        const char *tmp = "/tmp";
#endif

        if (stat (tmp, &tmp_st) < 0){
            ERR("Unable to stat: %s\n", tmp);
            return NULL;
        }
	if (dbh->tmpdir) {
                free(dbh->tmpdir);
        }
        dbh->tmpdir = NULL;
        // Make sure tmp dir is on same device.
        DBG("system temp dir is %s \n", tmp);
        if(dbh_st.st_dev == tmp_st.st_dev) {
            dbh->tmpdir = (char *)malloc (strlen (tmp) + 1);
	    if (dbh->tmpdir == NULL) {
		 ERR( "malloc dbh->tmpdir: %s\n", strerror(errno));
                 return NULL;
	    }   
	    strcpy (dbh->tmpdir, tmp);
        } else {
            // system tmp dir is not on the same device.    
#ifdef HAVE_WINDOWS_H 
            dbh->tmpdir = (char *)malloc (strlen (tmp) + 1);
	    strcpy (dbh->tmpdir, tmp);
#else
            if (*(dbh->path) =='/') {
                dbh->tmpdir = (char *)malloc (strlen (dbh->path) + 1);
	        if (dbh->tmpdir == NULL) {
		   ERR( "malloc dbh->tmpdir: %s\n", strerror(errno));
                   return NULL;
                }
        	strcpy (dbh->tmpdir, dbh->path);
                *(strrchr(dbh->tmpdir, '/')) = 0;
            } else {
                dbh->tmpdir = (char *)malloc (1024);
	        if (dbh->tmpdir == NULL) {
		   ERR( "malloc dbh->tmpdir: %s\n", strerror(errno));
                   return NULL;
                }
                if (!getcwd(dbh->tmpdir, 1024)){
		   ERR( "!getcwd: %s\n", strerror(errno));
                   return NULL;
                }
            }
#endif
        }
    }
    tmpdir = dbh->tmpdir;
 
    time (&segundos);
    int try=0;
retry:
    srand ((unsigned)segundos);
    divisor = RAND_MAX / 10000;
    // Usage of rand is not security related. Just to get a random temporary file name.
    // coverity[dont_call : FALSE]
    if((segundos = rand () / divisor) > 100000L)
        segundos = 50001;
    archivo = (char *)malloc (strlen (tmpdir) + 1 + 1 + 6 + 4 + 1);
    if (archivo == NULL) {
	 ERR( "malloc filename: %s\n", strerror(errno));
         return NULL;
    } 
#ifdef HAVE_WINDOWS_H 
    sprintf (archivo, "%s%c%c%ld.tmp", tmpdir, '\\', code, (long)segundos);
#else
    sprintf (archivo, "%s/%c%ld.tmp", tmpdir, code, (long)segundos);
#endif

    // if the file exists (fat chance), retry with a different random number...
    struct stat st;
    if (stat(archivo, &st)==0){
	if (segundos > 0) segundos--;
	else segundos++;
	if (try++ < 3) {
	    free(archivo);
            goto retry;
        }
	free(archivo);
	return NULL;
    }

    return archivo;
}  /*******************************************************************************/

#if 0
this is unreliable in Linux...
//mode=1 --> write lock
static void
lock_fd (int fd, int mode, const char *archivo) {
    //  return;
    // get the file lock, wait if necesary:
    struct flock lock;

    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;             // to eof  

    if(mode) {
        lock.l_type = F_WRLCK;
    } else {
        lock.l_type = F_RDLCK;
    }

    int try=0;
retry:
    if(fcntl (fd, F_SETLKW, &lock) < 0) {
	if (try++ < 3) {
	    sleep(1);
	    goto retry;
	}
        DBG( "fcntl(F_SETLKW)(%s): %s\n", archivo, strerror (errno));
    }
}

static void
unlock (int fd) {


    // release the lock:
    struct flock lock;

    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;             // to eof  

    lock.l_type = F_UNLCK;
    int try=0;
retry:
    if(fcntl (fd, F_SETLKW, &lock) < 0) {
	if (try++ < 3) {
	    sleep(1);
	    goto retry;
	}
        DBG( "fcntl(F_SETLKW): %s\n", strerror (errno));
    }
}
#endif


  /* Functions: ************************************************************** */
static unsigned char
sdbh_readBranches (DBHashTable * dbh, FILE_POINTER seek) {
    FILE_POINTER *rama;
    unsigned char ramas;
    rama = dbh->newbranch;
    TRACE ("sdbh_readBranches\n");

    if(!place_fp_at (dbh, (off_t)seek)){
        return 0;
    }
    if(read (dbh->fd, &ramas, 1) != 1) {
        ERR ("sdbh_readBranches error 3.1\n");
        return 0;
    }
    if (ramas > dbh->head_info->n_limit){
        ERR ("Corrupted loop boundary \"ramas\"\n");
        return 0;
    }
    DBG ("sdbh_readBranches ramas=%lld\n", (long long)ramas);
    if(!advance_fp (dbh, 1L + sizeof (FILE_POINTER))) {
        return 0;
    }

    if(read (dbh->fd, rama, sizeof (FILE_POINTER) * ramas) != sizeof (FILE_POINTER) * ramas) {
        ERR ("sdbh_readBranches error 3.2\n");
        return 0;
    }
    DBG ("read:");
    int i;
    for(i = 0; i < ramas; i++){
        DBG (" rama[%d]=0x%llu\n", i, (long long unsigned)(rama[i]));
    }
#ifdef TURN
    {
        int i;
        for(i = 0; i < ramas; i++)
            rama[i] = sdbh_turnaround (rama[i]);
    }
#endif
    return ramas;
} /***************************************************/

static int
sdbh_updateBranch (DBHashTable * dbh, FILE_POINTER seek) {
    TRACE ("sdbh_updateBranch\n");
    FILE_POINTER *rama;
    unsigned char ramas;
    rama = dbh->newbranch;
    if(!place_fp_at (dbh, (off_t)seek)) {
        return 0;
    }
    if(read (dbh->fd, &ramas, 1) != 1) {
        ERR ("sdbh_updateBranch read error 4.1\n");
        return 0;
    }

    if(!advance_fp (dbh, 1L + sizeof (FILE_POINTER))){
        return 0;
    }

#ifdef TURN
    {
        int i;
        for(i = 0; i < ramas; i++)
            rama[i] = sdbh_turnaround (rama[i]);
    }
#endif
    DBG ("Updating branch information for new record:");
    int i;
    if (ramas > dbh->head_info->n_limit){
        ERR ("Corrupted loop boundary \"ramas\"\n");
        return 0;
    }
    for(i = 0; i < ramas; i++){
        DBG (" rama[%d]=0x%llu\n", i, (long long unsigned)(rama[i]));
    }

    if(write (dbh->fd, rama, sizeof (FILE_POINTER) * ramas) != sizeof (FILE_POINTER) * ramas) {
        ERR ("sdbh_updateBranch write error 4.2\n");
        return 0;
    }
#ifdef TURN
    {
        int i;
        for(i = 0; i < ramas; i++)
            rama[i] = sdbh_turnaround (rama[i]);
    }
#endif
    return 1;
}  /*****************************************************/

#define BRANCHOFF (dbh->head_info->n_limit-dbh->newbranches)
#define CURRENTSEEK fp[0]
#define LASTSEEK fp[1]
#define CURR_BRANCH fp[2]
#define SWITCHED (dbh->head_info->reservedD)
//FILE_POINTER *fp must have size = 3
static FILE_POINTER *
sdbh_locate (DBHashTable * dbh, FILE_POINTER * fp) {
    TRACE ("sdbh_locate\n");
    FILE_POINTER currentseek,
      lastseek;
    int i;                      /*,offset; */
    lastseek = currentseek = 0;
    CURRENTSEEK = LASTSEEK = CURR_BRANCH = 0;   /* ERROR value assigned */
    currentseek = dbh->head_info->bof;

    if(!place_fp_at (dbh, (off_t)currentseek)) {
        return NULL;
    }

    if(!sdbh_read (NEW, dbh, 1)) {
        return fp;
    }
  loop:
#define OFFSET (i+(dbh->head_info->n_limit-dbh->newbranches))
    for(i = 0; i < dbh->newbranches; i++) {
        if(*(dbh->key + OFFSET) != *(dbh->newkey + OFFSET))
        {
            lastseek = currentseek;
            if((currentseek = *(dbh->newbranch + i)) == 0) {
                CURR_BRANCH = i;
                break;
            }
            if(!place_fp_at (dbh, (off_t)currentseek)) {
                return NULL;
            }
            if(!sdbh_read (NEW, dbh, 1)) {
                return fp;
            }
            goto loop;
        }
    }
    CURRENTSEEK = currentseek;
    LASTSEEK = lastseek;
    return fp;
}

static FILE_POINTER *
sdbh_locateFind (DBHashTable * dbh, FILE_POINTER * fp, int keys) {
    TRACE ("sdbh_locateFind\n");
    FILE_POINTER currentseek,
      lastseek;
    int i;
    lastseek = currentseek = 0;
    CURRENTSEEK = LASTSEEK = CURR_BRANCH = 0;   /* ERROR value assigned */
    currentseek = dbh->head_info->bof;

    if(!place_fp_at (dbh, (off_t)currentseek)) {
        return NULL;
    }

    if(!sdbh_read (NEW, dbh, 1)) {
        return fp;
    }
    if(keys > dbh->head_info->n_limit)
        keys = dbh->head_info->n_limit;
  loop:
    for(i = 0; i < dbh->newbranches - (dbh->head_info->n_limit - keys); i++) {
        /*  offset=i+(dbh->head_info->n_limit-dbh->newbranches); */
        if(*(dbh->key + OFFSET) != *(dbh->newkey + OFFSET)) {
            lastseek = currentseek;
            if((currentseek = *(dbh->newbranch + i)) == 0) {
                CURR_BRANCH = i;
                break;
            }
            if(!place_fp_at (dbh, (off_t)currentseek)) {
                return NULL;
            }
            if(!sdbh_read (NEW, dbh, 1)) {
                return fp;
            }
            goto loop;
        }
    }
    CURRENTSEEK = currentseek;
    LASTSEEK = lastseek;
    return fp;
}

#ifdef PARALLEL_SAFE
static char *lock_name(const char *archivo){
    if (!archivo || strlen(archivo)==0){
 	ERR("Cannot determine lock name for %s\n", 
		(archivo)?archivo:"NULL");
	return NULL;
   }

    char *path;

#ifdef HAVE_REALPATH
    path = realpath(archivo, NULL);
    if (!path) {
        ERR( "realpath(%s): %s\n", archivo, strerror(errno));
    }
#else
    path = (char *)malloc(strlen(archivo)+1);
    if (path) strcpy(path, archivo);
#endif

    if (!path){
        ERR( "1.malloc path %s: %s\n", path, strerror(errno));
        return NULL;
    }

 
    // Get inode information:
    struct stat st;
    if (stat(path, &st) < 0){
	// This is a valid return value. 
	NOOP("Cannot stat %s: %s\n", path, strerror(errno));
	free(path);
	return NULL;
    }

    // Build shm_name
    char buffer[1024];
    memset(buffer, 0, 1024);
    sprintf(buffer, "/%d-%d", (int)st.st_dev, (int)st.st_ino);

    char *shm_name = (char *)malloc((strlen(buffer)+1)*sizeof(char));
    if (!shm_name) {
 	ERR("Cannot malloc lock name for %s\n", path);
	return NULL;
   }
    memset(shm_name, 0, strlen(buffer)+1);
    sprintf(shm_name, "%s", buffer);
    free(path);
    return shm_name;
}

static char *sem_name(const char *archivo){
    char *untagged_name = lock_name(archivo);
    if (!untagged_name) return NULL;
    char *name = (char *)malloc((strlen(untagged_name)+strlen("-ns")+1)*sizeof(char));
    if (!name){
        errno=ENOMEM;
        free(untagged_name);
        return NULL;
    }
    sprintf(name, "%s-ns", untagged_name);
    free(untagged_name);
    return name;
}





#endif

static dbh_lock_t *
open_shm_lock(const char *archivo){
#ifndef PARALLEL_SAFE
    return NULL;
#else
    char *shm_name=lock_name(archivo);
    if (!shm_name){
	// This would indicate something is wrong...
	ERR("Cannot get lock name for %s\n", 
		archivo);
	return NULL;
    }

    int fd = shm_open (shm_name, O_RDWR, S_IRUSR | S_IWUSR);
    if (fd > 0){
	NOOP("Lockfile exists %s -> %s\n", archivo, shm_name);
    } else {
	fd = shm_open (shm_name, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
    }
    if(fd < 0) {
	ERR("Cannot open shared memory file descriptor for %s (%s): %s\n", 
		shm_name, archivo, strerror (errno));
	free(shm_name);
	return NULL;
    } else {
	if(ftruncate (fd, sizeof(dbh_lock_t)) < 0) {
	    ERR("Cannot ftruncate shared memory item for %s: %s\n", 
		    archivo, strerror (errno));
	    free(shm_name);
	    close(fd);
	    return NULL;
	}
    }
    dbh_lock_t *lock_p = (dbh_lock_t *) mmap (NULL, sizeof(dbh_lock_t),
	PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    if (lock_p == MAP_FAILED){
	ERR("Cannot mmap shared memory item for %s: %s\n", 
	    archivo, strerror (errno));
	close (fd);
	shm_unlink (shm_name);
        char *nsem_name = sem_name(archivo);
	if (nsem_name){
            sem_unlink (nsem_name);
            free(nsem_name);
        }
	free(shm_name);
	return NULL;
    } 

    
    free(shm_name);
    close (fd);
    return lock_p;
#endif
}

static void
destroy_shm_lock(char *archivo, dbh_lock_t *lock_p){
#ifdef PARALLEL_SAFE
    if (!lock_p) return;
    char *shm_name=lock_name(archivo);
    if (!shm_name) {
	// Nothing to destroy.
	return;
    }
    munmap (lock_p, sizeof(dbh_lock_t));
    shm_unlink (shm_name);
    free(shm_name);
#endif
}

static void
clear_shm_lock(const char *archivo){
#ifndef PARALLEL_SAFE
    return;
#else
    char *shm_name=lock_name(archivo);
    if (!shm_name){
	// Nothing to clear.
	return;
    }
    // open it.
    dbh_lock_t *lock_p = open_shm_lock(archivo);
    // clear it.

    lock_p->write_lock_count = 0;
    lock_p->write_lock = 0;
    lock_p->read_lock_count = 0;
    
    if(msync (lock_p,  sizeof(dbh_lock_t), MS_ASYNC) < 0){
            ERR("(1) Cannot msync shared memory item for %s: %s\n", 
                archivo, strerror (errno));
    }
        
    munmap (lock_p, sizeof(dbh_lock_t));
    // erase it (if not open elsewhere)
    shm_unlink (shm_name);
    free(shm_name);
    char *nsem_name = sem_name(archivo);
    if (nsem_name){
        sem_unlink(nsem_name);
        free(nsem_name);
    }
    return;
#endif
}


static DBHashTable * 
sdbh_create (const char *path, unsigned char key_length, int flags) {
    DBHashTable *dbh;
    if (key_length > 254){
        ERR("sdbh_create(%s): key_length is limited to 254 bytes...\n", path);
        return NULL;
    }
    dbh = (DBHashTable *) malloc (sizeof (DBHashTable));
    if(dbh == NULL){
	ERR( "malloc(%ld): %s\n", (long)sizeof (DBHashTable), strerror(errno));
	return NULL;
    }
    memset (dbh, 0, sizeof (DBHashTable));
    dbh->lock_attempt_limit = open_timeout;
    /////////////////////////////////////////////
    
    unlink (path);
    dbh->fd = my_open (path, O_CREAT | O_RDWR | O_TRUNC);
    if(dbh->fd < 0) {
        free (dbh);
        ERR("sdbh_create(%s): %s\n", path, strerror (errno));
        return NULL;
    }

    dbh->head_info = (dbh_header_t *) malloc (sizeof (dbh_header_t));
     if (dbh->head_info == NULL) {
        free (dbh);
	ERR( "malloc(%ld): %s\n", (long)sizeof (dbh_header_t), strerror(errno));
        return NULL;
     }  
     
    memset ((void *)(dbh->head_info), 0, sizeof (dbh_header_t));
#ifdef PARALLEL_SAFE
    if (flags & DBH_PARALLEL_SAFE){
        char *nsem_name = sem_name(path);
        if (!nsem_name) {
	    ERR( "malloc(%ld): %s\n", (long)strlen(path), strerror(errno));
            free (dbh);
            return NULL;
        }
        dbh->sem = sem_open(nsem_name, O_CREAT, 0700, 1);
        free(nsem_name);

        clear_shm_lock(path);
        dbh->lock_p = open_shm_lock(path);
    }
#endif

    dbh->head_info->bof = sizeof (dbh_header_t);
    dbh->head_info->n_limit = key_length;
    dbh->head_info->user_tmpdir = 0;
    dbh->head_info->sweep_erased = 0;
    dbh->head_info->total_space = dbh->head_info->data_space = dbh->head_info->erased_space = 0;
    strncpy (dbh->head_info->version, DBH_FILE_VERSION, 15);
    dbh->head_info->records = 0;
    dbh->operate = sdbh_operate;
    dbh->branch = (FILE_POINTER *) malloc (dbh->head_info->n_limit * sizeof (FILE_POINTER));
     if (dbh->branch == NULL) {
	 ERR( "malloc dbh->branch: %s\n", strerror(errno));
         dbh_close(dbh);
        return NULL;
     }      
     dbh->newbranch = (FILE_POINTER *) malloc (dbh->head_info->n_limit * sizeof (FILE_POINTER));
      if (dbh->newbranch == NULL) {
	 ERR( "malloc dbh->newbranch: %s\n", strerror(errno));
         dbh_close(dbh);
        return NULL;
     }        
     dbh->key = (unsigned char *)malloc (key_length);
     if (dbh->key == NULL) {
	 ERR( "malloc dbh->key: %s\n", strerror(errno));
         dbh_close(dbh);
        return NULL;
     }  
     dbh->newkey = (unsigned char *)malloc (key_length);
     if (dbh->newkey == NULL) {
	 ERR( "malloc dbh->newkey: %s\n", strerror(errno));
         dbh_close(dbh);
        return NULL;
     }          
     dbh->path = (char *)malloc (strlen (path) + 1);
     if (dbh->path == NULL || dbh->newbranch == NULL ||
	    dbh->key == NULL || dbh->newkey == NULL || dbh->path == NULL) {
	 ERR( "malloc dbh->path: %s\n", strerror(errno));
         dbh_close(dbh);
        return NULL;
     }  

    strcpy (dbh->path, path);

    dbh->head_info->record_length = 0;
    dbh->head_info->writeOK = 1;
    sdbh_size (dbh, DEFAULT_DBH_DATASIZE);
    dbh->bytes_userdata = 0;

    sdbh_writeheader (dbh, 1);
    DBG (" created %s \n", path);
    return dbh;
}

static DBHashTable *
sdbh_open_S (const char *archivo, int mode, int flags) {
    TRACE ("sdbh_open_S: %s\n", archivo);
    DBHashTable *dbh;
    int fd;
    if(mode) {
        fd = my_open (archivo, O_RDWR);
    } else {
        fd = my_open (archivo, O_RDONLY);
    }



    if(fd < 0) {
        DBG ("open(%s) %s failed: %s\n", archivo, (mode)?"O_RDWR":"O_RDONLY", strerror (errno));
        return NULL;
    }
    dbh = (DBHashTable *) malloc (sizeof (DBHashTable));
     if (dbh == NULL) {
	DBG( "malloc(%ld) failed: %s\n", (long)sizeof (DBHashTable), strerror(errno));
        close(fd);
        return NULL;
     }          
    memset (dbh, 0, sizeof (DBHashTable));
    dbh->lock_attempt_limit = open_timeout;
    dbh->fd = fd;

    /////////////////////////////////////////////
#ifdef PARALLEL_SAFE
    if (flags & DBH_PARALLEL_SAFE){
        char *nsem_name = sem_name(archivo);
        if (!nsem_name) {
	    DBG( "sem_name(%s) failed: %s\n", archivo, strerror(errno));
            dbh_close(dbh);
            return NULL;
        }
        dbh->sem = sem_open(nsem_name, O_CREAT, 0700, 1);
        free(nsem_name);
        dbh->lock_p = open_shm_lock(archivo);
    }
#endif

    dbh->head_info = (dbh_header_t *) malloc (sizeof (dbh_header_t));
     if (dbh->head_info == NULL) {
	DBG( "malloc(%ld) failed: %s\n", (long)sizeof (dbh_header_t), strerror(errno));
        dbh_close(dbh);
        return NULL;
     }      
    FILE_POINTER eof = place_eof (dbh);
    if(eof < 0) {
	DBG( "place_eof(%s) failed: %s\n", archivo, strerror(errno));
        dbh_close(dbh);
        return NULL;
    }

    dbh->path = (char *)malloc (strlen (archivo) + 1);
     if (dbh->path == NULL) {
	DBG( "malloc(%ld) failed: %s\n", (long)(strlen (archivo) + 1), strerror(errno));
        dbh_close(dbh);
        return NULL;
     }     
     strcpy (dbh->path, archivo);

    if(!sdbh_readheader (dbh)) {
        dbh_close(dbh);
	DBG( "readheader(%s) failed: %s\n", archivo, strerror(errno));
        return NULL;
    }
    dbh->operate = sdbh_operate;
    dbh->branch = (FILE_POINTER *) malloc (dbh->head_info->n_limit * sizeof (FILE_POINTER));
     if (dbh->branch  == NULL) {
	ERR( "malloc dbh->branch: %s\n", strerror(errno));
        dbh_close(dbh);
        return NULL;
     }     
    dbh->newbranch = (FILE_POINTER *) malloc (dbh->head_info->n_limit * sizeof (FILE_POINTER));
     if (dbh->newbranch  == NULL) {
	ERR( "malloc dbh->newbranch: %s\n", strerror(errno));
        dbh_close(dbh);
        return NULL;
     }     
    dbh->key = (unsigned char *)malloc (dbh->head_info->n_limit);
      if (dbh->key == NULL) {
	 ERR( "malloc dbh->key: %s\n", strerror(errno));
        dbh_close(dbh);
        return NULL;
     }     
   dbh->newkey = (unsigned char *)malloc (dbh->head_info->n_limit);
     if (dbh->newkey  == NULL) {
	 ERR( "malloc dbh->newkey: %s\n", strerror(errno));
        dbh_close(dbh);
        return NULL;
     }     
    if(mode == WRITE)
        dbh->head_info->writeOK = 1;
    else
        dbh->head_info->writeOK = 0;
    sdbh_size (dbh, (int)dbh->head_info->record_length);
    DBG (" opened %s \n", archivo);
    return dbh;
}  /****************************************************************************/

static void
sdbh_cuenta (unsigned char *numero, unsigned char orden, FILE_POINTER cuanto) {
    unsigned char q;
    FILE_POINTER m;
    if(orden == 0)
        numero[0] = 0;
    if(orden == 1)
        numero[0] = (unsigned char)cuanto;
    q = sdbh_cuantumR (cuanto, orden, &m);       /* en que cuantum estamos */
    orden--;
    sdbh_cifra (numero, q, orden, &m);
/* sdbh_cifra(numero,q,orden-1,&m);*/
} /***********************************************************************/

#define RECORD_EXTENT  ( 2 + sizeof(FILE_POINTER)*(*how_many_branches+1) + dbh->head_info->n_limit + the_user_databytes )
static FILE_POINTER
sdbh_write (char newinfo, DBHashTable * dbh, unsigned char write_branches) {
    TRACE ("sdbh_write\n");
    int i;
    void *the_data;
    unsigned char *how_many_branches;
    FILE_POINTER the_user_databytes;
    FILE_POINTER *the_branches;
    unsigned char *the_key;
    if(newinfo) {
        the_branches = dbh->newbranch;
    } else {
        the_branches = dbh->branch;
    }
    the_data = dbh->data;
    how_many_branches = &(dbh->newbranches);
    the_user_databytes = dbh->bytes_userdata;
    the_key = dbh->key;

    if(the_user_databytes > dbh->head_info->record_length) {
        ERR(
                 "dbh->bytes_userdata (%lld) is greater than dbh->head_info->record_length (%lld). This is wrong and I stubbornly refuse to write\n",
                 (long long)dbh->bytes_userdata, (long long)dbh->head_info->record_length);
        ERR("*** sdbh_write() error 1\n");
        return 0;
    }
    DBG("Starting write at offset %lld\n", (long long)lseek(dbh->fd, 0, SEEK_CUR));
    if(write (dbh->fd, how_many_branches, 1) != 1){
        ERR("*** sdbh_write() error 2\n");
        return 0;
    }
    if(write (dbh->fd, &(dbh->flag), 1) != 1){
        ERR("*** sdbh_write() error 3\n");
        return 0;
    }

    i = write (dbh->fd, &the_user_databytes, sizeof (FILE_POINTER));
    if(i != sizeof (FILE_POINTER)){
        ERR("*** sdbh_write() error 4\n");
        return 0;
    }

    if(write_branches) {
#ifdef TURN
        for(i = 0; i < *how_many_branches; i++)
            the_branches[i] = sdbh_turnaround (the_branches[i]);
#endif
        i = write (dbh->fd, the_branches, sizeof (FILE_POINTER) * (*how_many_branches));
        if(i != sizeof (FILE_POINTER) * (*how_many_branches)){
                ERR("*** sdbh_write() error 5\n");
                return 0;
        }

#ifdef TURN
        for(i = 0; i < *how_many_branches; i++)
            the_branches[i] = sdbh_turnaround (the_branches[i]);
#endif
    } else {
        if(!advance_fp (dbh, *how_many_branches * sizeof (FILE_POINTER))){
        ERR("*** sdbh_write() error 6\n");
        return 0;
    }


    }
    if(write (dbh->fd, the_key, dbh->head_info->n_limit) != dbh->head_info->n_limit) {
        ERR( "fwrite: %s\n", strerror (errno));
        return 0;
    }
    if(the_user_databytes) {
        if(dbh->head_info->reservedC){
            the_user_databytes = write (dbh->fd, the_data, dbh->head_info->record_length);
	} else {
            the_user_databytes = write (dbh->fd, the_data, the_user_databytes);
	}
        return (the_user_databytes);
    } 
        NOOP("*** sdbh_write() Data length for record is zero.\n");
    return (0);
    
}  /************************************************************************************/

static int
sdbh_read (char newinfo, DBHashTable * dbh, unsigned char read_data) {
    TRACE ("sdbh_read\n");
    unsigned char keylength;
    void *the_data;
    unsigned char *how_many_branches;
    FILE_POINTER *the_user_databytes;
    FILE_POINTER *the_branches;
    unsigned char *the_key;
    if(newinfo) {
        the_branches = dbh->newbranch;
        the_data = dbh->newdata;
        how_many_branches = &(dbh->newbranches);
        the_user_databytes = &(dbh->newbytes_userdata);
        the_key = dbh->newkey;
    } else {
        the_branches = dbh->branch;
        the_data = dbh->data;
        how_many_branches = &(dbh->branches);
        the_user_databytes = &(dbh->bytes_userdata);
        the_key = dbh->key;
    }
    keylength = dbh->head_info->n_limit;

    if(read (dbh->fd, how_many_branches, 1) != 1) {
        DBG ("fread error 2.1 (on empty dbh file this is OK)\n");
        return 0;
    }
    if(read (dbh->fd, &(dbh->flag), 1) != 1) {

        ERR ("fread error 2.2\n");
        return 0;
    }
    // the_user_databytes is *not* a string. It is a FILE_POINTER (gint64)
    // coverity[string_null_argument : FALSE]
    if(read (dbh->fd, the_user_databytes, sizeof (FILE_POINTER)) != sizeof (FILE_POINTER)) {
        ERR ("fread error 2.3\n");

        return 0;
    }

    if(*the_user_databytes > dbh->head_info->record_length) {
        ERR(
                 "sdbh_read(): the_user_databytes (%lld) is greater than dbh->head_info->record_length (%lld). This is wrong and I stubbornly refuse to read\n",
                 (long long)the_user_databytes, (long long)(dbh->head_info->record_length));
        return 0;
    }
    if(*the_user_databytes == 0) {
        ERR ( "dbh_RECORD_SIZE() == 0. If this is not intentional, use dbh_set_recordsize() to set record size for %s.\n", dbh->path);
        return 0;
    }
    // if data is tainted, there will be no overrun since value is checked against
    // dbh->head_info->record_length above.
    // coverity[tainted_data : FALSE]
    if(read (dbh->fd, the_branches, sizeof (FILE_POINTER) * (*how_many_branches)) != sizeof (FILE_POINTER) * (*how_many_branches)) {
        ERR ("fread error 2.4\n");

        return 0;
    }
#ifdef TURN
    {
        int i;
        for(i = 0; i < *how_many_branches; i++)
            the_branches[i] = sdbh_turnaround (the_branches[i]);
    }
#endif
    if(read (dbh->fd, the_key, keylength) != keylength) {
        ERR ("fread error 2.5\n");
        return 0;
    }
    if(read_data) {
        // if data is tainted, there will be no overrun since value is checked against
        // dbh->head_info->record_length above.
        // coverity[tainted_data : FALSE]
        if(read (dbh->fd, the_data, *the_user_databytes) != *the_user_databytes) {
            ERR ("fread error 2.51: blocksize=%lld\n", (long long)(*the_user_databytes));

            return 0;
        }
    }
    return 1;
} /************************************************************************************/

static int
sdbh_readheader (DBHashTable * dbh) {
    TRACE ("sdbh_readheader\n");
    if(!place_fp_at (dbh, (off_t)0)) {
            ERR("*** sdbh_readheader() error 1; !place_fp_at (dbh, 0)\n");
        return 0;
    }
    // dbh->head_info is *not* a string. It is a fixed length structure
    // coverity[string_null_argument : FALSE]
    size_t bytes = read (dbh->fd, (void *)(dbh->head_info), sizeof (dbh_header_t));

    if (bytes != sizeof (dbh_header_t)) {
        ERR( "Failed to read header for %s: %s\nsizeof (dbh_header_t)=%ld read=%ld\n", 
                dbh->path, strerror (errno),
                 (long)sizeof (dbh_header_t), (long)bytes);
        return 0;
    }
    if(strncmp (dbh->head_info->version, DBH_FILE_VERSION, 15) != 0) {
        ERR(
                 "Failed to read header for %s at sdbh_readheader(): strncmp (\"%s\",\"%s\")\n",
                 dbh->path, dbh->head_info->version, DBH_FILE_VERSION);
        return 0;
    }
    /* volatile values: */
    dbh->head_info->user_tmpdir = 0;

#ifdef TURN
    dbh->head_info->bof = sdbh_turnaround (dbh->head_info->bof);
    dbh->head_info->record_length = sdbh_turnaround (dbh->head_info->record_length);
    dbh->head_info->total_space = sdbh_turnaround (dbh->head_info->total_space);
    dbh->head_info->data_space = sdbh_turnaround (dbh->head_info->data_space);
    dbh->head_info->erased_space = sdbh_turnaround (dbh->head_info->erased_space);
    dbh->head_info->records = sdbh_turnaround (dbh->head_info->records);
#endif
    return 1;
}  /************************************************************************************/

#ifdef TURN
static FILE_POINTER
sdbh_turnaround (FILE_POINTER x) {
    char *where,
      temp;
    where = (char *)&x;
    temp = where[0];
    where[0] = where[3];
    where[3] = temp;
    temp = where[1];
    where[1] = where[2];
    where[2] = temp;
    return x;
} /************************************************************************************/
#endif

static FILE_POINTER
sdbh_z (unsigned char n, unsigned char m) {      /* n=cuantum m=orden */
    FILE_POINTER s = 0;
    unsigned char i;
    if(m < 2)
        return 1;
    if(m == 2)
        return (n + 1);
    m--;
    for(i = n; i > 0; i--)
        s += sdbh_z (i, m);
/* for (i=n;i>0;i--) s += sdbh_z(i,m-1);*/
    return (s + 1);
}  /************************************************************************************/

static unsigned char
sdbh_cuantumR (FILE_POINTER cuanto, unsigned char orden, FILE_POINTER * m) {
    unsigned char i;
    FILE_POINTER ztotal,
      grandtotal = 0;
    for(i = 0; 1; i++) {
        if(i == orden)
            ztotal = grandtotal;
        else
            ztotal = sdbh_z (i, orden);
        grandtotal += ztotal;
        if(cuanto <= grandtotal) {
            if(m)
                *m = grandtotal - cuanto + 1;
            break;
        }
    }
    return i;
}  /************************************************************************************/

static unsigned char
sdbh_cuantum (FILE_POINTER cuanto, unsigned char orden, FILE_POINTER * m) {
    unsigned char i;
    FILE_POINTER ztotal,
      grandtotal = 0;
    for(i = 0; 1; i++) {
        if(i == orden)
            ztotal = grandtotal;
        else
            ztotal = sdbh_z (i, orden);
        grandtotal += ztotal;
        if(cuanto <= grandtotal)
            break;
        if(m)
            *m = cuanto - grandtotal;
    }
    return i;
}  /************************************************************************************/

static void
sdbh_cifra (unsigned char *numero, unsigned char q, unsigned char orden, FILE_POINTER * m) {
    unsigned char subcuantum;
    int t;
    if(orden == 1) {
        t = q - (*m - 1);
        numero[0] = (unsigned char)t;
/*  numero[0]=q - (*m-1);*/
        t = *m - 1;
        numero[1] = (unsigned char)t;
/*  numero[1]= *m-1;*/
        return;
    }
    subcuantum = sdbh_cuantum (*m, orden, m);
    t = q - subcuantum;
    numero[0] = (unsigned char)t;
/* numero[0]=q-subcuantum;*/
    orden--;
    sdbh_cifra (numero + 1, subcuantum, orden, m);
    /* sdbh_cifra(numero+1,subcuantum,orden-1,m); */
    return;
}  /************************************************************************************/

#if 0
// legacy code
static int
sdbh_barrelong (DBHashTable * dbh, FILE_POINTER startadd, int ramas) {
    unsigned char i;
    if(dbh_load_address (dbh, startadd) == 0)
        return 0;
    if(dbh->head_info->dbh_exit)
        return 2;
    if(!ERASED || (ERASED && dbh->head_info->sweep_erased))
        (*(dbh->operate)) (dbh);
    if(ramas) {
        if((ramas = dbh->branches - ramas) < 0)
            ramas = 0;
    }
    for(i = dbh->branches; i > ramas; i--) {
        if(dbh->branch[i - 1])
            sdbh_barrelong (dbh, dbh->branch[i - 1], 0);
        dbh_load_address (dbh, startadd);
    }
    return 1;
}    /**********************************************************************/

static int
sdbh_reversebarrelong (DBHashTable * dbh, FILE_POINTER startadd, int ramas) {
    int i;
    if(dbh_load_address (dbh, startadd) == 0)
        return 0;
    if(dbh->head_info->dbh_exit)
        return 2;
    if(!ERASED || (ERASED && dbh->head_info->sweep_erased))
        (*(dbh->operate)) (dbh);
/* if (!ERASED) (*(dbh->operate))(dbh);*/
    if(ramas) {
        if((ramas = dbh->branches - ramas) < 0)
            ramas = 0;
    }
    for(i = ramas; i < dbh->branches; i++) {
        if(dbh->branch[i])
            sdbh_reversebarrelong (dbh, dbh->branch[i], 0);
        dbh_load_address (dbh, startadd);
    }
    return 1;
}   /******************************************************************************/
#endif

static int
sdbh_reversebarre (DBHashTable * dbh, FILE_POINTER startadd, int ramas) {
    int i;
    unsigned char oldbranches;
    FILE_POINTER *oldbranch;
    if(dbh_load_address (dbh, startadd) == 0)
        return 0;
    if(dbh->head_info->dbh_exit)
        return 2;
    if(!ERASED || (ERASED && dbh->head_info->sweep_erased))
        (*(dbh->operate)) (dbh);
/* if (!ERASED) (*(dbh->operate))(dbh);*/
    oldbranch = (FILE_POINTER *) malloc (dbh->branches * sizeof (FILE_POINTER));
     if (oldbranch == NULL) {
	 ERR( "malloc oldbranch: %s\n", strerror(errno));
        return 0;
     }     
    oldbranches = dbh->branches;
    if(ramas) {
        if((ramas = dbh->branches - ramas) < 0)
            ramas = 0;
    }
#if 1
    for(i = 0; i < oldbranches; i++)
        oldbranch[i] = dbh->branch[i];
    for(i = ramas; i < oldbranches; i++)
        if(oldbranch[i])
            sdbh_reversebarre (dbh, oldbranch[i], 0);
    free (oldbranch);
#else
// legacy code
    if(oldbranch == NULL) fullmemory = 1;
    char fullmemory = 0;
    if(!fullmemory) {
        for(i = 0; i < oldbranches; i++)
            oldbranch[i] = dbh->branch[i];
        for(i = ramas; i < oldbranches; i++)
            if(oldbranch[i])
                sdbh_reversebarre (dbh, oldbranch[i], 0);
        free (oldbranch);
    } else
        for(i = ramas; i < oldbranches; i++) {
            if(dbh->branch[i])
                sdbh_reversebarrelong (dbh, dbh->branch[i], 0);
            dbh_load_address (dbh, startadd);
        }
#endif
    return 1;
}  /******************************************************************************/

static int
sdbh_barre (DBHashTable * dbh, FILE_POINTER startadd, int ramas) {
    unsigned char i;
    FILE_POINTER *oldbranch;
    if(dbh_load_address (dbh, startadd) == 0)
        return 0;
    if(dbh->head_info->dbh_exit)
        return 2;
#if 0
    printf ("barre: currentseek at %lu\n", (long unsigned)startadd);
    printf ("record is erased? -> %d\n", ERASED);
#endif
    if(!ERASED || (ERASED && dbh->head_info->sweep_erased))
        (*(dbh->operate)) (dbh);
/* if (!ERASED) (*(dbh->operate))(dbh);*/
    oldbranch = (FILE_POINTER *) malloc (dbh->branches * sizeof (FILE_POINTER));
      if (oldbranch == NULL) {
	 ERR( "malloc oldbranch: %s\n", strerror(errno));
        return 0;
     }     
    if(ramas) {
        if((ramas = dbh->branches - ramas) < 0)
            ramas = 0;
    }
#if 1
    for(i = 0; i < dbh->branches; i++)
        oldbranch[i] = dbh->branch[i];
    for(i = dbh->branches; i > ramas; i--)
        if(oldbranch[i - 1])
            sdbh_barre (dbh, oldbranch[i - 1], 0);
    free (oldbranch);
#else
// legacy code
    char fullmemory = 0;
    if(!fullmemory) {
        for(i = 0; i < dbh->branches; i++)
            oldbranch[i] = dbh->branch[i];
        for(i = dbh->branches; i > ramas; i--)
            if(oldbranch[i - 1])
                sdbh_barre (dbh, oldbranch[i - 1], 0);
        free (oldbranch);
    } else
        for(i = dbh->branches; i > ramas; i--) {
            if(dbh->branch[i - 1])
                sdbh_barrelong (dbh, dbh->branch[i - 1], 0);
            dbh_load_address (dbh, startadd);
        }
#endif
    return 1;
}   /************************************************************************/

  /* Functions: */
static void
sdbh_transfer (DBHashTable * srcdbh) {
    DBHashTable *dbh_desdbh = srcdbh->dbh_desdbh;
    dbh_desdbh->bytes_userdata = srcdbh->bytes_userdata;
    dbh_update (dbh_desdbh);
/* if ((count++ % 100)==0) printf("\n %ld registros procesados",count);*/
    return;
} /***************************************************************************/

/*****************************************************************************/
/***    old   non static functions ***/
/*****************************************************************************/

static int
sdbh_size (DBHashTable * dbh, FILE_POINTER record_length) {

    /* despues de abrir DBHashTable */
    if(dbh == NULL) {
        ERR( "sdbh_size(): dbh != NULL not met\n");
        return 0;
    }
    if(dbh->head_info == NULL) {
        ERR( "sdbh_size(): dbh->head_info != NULL not met\n");
        return 0;
    }

    if(record_length < dbh->head_info->record_length) {
        ERR ("sdbh_size(): may not shrink! current size=%lld requested size=%ld\n",
               (long long)dbh->head_info->record_length, (long)record_length);
        return 0;
    }

    void *data = malloc (record_length);
     if (data == NULL) {
	 ERR( "malloc data: %s\n", strerror(errno));
        return 0;
     }     
    void *newdata = malloc (record_length);
     if (newdata == NULL) {
         free(data);
	 ERR( "malloc newdata: %s\n", strerror(errno));
        return 0;
     }     
    memset (data, 0, record_length);
    memset (newdata, 0, record_length);
    if(dbh->data != NULL) {
        memcpy (data, dbh->data, dbh->head_info->record_length);
        free (dbh->data);
    }
    if(dbh->newdata != NULL) {
        memcpy (newdata, dbh->newdata, dbh->head_info->record_length);
        free (dbh->newdata);
    }
    dbh->data = data;
    dbh->newdata = newdata;

    if (dbh->head_info->record_length != record_length) {
	dbh->head_info->record_length = record_length;
        if(dbh->head_info->writeOK) sdbh_writeheader (dbh, 1);
    }
    return 1;
} /****************************************************************************************/

#if 0
// legacy code
static int
sdbh_barrelong2 (DBHashTable * dbh, DBHashFunc2 operate, void *data, 
	    FILE_POINTER startadd, int ramas) {
    unsigned char i;
    if(dbh_load_address (dbh, startadd) == 0)
        return 0;
    if(dbh->head_info->dbh_exit)
        return 2;
    if(!ERASED || (ERASED && dbh->head_info->sweep_erased))
        (*(dbh->operate)) (dbh);
    if(ramas) {
        if((ramas = dbh->branches - ramas) < 0)
            ramas = 0;
    }
    for(i = dbh->branches; i > ramas; i--) {
        if(dbh->branch[i - 1])
            sdbh_barrelong2 (dbh, operate, data, dbh->branch[i - 1], 0);
        dbh_load_address (dbh, startadd);
    }
    return 1;
}    /**********************************************************************/
#endif

static int
sdbh_barre2 (DBHashTable * dbh, DBHashFunc2 operate, void *data, 
	FILE_POINTER startadd, int ramas) {
    unsigned char i;
    FILE_POINTER *oldbranch;
    if(dbh_load_address (dbh, startadd) == 0)
        return 0;
    if(dbh->head_info->dbh_exit)
        return 2;
#if 0
    printf ("barre: currentseek at %lu\n", (long unsigned)startadd);
    printf ("record is erased? -> %d\n", ERASED);
#endif
    if(!ERASED || (ERASED && dbh->head_info->sweep_erased))
        (operate) (dbh, data);
/* if (!ERASED) (*(dbh->operate))(dbh);*/
    oldbranch = (FILE_POINTER *) malloc (dbh->branches * sizeof (FILE_POINTER));
      if (oldbranch == NULL) {
	 ERR( "malloc oldbranch: %s\n", strerror(errno));
        return 0;
     }     
    if(ramas) {
        if((ramas = dbh->branches - ramas) < 0)
            ramas = 0;
    }
#if 1
    for(i = 0; i < dbh->branches; i++)
        oldbranch[i] = dbh->branch[i];
    for(i = dbh->branches; i > ramas; i--)
        if(oldbranch[i - 1])
            sdbh_barre2 (dbh, operate, data, oldbranch[i - 1], 0);
    free (oldbranch);
#else
// legacy code
    char fullmemory = 0;
    if(!fullmemory) 
    {
        for(i = 0; i < dbh->branches; i++)
            oldbranch[i] = dbh->branch[i];
        for(i = dbh->branches; i > ramas; i--)
            if(oldbranch[i - 1])
                sdbh_barre2 (dbh, operate, data, oldbranch[i - 1], 0);
        free (oldbranch);
    } else
        for(i = dbh->branches; i > ramas; i--) {
            if(dbh->branch[i - 1])
                sdbh_barrelong2 (dbh, operate, data, dbh->branch[i - 1], 0);
            dbh_load_address (dbh, startadd);
        }
#endif
    return 1;
}   /************************************************************************/


static int
sdbh_newbarre2 (DBHashTable * dbh, DBHashFunc2 operate, void *data) {
    if(!dbh) {
        ERR( "sdbh_newbarre(): %s\n", strerror (EBADF));
        return 0;
    }

    dbh->head_info->dbh_exit = 0;
    sdbh_barre2 (dbh, operate, data, dbh->head_info->bof, 0);



    return 1;
} /***************************************************************************************/

static int
sdbh_newbarre (DBHashTable * dbh, unsigned char *key1, unsigned char *key2, unsigned char keylength) {
    int i;
    FILE_POINTER pointer;
    if(!dbh) {
        ERR( "sdbh_newbarre(): %s\n", strerror (EBADF));

        return 0;
    }

    dbh->head_info->dbh_exit = 0;
    if(key1 == NULL) {
        sdbh_barre (dbh, dbh->head_info->bof, 0);
        return 1;
    }
    if(key2 != NULL) {
        memcpy ((void *)(dbh->key), (void *)key1, dbh->head_info->n_limit);

        for(i = 0; i < dbh->head_info->n_limit; i++)
            if(key1[i] != key2[i])
                break;
        /* que pedo con ERASED?? (load returns errorvalue on ERASED */
        pointer = dbh_load (dbh);
        if((pointer == 0) && !(ERASED))
            return 0;
/*  if ((pointer=dbh_load(dbh)) == 0) return 0;*/
        /* aqui hay que considerar que el nodo puede estar marcado como ERASED */
        else {
            pointer = dbh_find (dbh, i);
            sdbh_barre (dbh, pointer, dbh->head_info->n_limit - i);
        }
    } else if(keylength) {
        memcpy ((void *)(dbh->key), (void *)key1, keylength);
        pointer = dbh_find (dbh, keylength);
        if(pointer == 0)
            return 0;
        sdbh_barre (dbh, pointer, dbh->head_info->n_limit - keylength);
    }
    return 1;
} /***************************************************************************************/

static int
sdbh_newreversebarre (DBHashTable * dbh, unsigned char *key1, unsigned char *key2, unsigned char keylength) {
    int i;
    FILE_POINTER pointer;
    if(!dbh) {
        ERR( "sdbh_newreversebarre(): %s\n", strerror (EBADF));

        return 0;
    }

    dbh->head_info->dbh_exit = 0;
    if(key1 == NULL) {
        sdbh_reversebarre (dbh, dbh->head_info->bof, 0);
        return 1;
    }
    if(key2 != NULL) {
        memcpy ((void *)(dbh->key), (void *)key1, dbh->head_info->n_limit);
        for(i = 0; i < dbh->head_info->n_limit; i++)
            if(key1[i] != key2[i])
                break;
        /* que pedo con ERASED?? (load returns errorvalue on ERASED */
        pointer = dbh_load (dbh);
        if((pointer == 0) && !(ERASED))
            return 0;
/*  if ((pointer=dbh_load(dbh)) == 0) return 0;*/
        else {
            pointer = dbh_find (dbh, i);
            sdbh_reversebarre (dbh, pointer, i);
        }
    } else if(keylength != 0) {
        memcpy ((void *)dbh->key, (void *)key1, keylength);

        pointer = dbh_find (dbh, keylength);
        if(pointer == 0)
            return 0;
        sdbh_reversebarre (dbh, pointer, keylength);
    }
    return 1;
}  /************************************************************************************/

static DBHashTable *
sdbh_regen (DBHashTable * dbh, int sweep) {
    void *temp, *newtemp;
    char *archivo2, *archivo;
    DBHashTable *newdbh;
    unsigned char *tempkey, *newtempkey;
    void (*fun) (DBHashTable *);

    if(!dbh) {
        ERR( "sdbh_regen(): %s\n", strerror (EBADF));
        return NULL;
    }
    if (dbh->protection_flags & DBH_READ_ONLY){
        ERR( "DBH table is read only: %s\n", dbh->path);
        return NULL;
    }
    dbh_lock_write (dbh);

    archivo2 = sdbh_randomfilename (dbh, 's');
    if (!archivo2){
        ERR( "Cannot open temporary file: DBHashtable will not be regenerated\n");
	return NULL;
    }

    archivo = (char *)malloc (strlen (dbh->path) + 1);
    if (archivo == NULL) {
	 ERR( "malloc(%ld): %s\n", (long)(strlen (dbh->path) + 1), strerror(errno));
	free(archivo2);
        return NULL;
    } 
    strcpy (archivo, dbh->path);
    struct stat  st_old;
    // No problem with TOCTOU, since at TOU, rename will return error
    // if synchronization mismatch occurs.
    // coverity[fs_check_call : FALSE]
    if (stat (archivo, &st_old) < 0){
        ERR( "unable to stat: %s\n", archivo);
	free(archivo);
	free(archivo2);
        return NULL;
    }

    char *archivobak = (char *)malloc (strlen (archivo) + strlen (".bak") + 1);
    if (archivobak == NULL) {
	free(archivo);
	free(archivo2);
	 ERR( "malloc archivobak: %s\n", strerror(errno));
        return NULL;
    }     
    sprintf (archivobak, "%s.bak", archivo);

    // Set thread safe characteristics to new table.
    int flags = dbh->protection_flags & (DBH_PARALLEL_SAFE|DBH_THREAD_SAFE);
    DBHashTable *dbh_desdbh = sdbh_create (archivo2, dbh->head_info->n_limit, flags);
    if(!dbh_desdbh) {
        ERR( "Cannot open for write: %s\n", archivo2);
	free(archivo);
	free(archivo2);
	free(archivobak);
        return NULL;
    }
    
    struct stat  st_new;
    // No problem with TOCTOU, since at TOU, rename will return error
    // if synchronization mismatch occurs.
    // coverity[fs_check_call : FALSE]
    if (stat (archivo2, &st_new) < 0){
        ERR( "unable to stat: %s\n", archivo2);
	free(archivo);
	free(archivo2);
	free(archivobak);
        dbh_close(dbh_desdbh);
        return NULL;
    }
    dbh_lock_write (dbh_desdbh);

    
    dbh->dbh_desdbh = dbh_desdbh;

    tempkey = dbh_desdbh->key;
    newtempkey = dbh_desdbh->newkey;
    temp = dbh_desdbh->data;
    newtemp = dbh_desdbh->newdata;
    dbh_desdbh->key = dbh->key;
    dbh_desdbh->newkey = dbh->newkey;
    dbh_desdbh->data = dbh->data;
    dbh_desdbh->newdata = dbh->newdata;

    

    dbh_desdbh->head_info->record_length = dbh->head_info->record_length;
    fun = dbh->operate;

    dbh->operate = sdbh_transfer;
    if(sweep) {
        sdbh_newbarre (dbh, NULL, NULL, 0);
    } else {
        sdbh_newreversebarre (dbh, NULL, NULL, 0);
    }

    dbh_desdbh->data = temp;
    dbh_desdbh->newdata = newtemp;
    dbh_desdbh->key = tempkey;
    dbh_desdbh->newkey = newtempkey;

    dbh_unlock_write (dbh_desdbh);
    dbh_close (dbh_desdbh);
    dbh_unlock_write (dbh);
    dbh_close (dbh);


  
    
    if(unlink (archivobak) < 0 && errno != ENOENT) {
            ERR( "cannot remove old backup file: %s (%s)\n",
                            archivobak, strerror(errno));
    }

    DBG ("rename %s %s\n", archivo, archivobak);
    if(rename (archivo, archivobak) < 0) {
            ERR( "*** DBH: cannot rename %s to %s (%s)\n",
                            archivo, archivobak, strerror (errno));            ;
    }
    free (archivobak);

    if(st_old.st_dev == st_new.st_dev) {
        DBG (" rename %s %s\n", archivo2, archivo);
        if(rename (archivo2, archivo) < 0) {
            ERR( "*** DBH: cannot rename %s to %s (%s)\n",
                            archivo2, archivo, strerror (errno));
        }
    } else {
        DBG (" mv %s %s\n", archivo2, archivo);
#ifdef HAVE_FORK
        pid_t pid = fork ();
        if(!pid) {
            execlp ("mv", "mv", archivo2, archivo, NULL);
            _exit (123);
        } else {
            int status;
            waitpid (pid, &status, 0);
        }
#else
# ifdef HAVE__SPAWNVP
	const int mode = P_WAIT; // also P_NOWAIT
	const char *_argv[]={"copy", archivo2, archivo, NULL};
	const char *argv0 = _argv[0];
	_spawnvp (mode, argv0, _argv);
# else
        ERR( "Cannot regenerate %s\n", archivo);
        free (archivo);
        free (archivo2);
        return NULL;
# endif
#endif
    }

    newdbh = dbh_new (archivo, NULL, flags);
    free (archivo);
    free (archivo2);
    newdbh->operate = fun;
    return newdbh;
}

static void
prune_mark_erased (DBHashTable * dbh) {
    SET_ERASED;
    /* set file pointer at record flag byte */
    if(place_fp_at (dbh, (off_t)((dbh->reservedB) + 1)) < 0) {
        return;
    }
    /* write the flag to the file */
    /* printf("flag for erase=0x%x\n",dbh->flag); */
    if(write (dbh->fd, &(dbh->flag), 1) != 1) {
        ERR( "unable to erase: %s\n", strerror (errno));
        return;
    }
    /*printf("erasing %s flag at %lu\n",(char *)dbh->data,
       (long unsigned)(dbh->reservedB+1)); */
    /* update file header information */
    dbh->head_info->data_space -= dbh->bytes_userdata;
    dbh->head_info->erased_space += (dbh->bytes_userdata);
    return;
}

static void
prune_mark_unerased (DBHashTable * dbh) {
    SET_UNERASED;
    /* set file pointer at record flag byte */
    if(place_fp_at (dbh, (off_t)(dbh->reservedB + 1)) < 0){
        return;
    }
    /* write the flag to the file */
    if(write (dbh->fd, &(dbh->flag), 1) != 1) {
        ERR( "unable to unerase: %s\n", strerror (errno));
        return;
    }
    /* update file header information */
    dbh->head_info->data_space += dbh->bytes_userdata;
    dbh->head_info->erased_space -= (dbh->bytes_userdata);
    return;
}


static int sdbh_writeheader (DBHashTable * dbh, int flush) {
        TRACE("sdbh_writeheader\n");
    if(!dbh || !(dbh->head_info)) {
        ERR("sdbh_writeheader(): %s\n", strerror (EBADF));
        return 0;
    }
    if(!(dbh->head_info->writeOK)){
	ERR("sdbh_writeheader() is invalid in a read only DBH Table\n");
	return 0;
    }   

    if(strlen (COPYRIGHT) > 127) {
        ERR("This should never happen, strlen(\"%s\")>127\n", COPYRIGHT);
        return 0;
    }
    if (!strlen(dbh->head_info->copyright)){
            strcpy ((char *)(dbh->head_info->copyright), COPYRIGHT);
    }

#ifdef TURN
    dbh->head_info->bof = sdbh_turnaround (dbh->head_info->bof);
    dbh->head_info->record_length = sdbh_turnaround (dbh->head_info->record_length);
    dbh->head_info->total_space = sdbh_turnaround (dbh->head_info->total_space);
    dbh->head_info->data_space = sdbh_turnaround (dbh->head_info->data_space);
    dbh->head_info->erased_space = sdbh_turnaround (dbh->head_info->erased_space);
    dbh->head_info->records = sdbh_turnaround (dbh->head_info->records);
    dbh->head_info->fractalidad = sdbh_turnaround (dbh->head_info->fractalidad);
#endif


    dbh_lock_write (dbh);
    if(!place_fp_at (dbh, (off_t)0)){
        dbh_unlock_write (dbh);
        ERR("*** sdbh_writeheader(): error 1; cannot place at 0L\n");
        return 0;
    }


    size_t bytes = write (dbh->fd, (void *)dbh->head_info, sizeof (dbh_header_t));
    if(bytes != sizeof (dbh_header_t)) {
        dbh_unlock_write (dbh);
        ERR("*** sdbh_writeheader(): error 2; write count %lu != %lu\n",
                        (long unsigned)sizeof (dbh_header_t), 
                        (long unsigned)bytes);
        return 0;
    }
    dbh_unlock_write (dbh);


#ifdef TURN
    dbh->head_info->bof = sdbh_turnaround (dbh->head_info->bof);
    dbh->head_info->record_length = sdbh_turnaround (dbh->head_info->record_length);
    dbh->head_info->total_space = sdbh_turnaround (dbh->head_info->total_space);
    dbh->head_info->data_space = sdbh_turnaround (dbh->head_info->data_space);
    dbh->head_info->erased_space = sdbh_turnaround (dbh->head_info->erased_space);
    dbh->head_info->records = sdbh_turnaround (dbh->head_info->records);
    dbh->head_info->fractalidad = sdbh_turnaround (dbh->head_info->fractalidad);
#endif
    if (!flush) return 1;
  
#if HAVE_FSYNC
    fsync(dbh->fd);
#endif
#if HAVE_WINDOWS_H
    FlushFileBuffers ((HANDLE) _get_osfhandle (dbh->fd));
#endif
    
#ifdef DEBUG_TRACE
    dbh_header_t tmp;
    lseek(dbh->fd, 0, SEEK_SET);
    DBG("--- > testing written header... got %ld bytes\n",
                    (long)read(dbh->fd, (void *)(&tmp), sizeof(dbh_header_t)));
#endif
    return 1;
}

#ifdef PARALLEL_SAFE
static int
lock_it (DBHashTable * dbh, int for_write){
    TRACE("DBH: locking file %s (write = %d)\n", dbh->path, for_write);
    if (for_write) {
        dbh->lock_p->write_lock_count++;
        dbh->lock_p->write_lock = getpid();
    } else {
        dbh->lock_p->read_lock_count++;
    }
    if(msync (dbh->lock_p,  sizeof(dbh_lock_t), MS_ASYNC) < 0){
            ERR("(2:lock_p=%p) Cannot msync shared memory item for %s: %s\n", 
                dbh->lock_p, dbh->path, strerror (errno));
    }
    return 1;
}

static int
lock_ok (DBHashTable * dbh, int for_write){
    int lock_condition;
    if (for_write){
        lock_condition = (dbh->lock_p->read_lock_count ||
	    (dbh->lock_p->write_lock_count && dbh->lock_p->write_lock != getpid()));
    } else {
        lock_condition = dbh->lock_p->write_lock_count;
    }
    if (lock_condition) return 0;
    return 1;
}

static int
sdbh_lock (DBHashTable * dbh, int for_write){
    if (!(dbh->protection_flags & DBH_PARALLEL_SAFE)){
        return 0;
    }
    if (dbh->lock_attempt_limit == 0) sem_wait(dbh->sem);
    else { 
        struct timespec timeout;
        timeout.tv_sec = time(NULL) + dbh->lock_attempt_limit;
        timeout.tv_nsec = 0;
        if (sem_timedwait(dbh->sem, &timeout) < 0){
            ERR("DBH: sdbh_lock() unable to unlock semaphore for %s (%s), proceeding on timeout...\n", dbh->path, strerror(errno));
        }
    }
    if (lock_ok(dbh, for_write)){
        lock_it(dbh, for_write);
        sem_post(dbh->sem);
        return 1;
    }
    sem_post(dbh->sem);
    return 0;
}

static int
sdbh_unlock (DBHashTable * dbh, int for_write){
    if (!(dbh->protection_flags & DBH_PARALLEL_SAFE)){
        return 0;
    }
    if (dbh->lock_attempt_limit == 0) sem_wait(dbh->sem);
    else { 
        struct timespec timeout;
        timeout.tv_sec = time(NULL) + dbh->lock_attempt_limit;
        timeout.tv_nsec = 0;
        if (sem_timedwait(dbh->sem, &timeout) < 0){
            ERR("DBH: sdbh_unlock() unable to unlock semaphore for %s (%s), proceeding on timeout...\n", dbh->path, strerror(errno));
        }
    }
    if (for_write){
        dbh->lock_p->write_lock_count = 0;
        dbh->lock_p->write_lock = 0;
    } else {
        if (dbh->lock_p->read_lock_count > 0) dbh->lock_p->read_lock_count--;
    }
    if(msync (dbh->lock_p,  sizeof(dbh_lock_t), MS_ASYNC) < 0){
            ERR("(3:lock_p=%p) Cannot msync shared memory item for %s: %s\n", 
                dbh->lock_p, dbh->path, strerror (errno));
    }
    sem_post(dbh->sem);
    return 1;
}
#endif

