/*
 * This file contains the private interface of libbubblemon.
 * Libbubblemon is a library for making bubbling load monitors for
 * different platforms.  See for example
 * "http://www.nongnu.org/bubblemon/".
 *
 *  Copyright (C) 1999-2000 Johan Walles - d92-jwa@nada.kth.se
 *  http://www.nongnu.org/libbubblemon/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#ifndef BUBBLEMON_PRIVATE_H
#define BUBBLEMON_PRIVATE_H

#include "bubblemon.h"

// Physics constants
static const int MSECS_PER_PHYSICS_FRAME = 20;
static const int MINIMUM_FRAMERATE = 8; // As long as you render at
					// least this many frames per
					// second, the animation will
					// be smooth.
// How fast do the bubbles rise?
static const float GRAVITY = -0.02;
// The drag coefficient of the bottle
static const float FLOATER_DRAG = 0.1;
// How fast do the water levels accelerate?
static const float VOLATILITY = 1.0;
// 0.0 means the liquid never moves.
// 1.0 means the liquid will continue to oscillate forever.
static const float VISCOSITY = 0.99;
// How fast are the water levels allowed to move?
static const float SPEED_LIMIT = 1.0;
// How much newly created bubbles make the surface ripple
static const float RIPPLES = 0.2;
// How fast background bubbles move relative to the foreground
// bubbles.
static const float BGBUBBLE_SPEED = 0.4;
// Time in the simulated universe moves this much slower than real
// time
static const float TIME_SLOWDOWN = 30.0;
// At maximum bubbling, how many bubbles are created per second per
// pixel?  The number of pixels used is the width of the bottom.
static const float MAX_BUBBLE_INTENSITY = 0.5;

// Weed constants
static const int WEED_HEIGHT = 40;  // The weeds grow to 40% of the
				    // image height
// How fast the weeds grow in pixels/sec
static const int WEED_MAXSPEED = 10;
static const int WEED_MINSPEED = 1;
static const int WEED_SPEEDFACTOR = 1;
// How often we move to the next weed (in milliseconds)
static const int WEED_INTERVAL = 500;


// Color settings                           rrggbbaa
static const unsigned int MAXAIRCOLOR   = 0xff000040;
static const unsigned int MINAIRCOLOR   = 0x2299ff00;
static const unsigned int MAXWATERCOLOR = 0xaa000080;
static const unsigned int MINWATERCOLOR = 0x0055ff80;
// How many seconds does it take for the color to change from min
// intensity to max intensity?
static const float COLOR_CHANGE_TIME = 0.8;

// Bottom growth has random colors between these two
static const unsigned int GROWTHCOLOR1  = 0x00ff0080;
static const unsigned int GROWTHCOLOR2  = 0xffff40ff;

// This is the number of levels of antialiasing available.  0 is pure
// air, ANTIALIASING - 1 is pure water.
typedef signed char antialias_t;
static const antialias_t ANTIALIASING = 5;

// Floater behaviour
typedef enum
{
  GONE,
  FALLING,  // The floater is falling until it is entirely visible
  FLOATING, // The floater floats at the surface
  SINKING   // The floater is sinking out of sight
} floaterState_t;

// Bubble layers.  There are two layers of water, and a bubble can
// belong to either the front or the back layer.
typedef enum { BACKGROUND, FOREGROUND } layerType_t;

// A bubble
typedef struct
{
  int x;
  float y;
  float dy;

  layerType_t layer;
} bubble_t;

// A water level
typedef struct
{
  float y;
  float dy;
} waterLevel_t;

// A weed
typedef struct
{
  float height;
  float nourishment;

  bubblemon_color_t color;
} weed_t;

// For keeping track of how many milliseconds ago we last rendered a
// frame
typedef long int timestamp_t;

// Bubblemon state
typedef struct
{
  // Stuff to visualize
  int targetWaterlevel;
  float colorIntensity;
  int bubblingIntensity;
  int growthIntensity;
  int floaterVisible;
  
  // Settings
  bubblemon_color_t maxWaterColor;
  bubblemon_color_t minWaterColor;
  bubblemon_color_t maxAirColor;
  bubblemon_color_t minAirColor;
  bubblemon_color_t growthColor1;
  bubblemon_color_t growthColor2;
  
  bubblemon_picture_t floaterImage;
  int floaterXCenter;
  int floaterYCenter;
  
  // Internal state
  timestamp_t lastUpdate;
  int width;
  int height;
  
  float colorIntensityGoal;
  
  // This is a width * height big array used as an intermediate result
  // while rendering air, water and bubbles
  antialias_t *antialiasingMap;
  
  waterLevel_t *waterLevels;
  
  bubble_t *bubbles;
  int nBubbles;
  int maxBubbles;
  float createNNewBubbles;
  layerType_t lastNewBubbleLayer;
  
  weed_t *weeds;
  int msecsToNextWeedUpdate;
  int lastUpdatedWeed;
  
  floaterState_t floaterState;
  float floaterY;
  float floaterDy;
  
} bubblemon_private_t;

#endif // !defined(BUBBLEMON_PRIVATE_H)
