/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "sql.h"

static void
sql_node_discharge(
  struct sql_node*const                 io_node);

extern void
sql_assign(
  struct sql*const                      o_sql)
{

  memset(o_sql, 0, sizeof(*o_sql));

  return;
}

extern void
sql_discharge(
  struct sql*const                      io_sql)
{
  struct sql_node*                      l_node;
  struct sql_node*                      l_next;

  l_node= (*io_sql).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_next= (*l_node).m_next;

    sql_node_discharge(l_node);

    g_free(l_node);
    l_node= l_next;

  }while(1);

  memset(io_sql, 0, sizeof(*io_sql));

  return;
}

static void
sql_node_segment_discharge(
  struct sql_segment*const              io_segment)
{

  if ((*io_segment).m_text)
  {
    g_free((*io_segment).m_text);
  }

  memset(io_segment, 0, sizeof(*io_segment));

  return;
}

static void
sql_node_input_discharge(
  struct sql_input*const                io_input)
{

  if ((*io_input).m_prompt)
  {
    g_free((*io_input).m_prompt);
  }

  if ((*io_input).m_answer)
  {
    g_free((*io_input).m_answer);
  }

  memset(io_input, 0, sizeof(*io_input));

  return;
}

static void
sql_node_parm_discharge(
  struct sql_parm*const                 io_parm)
{

  if ((*io_parm).m_answer)
  {
    g_free((*io_parm).m_answer);
  }

  memset(io_parm, 0, sizeof(*io_parm));

  return;
}

static void
sql_node_discharge(
  struct sql_node*const                 io_node)
{

  switch((*io_node).m_type)
  {
    case sql_type_segment:
      sql_node_segment_discharge(&(*io_node).m_object.m_segment);
      break;
    case sql_type_input:
      sql_node_input_discharge(&(*io_node).m_object.m_input);
      break;
    case sql_type_parm:
      sql_node_parm_discharge(&(*io_node).m_object.m_parm);
      break;
    default:
      break;
  }

  memset(io_node, 0, sizeof(*io_node));

  return;
}

extern void
sql_node_new(
  struct sql_node**                     o_node,
  struct sql*const                      io_sql)
{
  struct sql_node*                      l_node;

  l_node= (struct sql_node*)g_malloc0(sizeof(*l_node));
  (*o_node)= l_node;

  do
  {

    if (0 == (*io_sql).m_head)
    {
      (*io_sql).m_head= l_node; 
      (*io_sql).m_tail= l_node;
      break;
    }

    (*io_sql).m_tail->m_next= l_node;
    (*l_node).m_prev= (*io_sql).m_tail;
    (*io_sql).m_tail= l_node;

  }while(0);

  return;
}

static void
sql_segment_new(
  struct sql*const                      io_sql,
  char const*const                      i_head,
  char const*const                      i_tail)
{
  struct sql_node*                      l_node;
  ptrdiff_t                             l_size;

  sql_node_new(&l_node, io_sql);
  (*l_node).m_type= sql_type_segment;
  l_size= (i_tail - i_head);
  (*l_node).m_object.m_segment.m_text= g_malloc0(1+l_size);
  (*l_node).m_object.m_segment.m_size= l_size;
  memcpy((*l_node).m_object.m_segment.m_text,i_head,l_size);

  return;
}

static void
sql_input_new(
  struct sql*const                      io_sql,
  char const*const                      i_head,
  char const*const                      i_tail)
{
  struct sql_node*                      l_node;
  ptrdiff_t                             l_size;

  sql_node_new(&l_node, io_sql);
  (*l_node).m_type= sql_type_input;
  l_size= (i_tail - i_head);
  (*l_node).m_object.m_input.m_prompt= (char*)g_malloc0(1+l_size);
  memcpy((*l_node).m_object.m_input.m_prompt, i_head, l_size);

  return;
}

static void
sql_parm_new(
  struct sql*const                      io_sql,
  char const*const                      i_head,
  char const*const                      i_tail,
  char const*const                      i_break)
{
  struct sql_node*                      l_node;
  ptrdiff_t                             l_size;

  sql_node_new(&l_node, io_sql);
  (*l_node).m_type= sql_type_parm;

  if (i_break)
  {
    l_size= (i_break - i_head);
    (*l_node).m_object.m_parm.m_tag= g_malloc0(1+l_size);
    memcpy((*l_node).m_object.m_parm.m_tag, i_head, l_size);

    l_size= (i_tail - (i_break+1));
    (*l_node).m_object.m_parm.m_field= g_malloc0(1+l_size);
    memcpy((*l_node).m_object.m_parm.m_field, i_break+1, l_size);
  }
  else
  {
    l_size= (i_tail - i_head);
    (*l_node).m_object.m_parm.m_field= g_malloc0(1+l_size);
    memcpy((*l_node).m_object.m_parm.m_field, i_head, l_size);
  }

  return;
}

/*
 * ':'['?']|[table'.']field
 */
extern void
sql_parse(
  struct sql*const                      io_sql,
  struct query_node const*const         i_node)
{
  char*                                 l_head;
  char*                                 l_tail;
  char*                                 l_break;
  int                                   l_input;

  l_head= (*i_node).m_text;
  l_tail= (*i_node).m_text;

  g_strlcpy((*io_sql).m_tag, (*i_node).m_tag, sizeof((*io_sql).m_tag));

  do
  {

    if (0 == l_head)
    {
      break;
    }

    do
    {

      if (0 == l_tail[0])
      {
        sql_segment_new(io_sql, l_head, l_tail);
        break;
      }

      l_input= 0;

      do
      {

        if (':' != l_tail[0])
        {
          break;
        }

        sql_segment_new(io_sql, l_head, l_tail);
        l_tail++;

        if ('?' == l_tail[0])
        {
          l_input= 1;
          l_tail++;
        }

        l_head= l_tail;
        l_break= 0;

        do
        {

          if ((0 == l_tail[0]) || 
            (' ' == l_tail[0]) || 
            ('"' == l_tail[0]) || 
            ('\'' == l_tail[0]) || 
            (',' == l_tail[0]) || 
            ('\n' == l_tail[0]) || 
            ('\t' == l_tail[0]))
          {
            if (l_input)
            {
              sql_input_new(io_sql, l_head, l_tail);
            }
            else
            {
              sql_parm_new(io_sql, l_head, l_tail, l_break);
            }
            l_head= l_tail;
            break;
          }

          if (('.' == l_tail[0]) && (0 == l_break))
          {
            l_break= l_tail;
          }

          l_tail++;

        }while(1);

      }while(0);

      l_tail++;

    }while(1);

  }while(0);

  return;
}
