/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "rw.h"

extern void
rw_assign(
  struct rw*                            o_rw)
{

  memset(o_rw, 0, sizeof(*o_rw));
  (*o_rw).m_page_number= 1;
  (*o_rw).m_time= time(0);
  query_exec_assign(&(*o_rw).m_exec);

  return;
}

extern void
rw_attr_lookup_by_name(
  struct iwsql_attr const**             o_attr,
  struct query_exec const*const         i_exec,
  char const*                           i_table,
  char const*                           i_column)
{
  struct query_exec_node const*         l_query;
  char const*                           l_table;

  (*o_attr)= 0;

  do
  {

    if (0 == i_column)
    {
      break;
    }

    if (i_table)
    {
      l_table= i_table;
    }
    else
    {
      l_table= "primary";
    }

    query_exec_lookup(&l_query, l_table, i_exec);

    if (0 == l_query)
    {
      break;
    }

    query_exec_node_lookup(o_attr, i_column, l_query);

  }while(0);

  return;
}

extern int
rw_cairo_pdf_surface_create(
  GError**                              o_error,
  struct rw*                            io_rw,
  double const                          i_page_width,
  double const                          i_page_height,
  char const*                           i_outfile)
{
  GError*                               l_error;
  int                                   l_exit;
  cairo_status_t                        l_status;

  l_error= 0;
  l_exit= 0;

  do
  {

    (*io_rw).m_surface= cairo_pdf_surface_create(
      i_outfile,
      i_page_width,
      i_page_height);

    l_status= cairo_surface_status((*io_rw).m_surface);

    if (CAIRO_STATUS_SUCCESS != l_status)
    {
      l_error= g_error_new(
        RW, 
        RW_CREATE_SURFACE,
        "Unable to create cairo surface: file='%s' status(%d)",
        i_outfile,
        l_status);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    (*io_rw).m_cr= cairo_create((*io_rw).m_surface);

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
rw_discharge(
  struct rw*                            io_rw)
{

  query_exec_discharge(&(*io_rw).m_exec);

  if ((*io_rw).m_cr)
  {
    cairo_destroy((*io_rw).m_cr);
  }

  if ((*io_rw).m_surface)
  {
    cairo_surface_destroy((*io_rw).m_surface);
  }

  memset(io_rw, 0, sizeof(*io_rw));

  return;
}

static void
rw_fbarcode_set_value(
  struct item_fbarcode*                 io_object,
  struct query_exec*const               io_exec)
{
  struct iwsql_attr const*              l_attr;

  do
  {

    if ((*io_object).m_value)
    {
      g_free((*io_object).m_value);
      (*io_object).m_value= 0;
      (*io_object).m_value_size= 0;
    }

    rw_attr_lookup_by_name(
      &l_attr,
      io_exec,
      (*io_object).m_table.m_query,
      (*io_object).m_table.m_column);

    if (0 == l_attr)
    {
      break;
    }

    (*io_object).m_value= g_malloc0((*l_attr).m_size);
    memcpy((*io_object).m_value, (*l_attr).m_value, (*l_attr).m_size);
    (*io_object).m_value_size= (*l_attr).m_size;

  }while(0);

  return;
}

static void
rw_sum_set_value(
  struct item_fsum*const                io_object,
  struct query_exec*const               io_exec)
{
  struct iwsql_attr const*              l_attr;
  decNumber                             l_value;

  memset(&l_value, 0, sizeof(l_value));

  do
  {

    rw_attr_lookup_by_name(
      &l_attr,
      io_exec,
      (*io_object).m_table.m_query,
      (*io_object).m_table.m_column);

    if (0 == l_attr)
    {
      break;
    }

    decNumberFromString(&l_value, (*l_attr).m_value, &(*io_object).m_ctx);
    decNumberAdd(
      &(*io_object).m_accumulator, 
      &(*io_object).m_accumulator, 
      &l_value,
      &(*io_object).m_ctx);

  }while(0);

  return;
}

static void
rw_fimage_set_value(
  struct item_fimage*                   io_object,
  struct query_exec*const               io_exec)
{
  struct iwsql_attr const*              l_attr;

  do
  {

    if ((*io_object).m_value)
    {
      g_free((*io_object).m_value);
      (*io_object).m_value= 0;
      (*io_object).m_value_size= 0;
    }

    rw_attr_lookup_by_name(
      &l_attr,
      io_exec,
      (*io_object).m_table.m_query,
      (*io_object).m_table.m_column);

    if (0 == l_attr)
    {
      break;
    }

    (*io_object).m_value= g_malloc0((*l_attr).m_size);
    memcpy((*io_object).m_value, (*l_attr).m_value, (*l_attr).m_size);
    (*io_object).m_value_size= (*l_attr).m_size;

  }while(0);

  return;
}

static void
rw_fnumber_set_value(
  struct item_fnumber*                  io_object,
  struct query_exec*const               io_exec)
{
  struct iwsql_attr const*              l_attr;

  do
  {

    if ((*io_object).m_value)
    {
      g_free((*io_object).m_value);
      (*io_object).m_value= 0;
    }

    rw_attr_lookup_by_name(
      &l_attr,
      io_exec,
      (*io_object).m_table.m_query,
      (*io_object).m_table.m_column);

    if (0 == l_attr)
    {
      break;
    }

    (*io_object).m_value= g_strdup((*l_attr).m_value);

  }while(0);

  return;
}

static void
rw_ftext_set_value(
  struct item_ftext*                    io_object,
  struct query_exec*const               io_exec)
{
  struct iwsql_attr const*              l_attr;

  do
  {

    if ((*io_object).m_value)
    {
      g_free((*io_object).m_value);
      (*io_object).m_value= 0;
    }

    rw_attr_lookup_by_name(
      &l_attr,
      io_exec,
      (*io_object).m_table.m_query,
      (*io_object).m_table.m_column);

    if (0 == l_attr)
    {
      break;
    }

    (*io_object).m_value= g_strdup((*l_attr).m_value);

  }while(0);

  return;
}

/*
 * ':{'[table'.']column'}'
 * escape '::{' results in ':{'
 */
static void
rw_text_set_value(
  struct item_text*                     io_text,
  struct query_exec*const               io_exec)
{
  struct iwsql_attr const*              l_attr;
  gchar*                                l_buf;
  gchar*                                l_column;
  gchar*                                l_ptr;
  GString*                              l_lookup;
  gchar*                                l_table;
  gchar*                                l_tail;
  GString*                              l_value;

  l_value= g_string_new(0);

  do
  {

    if ((*io_text).m_value)
    {
      g_free((*io_text).m_value);
      (*io_text).m_value= 0;
    }

    l_tail= (*io_text).m_text;

    if (0 == l_tail)
    {
      break;
    }

    do
    {

      if (0 == l_tail[0])
      {
        break;
      }

      if (':' != l_tail[0])
      {
rw_text_fill_store:
        g_string_append_c(l_value, l_tail[0]);
        l_tail++;
        continue;
      }

      l_tail++;

      if ((':' == l_tail[0]) && ('{' == l_tail[1]))
      {
        goto rw_text_fill_store;
      }

      if ('{' != l_tail[0])
      {
        g_string_append_c(l_value, ':');
        goto rw_text_fill_store;
      }

      l_tail++;
      l_lookup= g_string_new(0);

      do  
      {

        if (0 == l_tail[0])
        {
          break;
        }

        if ('}' == l_tail[0])
        {
          break;
        }

        g_string_append_c(l_lookup, l_tail[0]);
        l_tail++;

      }while(1);

      l_buf= g_string_free(l_lookup, 0);
      l_ptr= strchr(l_buf, '.');

      if (0 == l_ptr)
      {
        l_table= "primary";
        l_column= l_buf;
      } 
      else
      {
        (*l_ptr)= 0;
        l_table= l_buf;
        l_column= l_ptr+1;
      }

      rw_attr_lookup_by_name(&l_attr, io_exec, l_table, l_column);
      g_free(l_buf);

      if (l_attr && (*l_attr).m_value)
      {
        g_string_append(l_value, (*l_attr).m_value);
      }

      l_tail++;

    }while(1);

  }while(0);

  (*io_text).m_value= g_string_free(l_value, 0);

  return;
}

static void
rw_date_set_value(
  struct item_date*const                io_date,
  struct rw*const                       io_rw)
{

  (*io_date).m_time= (*io_rw).m_time;

  return;
}

static void
rw_page_number_set_value(
  struct item_page_number*const         io_page_number,
  struct rw*const                       io_rw)
{

  (*io_page_number).m_page_number= (*io_rw).m_page_number;

  return;
}

extern void
rw_item_clear_accumulator(
  struct item*const                     io_item,
  char const*                           i_group)
{
  struct item_fsum*                     l_fsum;
  struct item_node*                     l_node;
  int                                   l_rc;

  do
  {

    if (0 == io_item || 0 == (*io_item).m_head)
    {
      break;
    }

    l_node= (*io_item).m_head;

    do
    {

      if (0 == l_node)
      {
        break;
      }

      do
      {

        if (item_type_fsum != (*l_node).m_type)
        {
          break;
        }

        l_fsum= &(*l_node).m_object.m_fsum;

        if (i_group)
        {
          if (0 == (*l_fsum).m_group)
          {
            break;
          }

          l_rc= strcmp((*l_fsum).m_group, i_group);

          if (l_rc)
          {
            break;
          }
        }

        memset(&(*l_fsum).m_accumulator, 0, sizeof((*l_fsum).m_accumulator));
        decNumberFromString(&(*l_fsum).m_accumulator, "0.0", &(*l_fsum).m_ctx);

      }while(0);

      l_node= (*l_node).m_next;

    }while(1);

  }while(0);

  return;
}

extern void
rw_item_set_value(
  struct item*const                     io_item,
  struct rw*const                       io_rw)
{
  struct item_node*                     l_node;

  do
  {

    if (0 == io_item || 0 == (*io_item).m_head)
    {
      break;
    }

    l_node= (*io_item).m_head;

    do
    {

      if (0 == l_node)
      {
        break;
      }

      switch((*l_node).m_type)
      {
        case item_type_fbarcode:
          rw_fbarcode_set_value(&(*l_node).m_object.m_fbarcode, &(*io_rw).m_exec);
          break;
        case item_type_fsum:
          rw_sum_set_value(&(*l_node).m_object.m_fsum, &(*io_rw).m_exec);
          break;
        case item_type_fimage:
          rw_fimage_set_value(&(*l_node).m_object.m_fimage, &(*io_rw).m_exec);
          break;
        case item_type_ftext:
          rw_ftext_set_value(&(*l_node).m_object.m_ftext, &(*io_rw).m_exec);
          break;
        case item_type_fnumber:
          rw_fnumber_set_value(&(*l_node).m_object.m_fnumber, &(*io_rw).m_exec);
          break;
        case item_type_text:
          rw_text_set_value(&(*l_node).m_object.m_text, &(*io_rw).m_exec);
          break;
        case item_type_page_number:
          rw_page_number_set_value(&(*l_node).m_object.m_page_number, io_rw);
          break;
        case item_type_date:
          rw_date_set_value(&(*l_node).m_object.m_date, io_rw);
          break;
        default:
          break;
      }

      l_node= (*l_node).m_next;

    }while(1);

  }while(0);

  return;
}

extern void
rw_layout_clear_accumulator(
  struct layout_spec*const              io_layout)
{
  unsigned                              l_slot;

  rw_item_clear_accumulator((*io_layout).m_cover.m_header.m_item, 0);
  rw_item_clear_accumulator((*io_layout).m_cover.m_footer.m_item, 0);
  rw_item_clear_accumulator((*io_layout).m_report.m_header.m_item, 0);
  rw_item_clear_accumulator((*io_layout).m_report.m_detail.m_item, 0);
  rw_item_clear_accumulator((*io_layout).m_report.m_footer.m_item, 0);

  for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
  {
    g_free((*io_layout).m_group[l_slot].m_break.m_value);
    (*io_layout).m_group[l_slot].m_break.m_value= 0;
    rw_item_clear_accumulator((*io_layout).m_group[l_slot].m_header.m_item, 0);
    rw_item_clear_accumulator((*io_layout).m_group[l_slot].m_footer.m_item, 0);
  }

  return;
}

extern void
rw_level_break_set_value(
  struct layout_spec*const              io_layout,
  struct rw *const                      i_rw)
{
  struct iwsql_attr const*              l_attr;
  struct layout_spec_division_group*    l_group;
  unsigned                              l_slot;

  l_attr= 0;

  if ((*io_layout).m_page_break.m_value)
  {
    g_free((*io_layout).m_page_break.m_value);
    (*io_layout).m_page_break.m_value= 0;
  }

  if ((*io_layout).m_page_break.m_column && (*io_layout).m_page_break.m_column[0])
  {
    rw_attr_lookup_by_name(
      &l_attr,
      &(*i_rw).m_exec,
      "primary",
      (*io_layout).m_page_break.m_column);

    if (l_attr)
    {
      (*io_layout).m_page_break.m_value= g_strdup((*l_attr).m_value);
    }
  }

  for(
    l_group= &(*io_layout).m_group[0], 
    l_slot= 0; 
    (*io_layout).m_group_slots > l_slot; 
    l_slot++,
    l_group++)
  {

    l_attr= 0;
    rw_attr_lookup_by_name(
      &l_attr,
      &(*i_rw).m_exec,
      "primary",
      (*l_group).m_break.m_column);

    if ((*l_group).m_break.m_value)
    {
      g_free((*l_group).m_break.m_value);
      (*l_group).m_break.m_value= 0;
    }

    if (l_attr)
    {
      (*l_group).m_break.m_value= g_strdup((*l_attr).m_value);
    }

  }

  return;
}


extern void
rw_layout_group_set_value(
  struct layout_spec*const              io_layout,
  struct rw*const                       io_rw)
{
  unsigned                              l_slot;
  struct layout_spec_division_group*    l_group;

  l_group= &(*io_layout).m_group[0];

  for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
  {
    rw_item_set_value((*l_group).m_header.m_item, io_rw);
    rw_item_set_value((*l_group).m_footer.m_item, io_rw);
    l_group++;
  }

  return;
}

extern void
rw_layout_set_fence(
  struct rw*const                       io_rw,
  struct layout_spec*const              io_layout)
{

  (*io_rw).m_fence_bottom= 
    ((*io_layout).m_paper.m_height - (*io_layout).m_paper.m_margin_bottom);

  do
  {
    if (0 == (*io_layout).m_report.m_footer.m_item ||
      0 == (*io_layout).m_report.m_footer.m_item->m_head)
    {
      break;
    }

    (*io_rw).m_fence_bottom= 
      (*io_layout).m_paper.m_height - 
      (*io_layout).m_report.m_footer.m_paper.m_height;

  }while(0);

  return;
}


