/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "ruler.h"
#include <gdk/gdk.h>
#include <pango/pangocairo.h>

#include <string.h>
#include <stdio.h>

#define FONT_SCALE_MIN 8.0
#define FONT_SCALE_MAX 24.0

enum
{
  SIGNAL_MARGIN_CHANGED,
  SIGNAL_LAST
};

static guint g_ruler_signals[SIGNAL_LAST];

enum
{
  PROP_0,
  PROP_ORIENTATION,
  PROP_PAPER_LENGTH,
  PROP_MARGIN_LEFT,
  PROP_MARGIN_RIGHT,
  PROP_SCALE
};

enum event_type
{
  event_none= 0,
  event_margin_left=1,
  event_margin_right=2,
  event_margin_top=1,
  event_margin_bottom=2
};

union margin
{
  struct
  {
    gdouble                           m_left;
    gdouble                           m_right;
  }                                   m_h;
  struct
  {
    gdouble                           m_top;
    gdouble                           m_bottom;
  }                                   m_v;
};

struct event
{
  enum event_type                       m_type;
  GdkCursorType                         m_cursor_type;
  int                                   m_dragging;
  gdouble                               m_pos_x;
  gdouble                               m_pos_y;
}                                       m_event;

struct _RulerPrivate
{
  GdkWindow*				                    m_event_window;
  cairo_surface_t*                      m_ruler_surface;
  GtkOrientation                        m_orientation;
  gdouble                               m_scale;
  gdouble                               m_paper_length;
  gdouble                               m_adjustment;
  union margin                          m_margin;
  struct event                          m_event;
};

G_DEFINE_TYPE_WITH_PRIVATE(Ruler, ruler, GTK_TYPE_WIDGET)

static void
ruler_finalize(
  GObject*                              i_object)
{

  G_OBJECT_CLASS(ruler_parent_class)->finalize(i_object);

  return;
}

static void
ruler_create_surface(
  GtkWidget*                            i_widget)
{
  RulerPrivate*                         l_priv; 
  GtkAllocation                         l_alloc;

  l_priv= RULER(i_widget)->m_priv;

  if ((*l_priv).m_ruler_surface)
  {
    cairo_surface_destroy((*l_priv).m_ruler_surface);
  }

  gtk_widget_get_allocation (i_widget, &l_alloc);

  (*l_priv).m_ruler_surface= gdk_window_create_similar_surface(
    gtk_widget_get_window(i_widget),
    CAIRO_CONTENT_COLOR,
    l_alloc.width,
    l_alloc.height);

  return;
}

static void
ruler_realize(
  GtkWidget*                            i_widget)
{
  RulerPrivate*                         l_priv; 
  GdkWindow*                            l_win;
  GdkWindowAttr                         l_attr;
  gint                                  l_attr_mask;
  GtkAllocation                         l_alloc;

  l_priv= RULER(i_widget)->m_priv;
  memset(&l_attr, 0, sizeof(l_attr));
  memset(&l_alloc, 0, sizeof(l_alloc));

  GTK_WIDGET_CLASS(ruler_parent_class)->realize(i_widget);

  gtk_widget_get_allocation(i_widget, &l_alloc);

  l_attr.x= l_alloc.x;
  l_attr.y= l_alloc.y;
  l_attr.width= l_alloc.width;
  l_attr.height= l_alloc.height;
  l_attr.wclass= GDK_INPUT_ONLY;
  l_attr.window_type= GDK_WINDOW_CHILD;
  l_attr_mask= (GDK_WA_X | GDK_WA_Y);
  l_attr.event_mask= gtk_widget_get_events(i_widget);
  l_attr.event_mask|= 
    (GDK_POINTER_MOTION_MASK |
    GDK_BUTTON_MOTION_MASK |
    GDK_BUTTON_PRESS_MASK |
    GDK_BUTTON_RELEASE_MASK);

  l_win= gtk_widget_get_window(i_widget),
  (*l_priv).m_event_window= gdk_window_new(l_win, &l_attr, l_attr_mask);
  gtk_widget_register_window(i_widget, (*l_priv).m_event_window);

  ruler_create_surface(i_widget);

  return;
}

static void
ruler_unrealize(
  GtkWidget*                            i_widget)
{
  RulerPrivate *                       l_priv; 

  do
  {

    l_priv= RULER(i_widget)->m_priv;

    if ((*l_priv).m_ruler_surface)
    {
      cairo_surface_destroy((*l_priv).m_ruler_surface);
      (*l_priv).m_ruler_surface= 0;
    }

    if (0 ==(*l_priv).m_event_window)
    {
      break;
    }

    gtk_widget_unregister_window(i_widget, (*l_priv).m_event_window);
    gdk_window_destroy((*l_priv).m_event_window);
    (*l_priv).m_event_window= 0;

  }while(0);

  GTK_WIDGET_CLASS(ruler_parent_class)->unrealize(i_widget);

  return;
}

static void
ruler_destroy(
  GtkWidget*                            i_widget)
{

  GTK_WIDGET_CLASS(ruler_parent_class)->destroy(i_widget);

  return;
}

extern GtkWidget *
ruler_new(
  GtkOrientation const                  i_orientation)
{
  GObject*                              l_object;

  l_object= g_object_new(
    TYPE_RULER,
    "orientation", i_orientation,
    0);

  return GTK_WIDGET(l_object);
}

static gboolean
ruler_draw_vertical(
  GtkWidget*                            i_widget,
  cairo_t*                              io_cr)
{
  GtkAllocation                         l_alloc;
  gboolean                              l_exit;
  PangoFontDescription*                 l_font;
  int                                   l_font_height;
  double                                l_font_scale;
  double                                l_height;
  PangoLayout*                          l_layout;
  unsigned                              l_number;
  double                                l_number_pos_x;
  PangoContext*                         l_pango_context;
  double                                l_pos_y;
  RulerPrivate*                         l_priv; 
  char                                  l_text[16];
  double                                l_width;
  char const*                           l_ptr;
  int                                   l_digits;
  double                                l_char_y;

  l_exit= 0;
  l_priv= RULER(i_widget)->m_priv;

  gtk_widget_get_allocation(i_widget, &l_alloc);

  l_width= l_alloc.width;
  l_height= (*l_priv).m_scale * (72.0*(*l_priv).m_paper_length);
  l_height+= (*l_priv).m_adjustment;

  /* fill entire space */
  cairo_set_source_rgb(io_cr, 1.0, 1.0, 1.0);
  cairo_rectangle(io_cr, 0, 0, l_alloc.width, l_height);
  cairo_fill(io_cr);

  /* fill top margin */
  cairo_set_source_rgb(io_cr, 0.780, 0.780, 0.780);
  l_pos_y= (*l_priv).m_scale * (72.0*(*l_priv).m_margin.m_v.m_top);
  l_pos_y+= (*l_priv).m_adjustment;
  cairo_rectangle(
    io_cr,
    0,
    0,
    l_alloc.width,
    l_pos_y);
    cairo_fill(io_cr);

  /* fill bottom margin */
  cairo_set_source_rgb(io_cr, 0.780, 0.780, 0.780);
  l_pos_y= (*l_priv).m_scale * 
    (72.0*((*l_priv).m_paper_length - (*l_priv).m_margin.m_v.m_bottom));
  l_pos_y+= (*l_priv).m_adjustment;
  cairo_rectangle(
    io_cr,
    0,    
    l_pos_y,
    l_alloc.width,
    l_height-l_pos_y);
  cairo_fill(io_cr);

  /* fill remainder */
  cairo_set_source_rgb(io_cr, 0.920, 0.920, 0.920);
  cairo_rectangle(
    io_cr,
    0, 
    l_height,
    l_alloc.width,
    l_alloc.height);
  cairo_fill(io_cr);

  l_layout= pango_cairo_create_layout(io_cr);

  l_pango_context= pango_layout_get_context(l_layout);
  pango_context_set_gravity_hint(l_pango_context,PANGO_GRAVITY_HINT_STRONG);
  pango_context_set_base_gravity(l_pango_context, PANGO_GRAVITY_EAST);

  l_font= pango_font_description_new();
  pango_font_description_set_family(l_font, "Serif");
  l_font_scale=  FONT_SCALE_MIN;
  pango_font_description_set_size(l_font, (PANGO_SCALE*l_font_scale));
  pango_layout_set_font_description(l_layout, l_font);
  pango_font_description_free(l_font);

  pango_layout_context_changed(l_layout);
  pango_cairo_update_context(io_cr,l_pango_context);

  l_number_pos_x= (l_width/2) - (l_font_scale/2);

  l_number= 0;

  cairo_set_source_rgb(io_cr, 0.590, 0.590, 0.590);

  /* 1" */
  for(l_pos_y= (*l_priv).m_adjustment;
    l_height >= l_pos_y;
    l_pos_y += (72.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, 0.0, l_pos_y);
    cairo_line_to(io_cr, l_width, l_pos_y);
    cairo_stroke(io_cr);
    g_snprintf(l_text, sizeof(l_text), "%d", l_number);
    l_digits= strlen(l_text);
    l_ptr= &l_text[l_digits-1];
    l_char_y= l_pos_y;
    while(l_digits)
    {
      cairo_move_to(io_cr, l_number_pos_x, l_pos_y);
      pango_layout_set_text(l_layout, l_ptr, 1);
      pango_cairo_show_layout(io_cr, l_layout);
#if 0
      pango_layout_get_pixel_size(l_layout, &l_font_height, 0);
      l_pos_y+= l_font_height;
#else
      l_pos_y+= FONT_SCALE_MIN;
#endif
      l_digits--;
      l_ptr--;
    }
    l_number++;
  }

  g_object_unref(l_layout);
  
  /* 1/2" */
  for(l_pos_y= (*l_priv).m_adjustment;
    l_height > l_pos_y;
    l_pos_y += (36.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, l_width - (l_width/2), l_pos_y);
    cairo_line_to(io_cr, l_width, l_pos_y);
    cairo_stroke(io_cr);
  }

  /* 1/4" */
  for(l_pos_y= (*l_priv).m_adjustment;
    l_height > l_pos_y;
    l_pos_y += (18.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, l_width - (l_width/4), l_pos_y);
    cairo_line_to(io_cr, l_width, l_pos_y);
    cairo_stroke(io_cr);
  }

  /* 1/8" */
  for(l_pos_y= (*l_priv).m_adjustment;
    l_width > l_pos_y;
    l_pos_y += (9.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, l_width - (l_width/8), l_pos_y);
    cairo_line_to(io_cr, l_width, l_pos_y);
    cairo_stroke(io_cr);
  }

  return l_exit;
}

static gboolean
ruler_draw_horizontal(
  GtkWidget*                            i_widget,
  cairo_t*                              io_cr)
{
  GtkAllocation                         l_alloc;
  gboolean                              l_exit;
  PangoFontDescription*                 l_font;
  double                                l_font_scale;
  double                                l_height;
  PangoLayout*                          l_layout;
  unsigned                              l_number;
  double                                l_number_pos_y;
  double                                l_pos_x;
  RulerPrivate*                         l_priv; 
  char                                  l_text[16];
  double                                l_width;

  l_exit= 0;
  l_priv= RULER(i_widget)->m_priv;

  gtk_widget_get_allocation(i_widget, &l_alloc);

  l_width= (*l_priv).m_scale*(72.0*(*l_priv).m_paper_length);
  l_width+= (*l_priv).m_adjustment;
  l_height= l_alloc.height;

  /* fill entire space */
  cairo_set_source_rgb(io_cr, 1.0, 1.0, 1.0);
  cairo_rectangle(io_cr, 0, 0, l_width, l_alloc.height);
  cairo_fill(io_cr);

  /* fill left margin */
  cairo_set_source_rgb(io_cr, 0.780, 0.780, 0.780);
  l_pos_x= (*l_priv).m_scale * (72.0*(*l_priv).m_margin.m_h.m_left);
  l_pos_x+= (*l_priv).m_adjustment;
  cairo_rectangle(
    io_cr,
    0,
    0,
    l_pos_x, 
    l_alloc.height);
    cairo_fill(io_cr);

  /* fill right margin */
  cairo_set_source_rgb(io_cr, 0.780, 0.780, 0.780);
  l_pos_x= (*l_priv).m_scale *
    (72.0*((*l_priv).m_paper_length - (*l_priv).m_margin.m_h.m_right));
  l_pos_x+= (*l_priv).m_adjustment;
  cairo_rectangle(
    io_cr,
    l_pos_x,    
    0,
    l_width-l_pos_x,
    l_alloc.height);
  cairo_fill(io_cr);

  /* fill remainder */
  cairo_set_source_rgb(io_cr, 0.920, 0.920, 0.920);
  cairo_rectangle(
    io_cr,
    l_width,    
    0,
    l_alloc.width,
    l_alloc.height);
  cairo_fill(io_cr);

  l_layout= pango_cairo_create_layout(io_cr);
  l_font= pango_font_description_new();
  pango_font_description_set_family(l_font, "Serif");
  l_font_scale=  FONT_SCALE_MIN;
  pango_font_description_set_size(l_font, (PANGO_SCALE*l_font_scale));
  pango_layout_set_font_description(l_layout, l_font);
  pango_font_description_free(l_font);

  l_number_pos_y= (l_height/2) - (l_font_scale/2);

  l_number= 0;

  cairo_set_source_rgb(io_cr, 0.590, 0.590, 0.590);
  /* 1" */
  for(l_pos_x= (*l_priv).m_adjustment;
    l_width >= l_pos_x;
    l_pos_x += (72.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, l_pos_x, 0.0);
    cairo_line_to(io_cr, l_pos_x, l_height);
    cairo_stroke(io_cr);
    g_snprintf(l_text, sizeof(l_text), "%d", l_number);
    cairo_move_to(io_cr, l_pos_x, l_number_pos_y);
    pango_layout_set_text(l_layout, l_text, -1);
    pango_cairo_show_layout(io_cr, l_layout);
    l_number++;
  }

  g_object_unref(l_layout);
  
  /* 1/2" */
  for(l_pos_x= (*l_priv).m_adjustment;
    l_width > l_pos_x;
    l_pos_x += (36.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, l_pos_x, l_height - (l_height/2));
    cairo_line_to(io_cr, l_pos_x, l_height);
    cairo_stroke(io_cr);
  }

  /* 1/4" */
  for(l_pos_x= (*l_priv).m_adjustment;
    l_width > l_pos_x;
    l_pos_x += (18.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, l_pos_x, l_height - (l_height/4));
    cairo_line_to(io_cr, l_pos_x, l_height);
    cairo_stroke(io_cr);
  }

  /* 1/8" */
  for(l_pos_x= (*l_priv).m_adjustment;
    l_width > l_pos_x;
    l_pos_x += (9.0*(*l_priv).m_scale))
  {
    cairo_move_to(io_cr, l_pos_x, l_height - (l_height/8));
    cairo_line_to(io_cr, l_pos_x, l_height);
    cairo_stroke(io_cr);
  }

  return l_exit;
}

static gboolean
ruler_guide_draw_horizontal(
  GtkWidget*                            i_widget)
{
  RulerPrivate*                         l_priv; 
  cairo_t*                              l_cr;
  GtkAllocation                         l_alloc;
  gdouble                               l_width;
  gdouble                               l_height;
  gdouble                               l_pos_x;
  gdouble                               l_pos_y;
  gdouble                               l_adjustment;

  l_cr= 0;
  l_priv= RULER(i_widget)->m_priv;

  do
  {

    if ((*l_priv).m_event.m_dragging)
    {
      break;
    }

    if ((int)(*l_priv).m_event.m_pos_y > l_alloc.height)
    {
      l_adjustment= (*l_priv).m_adjustment;
    }
    else
    {
      l_adjustment= 0.0;
    }

    l_cr = gdk_cairo_create(gtk_widget_get_window(i_widget));
  
    gtk_widget_get_allocation(i_widget, &l_alloc);

    l_width= (*l_priv).m_scale*(72.0*(*l_priv).m_paper_length);
    l_width+= l_adjustment;

    cairo_rectangle(l_cr, l_alloc.x, l_alloc.y, l_width, l_alloc.height);
    cairo_clip(l_cr);

    cairo_translate(l_cr, l_alloc.x, l_alloc.y);
    cairo_set_source_surface(l_cr, (*l_priv).m_ruler_surface, 0, 0);
    cairo_paint(l_cr);

    l_width= (1 | (l_alloc.height / 2));
    l_height= (l_width / 2);

    l_pos_x= (*l_priv).m_event.m_pos_x;
    l_pos_x+= l_adjustment;
    l_pos_y= l_width;

    cairo_set_source_rgb(l_cr, 0.0, 0.0, 0.0);
    cairo_move_to(l_cr, l_pos_x-l_height, l_pos_y);
    cairo_line_to(l_cr, l_pos_x+l_height, l_pos_y);
    cairo_line_to(l_cr, l_pos_x, l_pos_y+l_width);
    cairo_line_to(l_cr, l_pos_x-l_height, l_pos_y);
    cairo_fill(l_cr);

  }while(0);

  if (l_cr)
  {
    cairo_destroy(l_cr);
  }
  
  return 0;
}

static gboolean
ruler_guide_draw_vertical(
  GtkWidget*                            i_widget)
{
  RulerPrivate*                         l_priv; 
  cairo_t*                              l_cr;
  GtkAllocation                         l_alloc;
  gdouble                               l_width;
  gdouble                               l_height;
  gdouble                               l_pos_x;
  gdouble                               l_pos_y;
  gdouble                               l_adjustment;

  l_cr= 0;
  l_priv= RULER(i_widget)->m_priv;

  do
  {

    if ((*l_priv).m_event.m_dragging)
    {
      break;
    }

    if ((int)(*l_priv).m_event.m_pos_x > l_alloc.width)
    {
      l_adjustment= (*l_priv).m_adjustment;
    }
    else
    {
      l_adjustment= 0.0;
    }

    l_cr = gdk_cairo_create(gtk_widget_get_window(i_widget));
  
    gtk_widget_get_allocation(i_widget, &l_alloc);

    l_height= (*l_priv).m_scale*(72.0*(*l_priv).m_paper_length);
    l_height+= l_adjustment;

    cairo_rectangle(l_cr, l_alloc.x, l_alloc.y, l_alloc.width, l_height);
    cairo_clip(l_cr);

    cairo_translate(l_cr, l_alloc.x, l_alloc.y);
    cairo_set_source_surface(l_cr, (*l_priv).m_ruler_surface, 0, 0);
    cairo_paint(l_cr);

    l_width= (1 | (l_alloc.width / 2));
    l_height= (l_width / 2);

    l_pos_x= l_alloc.width - l_width;
    l_pos_y= (*l_priv).m_event.m_pos_y;
    l_pos_y+= l_adjustment;

    cairo_set_source_rgb(l_cr, 0.0, 0.0, 0.0);
    cairo_move_to(l_cr, l_pos_x, l_pos_y-l_height);
    cairo_line_to(l_cr, l_pos_x, l_pos_y+l_height);
    cairo_line_to(l_cr, l_pos_x+l_width, l_pos_y);
    cairo_line_to(l_cr, l_pos_x, l_pos_y-l_height);
    cairo_fill(l_cr);

  }while(0);

  if (l_cr)
  {
    cairo_destroy(l_cr);
  }
  
  return 0;

  return 0;
}

static gboolean
ruler_draw(
  GtkWidget*                            i_widget,
  cairo_t*                              io_cr)
{
  gboolean                              l_exit;
  RulerPrivate*                         l_priv; 
  cairo_t*                              l_cr;

  l_priv= RULER(i_widget)->m_priv;

  l_cr= cairo_create((*l_priv).m_ruler_surface);

  switch((*l_priv).m_orientation)
  {
    case GTK_ORIENTATION_HORIZONTAL:
      l_exit= ruler_draw_horizontal(i_widget, l_cr);
      break;
    case GTK_ORIENTATION_VERTICAL:
      l_exit= ruler_draw_vertical(i_widget, l_cr);
      break;
    default:
     break; 
  }

  cairo_destroy(l_cr);
  cairo_set_source_surface(io_cr, (*l_priv).m_ruler_surface, 0, 0);
  cairo_paint(io_cr);

  switch((*l_priv).m_orientation)
  {
    case GTK_ORIENTATION_HORIZONTAL:
      l_exit= ruler_guide_draw_horizontal(i_widget);
      break;
    case GTK_ORIENTATION_VERTICAL:
      l_exit= ruler_guide_draw_vertical(i_widget);
      break;
    default:
     break; 
  }

  return l_exit;
}

static void
ruler_map(
  GtkWidget*                            i_widget)
{
  RulerPrivate*                        l_priv; 

  l_priv= RULER(i_widget)->m_priv;

  GTK_WIDGET_CLASS(ruler_parent_class)->map(i_widget);

  gdk_window_show(l_priv->m_event_window);

  return;
}

static void
ruler_unmap(
  GtkWidget*                            i_widget)
{
  RulerPrivate*                        l_priv; 

  l_priv= RULER(i_widget)->m_priv;

  gdk_window_hide(l_priv->m_event_window);

  GTK_WIDGET_CLASS(ruler_parent_class)->unmap(i_widget);

  return;
}

static void
ruler_size_allocate(
  GtkWidget*                            i_widget,
  GtkAllocation*                        i_alloc)
{
  RulerPrivate*                        l_priv; 
  int                                   l_rc;

  l_priv= RULER(i_widget)->m_priv;

  gtk_widget_set_allocation(i_widget, i_alloc);

  do
  {

    l_rc= gtk_widget_get_realized(i_widget);

    if (0 ==l_rc)
    {
      break;
    }

    gdk_window_move_resize (
      (*l_priv).m_event_window,
      (*i_alloc).x, 
      (*i_alloc).y,
      (*i_alloc).width, 
      (*i_alloc).height);

    ruler_create_surface(i_widget);

  }while(0);

  return;
}

static gboolean
ruler_button_press(
  GtkWidget*                            i_widget,
  GdkEventButton*                       i_event)
{
  RulerPrivate*                        l_priv; 
  Ruler*                               l_ruler;

  l_ruler= RULER(i_widget);
  l_priv= (*l_ruler).m_priv;

  do
  {

    (*l_priv).m_event.m_pos_x= (*i_event).x;
    (*l_priv).m_event.m_pos_y= (*i_event).y;

    if (event_none == (*l_priv).m_event.m_type)
    {
      break;
    }

    (*l_priv).m_event.m_dragging= 1;

  }while(0);

  return 1;
}

static gboolean
ruler_button_release(
  GtkWidget*                            i_widget,
  GdkEventButton*                       i_event)
{
  RulerPrivate*                         l_priv; 
  Ruler*                                l_ruler;

  l_ruler= RULER(i_widget);
  l_priv= (*l_ruler).m_priv;

  if ((*l_priv).m_event.m_dragging)
  {
    g_signal_emit(l_ruler, g_ruler_signals[SIGNAL_MARGIN_CHANGED], 0); 
  }

  (*l_priv).m_event.m_pos_x= (*i_event).x;
  (*l_priv).m_event.m_pos_y= (*i_event).y;
  (*l_priv).m_event.m_dragging= 0;

  return 1;
}

static gboolean
ruler_horizontal_margin_test(
  GtkWidget*                            i_widget,
  GdkEventMotion*                       i_event)
{
  GtkAllocation                         l_alloc;
  GdkCursor*                            l_cursor;
  GdkCursorType                         l_cursor_type;
  cairo_bool_t                          l_hit;
  double                                l_margin_left;
  double                                l_margin_right;
  RulerPrivate*                         l_priv; 
  cairo_rectangle_int_t                 l_rect;
  cairo_region_t*                       l_region;
  Ruler*                                l_ruler;

  l_ruler= RULER(i_widget);
  l_priv= (*l_ruler).m_priv;

  (*l_priv).m_event.m_type= event_none;

  gtk_widget_get_allocation(i_widget, &l_alloc);

  l_margin_left= (*l_priv).m_scale * (72.0 * (*l_priv).m_margin.m_h.m_left);
  l_margin_left+= (*l_priv).m_adjustment;
  l_margin_right= (*l_priv).m_scale * (72.0*((*l_priv).m_paper_length - (*l_priv).m_margin.m_h.m_right));
  l_margin_right+= (*l_priv).m_adjustment;

  do
  {

    l_rect.x= (int)(l_margin_left-2.0);
    l_rect.y= 0;
    l_rect.width= 4;
    l_rect.height= l_alloc.height;
    l_region= cairo_region_create_rectangle(&l_rect);
    l_hit= cairo_region_contains_point(l_region, (*i_event).x, (*i_event).y);
    cairo_region_destroy(l_region);

    if (l_hit)
    {
      (*l_priv).m_event.m_type= event_margin_left;
      break;
    }

    l_rect.x= (int)(l_margin_right-2.0);
    l_region= cairo_region_create_rectangle(&l_rect);
    l_hit= cairo_region_contains_point(l_region, (*i_event).x, (*i_event).y);
    cairo_region_destroy(l_region);

    if (l_hit)
    {
      (*l_priv).m_event.m_type= event_margin_right;
      break;
    }

  }while(0);

  do
  {

    if (event_none == (*l_priv).m_event.m_type)
    {
      l_cursor_type= GDK_ARROW;
    }
    else
    {
      l_cursor_type= GDK_SB_H_DOUBLE_ARROW;
    }

    if (l_cursor_type == (*l_priv).m_event.m_cursor_type)
    {
      break;
    }

    l_cursor= gdk_cursor_new_for_display(
      gdk_display_get_default(),
      l_cursor_type);

    gdk_window_set_cursor((*l_priv).m_event_window, l_cursor);
    g_object_unref(l_cursor);

    (*l_priv).m_event.m_cursor_type= l_cursor_type;

  }while(0);

  return 0;
}

static gboolean
ruler_horizontal_motion_notify(
  GtkWidget*                            i_widget,
  GdkEventMotion*                       i_event)
{
  gboolean                              l_exit;
  RulerPrivate*                         l_priv; 
  Ruler*                                l_ruler;
  gdouble                               l_delta_x;

  l_ruler= RULER(i_widget);
  l_priv= (*l_ruler).m_priv;

  l_exit= 0;

  do
  {

    if (0 == (*l_priv).m_event.m_dragging)
    {
      l_exit= ruler_horizontal_margin_test(i_widget, i_event);
      break;
    }

    l_delta_x= ((*i_event).x - (*l_priv).m_event.m_pos_x);

    if (event_margin_left == (*l_priv).m_event.m_type)
    {
      (*l_priv).m_margin.m_h.m_left+= (l_delta_x/72.0);
    }
    else
    {
      (*l_priv).m_margin.m_h.m_right-= (l_delta_x/72.0);
    }

    gtk_widget_queue_draw(i_widget);

  }while(0);

  (*l_priv).m_event.m_pos_x= (*i_event).x;
  (*l_priv).m_event.m_pos_y= (*i_event).y;

  ruler_guide_draw_horizontal(i_widget);

  return l_exit;
}

static gboolean
ruler_vertical_margin_test(
  GtkWidget*                            i_widget,
  GdkEventMotion*                       i_event)
{
  GtkAllocation                         l_alloc;
  GdkCursor*                            l_cursor;
  GdkCursorType                         l_cursor_type;
  cairo_bool_t                          l_hit;
  double                                l_margin_top;
  double                                l_margin_bottom;
  RulerPrivate*                         l_priv; 
  cairo_rectangle_int_t                 l_rect;
  cairo_region_t*                       l_region;
  Ruler*                                l_ruler;

  l_ruler= RULER(i_widget);
  l_priv= (*l_ruler).m_priv;

  (*l_priv).m_event.m_type= event_none;

  gtk_widget_get_allocation(i_widget, &l_alloc);

  l_margin_top= (*l_priv).m_scale * (72.0 * (*l_priv).m_margin.m_v.m_top);
  l_margin_top+= (*l_priv).m_adjustment;
  l_margin_bottom= (*l_priv).m_scale * (72.0*((*l_priv).m_paper_length - (*l_priv).m_margin.m_v.m_bottom));
  l_margin_bottom+= (*l_priv).m_adjustment;

  do
  {

    l_rect.x= 0;
    l_rect.y= (int)(l_margin_top-2.0); 
    l_rect.width= l_alloc.width;
    l_rect.height= 4;
    l_region= cairo_region_create_rectangle(&l_rect);
    l_hit= cairo_region_contains_point(l_region, (*i_event).x, (*i_event).y);
    cairo_region_destroy(l_region);

    if (l_hit)
    {
      (*l_priv).m_event.m_type= event_margin_top;
      break;
    }

    l_rect.y= (int)(l_margin_bottom-2.0);
    l_region= cairo_region_create_rectangle(&l_rect);
    l_hit= cairo_region_contains_point(l_region, (*i_event).x, (*i_event).y);
    cairo_region_destroy(l_region);

    if (l_hit)
    {
      (*l_priv).m_event.m_type= event_margin_bottom;
      break;
    }

  }while(0);

  do
  {

    if (event_none == (*l_priv).m_event.m_type)
    {
      l_cursor_type= GDK_ARROW;
    }
    else
    {
      l_cursor_type= GDK_SB_V_DOUBLE_ARROW;
    }

    if (l_cursor_type == (*l_priv).m_event.m_cursor_type)
    {
      break;
    }

    l_cursor= gdk_cursor_new_for_display(
      gdk_display_get_default(),
      l_cursor_type);

    gdk_window_set_cursor((*l_priv).m_event_window, l_cursor);
    g_object_unref(l_cursor);

    (*l_priv).m_event.m_cursor_type= l_cursor_type;

  }while(0);

  return 0;
}

static gboolean
ruler_vertical_motion_notify(
  GtkWidget*                            i_widget,
  GdkEventMotion*                       i_event)
{
  gboolean                              l_exit;
  RulerPrivate*                         l_priv; 
  Ruler*                                l_ruler;
  gdouble                               l_delta_y;

  l_ruler= RULER(i_widget);
  l_priv= (*l_ruler).m_priv;

  l_exit= 0;

  do
  {

    if (0 == (*l_priv).m_event.m_dragging)
    {
      l_exit= ruler_vertical_margin_test(i_widget, i_event);
      break;
    }

    l_delta_y= ((*i_event).y - (*l_priv).m_event.m_pos_y);

    if (event_margin_top == (*l_priv).m_event.m_type)
    {
      (*l_priv).m_margin.m_v.m_top+= (l_delta_y/72.0);
    }
    else
    {
      (*l_priv).m_margin.m_v.m_bottom-= (l_delta_y/72.0);
    }

    gtk_widget_queue_draw(i_widget);

  }while(0);

  (*l_priv).m_event.m_pos_x= (*i_event).x;
  (*l_priv).m_event.m_pos_y= (*i_event).y;
  ruler_guide_draw_vertical(i_widget);

  return l_exit;
}

extern gboolean
ruler_motion_notify(
  GtkWidget*                            i_widget,
  GdkEventMotion*                       i_event)
{
  int                                   l_exit;
  Ruler*                                l_ruler;
  RulerPrivate*                         l_priv;

  l_ruler= RULER(i_widget);
  l_priv= (*l_ruler).m_priv;
  
  switch((*l_priv).m_orientation)
  {
    case GTK_ORIENTATION_HORIZONTAL:
      l_exit= ruler_horizontal_motion_notify(i_widget, i_event);
      break;
    case GTK_ORIENTATION_VERTICAL:
      l_exit= ruler_vertical_motion_notify(i_widget, i_event);
      break;
    default:
      l_exit= 0;
      break;
    break;
  }

  return l_exit;
}

static void
ruler_set_property(
  GObject*                              i_object,
  guint                                 i_prop_id,
  const GValue*                         i_value,
  GParamSpec*                           i_pspec)
{
  Ruler*                               l_ruler;
  RulerPrivate*                        l_priv;

  l_ruler= RULER(i_object);
  l_priv= (*l_ruler).m_priv;

  switch (i_prop_id)
  {
    case PROP_ORIENTATION:
      (*l_priv).m_orientation = g_value_get_enum(i_value);
      break;
    case PROP_PAPER_LENGTH:
      (*l_priv).m_paper_length= g_value_get_double(i_value);
      break;
    case PROP_MARGIN_LEFT:
      (*l_priv).m_margin.m_h.m_left= g_value_get_double(i_value);
      break;
    case PROP_MARGIN_RIGHT:
      (*l_priv).m_margin.m_h.m_right= g_value_get_double(i_value);
      break;
    case PROP_SCALE:
      (*l_priv).m_scale= g_value_get_double(i_value);
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(i_object, i_prop_id, i_pspec);
      break;
  }

  return;
}

static void
ruler_get_property(
  GObject*                              i_object,
  guint                                 i_prop_id,
  GValue*                               o_value,
  GParamSpec*                           i_pspec)
{
  Ruler*                               l_ruler;
  RulerPrivate*                        l_priv;

  l_ruler= RULER(i_object);
  l_priv= (*l_ruler).m_priv;

  switch (i_prop_id)
  {
    case PROP_ORIENTATION:
      g_value_set_enum(o_value, (*l_priv).m_orientation);
      break;
    case PROP_PAPER_LENGTH:
      g_value_set_double(o_value, (*l_priv).m_paper_length);
      break;
    case PROP_MARGIN_LEFT:
      g_value_set_double(o_value, (*l_priv).m_margin.m_h.m_left);
      break;
    case PROP_MARGIN_RIGHT:
      g_value_set_double(o_value, (*l_priv).m_margin.m_h.m_right);
      break;
    case PROP_SCALE:
      g_value_set_double(o_value, (*l_priv).m_scale);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(i_object, i_prop_id, i_pspec);
      break;
  }

  return;
}

static void
ruler_install_class_property(
  GObjectClass*                        i_object_class)
{

  g_object_class_install_property(
    i_object_class,
    PROP_ORIENTATION,
    g_param_spec_enum (
    "orientation",
    "Orientation",
    "The orientation of the ruler",
    GTK_TYPE_ORIENTATION,
    GTK_ORIENTATION_HORIZONTAL,
    G_PARAM_READWRITE));

  g_object_class_install_property(
    i_object_class,
    PROP_PAPER_LENGTH,
    g_param_spec_double(
      "paper-length",
      "Paper length",
      "Length of paper (in)",
      0.0,
      G_MAXDOUBLE,
      0.0,
      G_PARAM_READWRITE));  

  g_object_class_install_property(
    i_object_class,
    PROP_MARGIN_LEFT,
    g_param_spec_double(
      "margin-left",
      "Left margin",
      "Left margin of the ruler",
      0.0,
      G_MAXDOUBLE,
      0.0,
      G_PARAM_READWRITE));  

  g_object_class_install_property(
    i_object_class,
    PROP_MARGIN_RIGHT,
    g_param_spec_double(
      "margin-right",
      "Right margin",
      "Right margin of the ruler",
      0.0,
      G_MAXDOUBLE,
      0.0,
      G_PARAM_READWRITE));  

  g_object_class_install_property(
    i_object_class,
    PROP_MARGIN_RIGHT,
    g_param_spec_double(
      "scale",
      "Scale",
      "Ruler scale",
      0.0,
      G_MAXDOUBLE,
      0.0,
      G_PARAM_READWRITE));  

  return;
}

static void
ruler_install_class_signals(
  GObjectClass*                        i_object_class)
{

  g_ruler_signals[SIGNAL_MARGIN_CHANGED]=
    g_signal_new(
      "margin-changed",
      G_OBJECT_CLASS_TYPE (i_object_class),
      G_SIGNAL_RUN_FIRST,
      G_STRUCT_OFFSET (RulerClass, margin_changed),
      NULL,
      NULL,
      g_cclosure_marshal_VOID__VOID,
      G_TYPE_NONE, 
      0);

  return;
}

static void
ruler_get_preferred_width(
  GtkWidget*                            i_widget,
  gint*                                 o_minimum_size,
  gint*                                 o_natural_size)
{
  RulerPrivate*                        l_priv; 

  l_priv= RULER(i_widget)->m_priv;

  switch((*l_priv).m_orientation)
  {
    case GTK_ORIENTATION_HORIZONTAL:
      (*o_natural_size) = (72.0 * (*l_priv).m_paper_length * (*l_priv).m_scale);
      break;
    case GTK_ORIENTATION_VERTICAL:
      (*o_natural_size) = 72 * 0.25;
      break;
    default:
      break;
  }

  (*o_minimum_size)= 1;

  return;
}

static void
ruler_get_preferred_height(
  GtkWidget *                           i_widget,
  gint *                                o_minimum_size,
  gint *                                o_natural_size)
{
  RulerPrivate*                        l_priv; 

  l_priv= RULER(i_widget)->m_priv;

  switch((*l_priv).m_orientation)
  {
    case GTK_ORIENTATION_HORIZONTAL:
      (*o_natural_size) = 72 * 0.25;
      break;
    case GTK_ORIENTATION_VERTICAL:
      (*o_natural_size) = (72.0 * (*l_priv).m_paper_length * (*l_priv).m_scale);
      break;
    default:
      break;
  }

  (*o_minimum_size)= 1;

  return;

  return;
}

static void
ruler_dispose(
  GObject*                              i_object)
{

  G_OBJECT_CLASS(ruler_parent_class)->dispose(i_object);

  return;
}

static void
ruler_class_init(
  RulerClass *                        i_klass)
{
  GObjectClass*                        l_object_class;
  GtkWidgetClass*                      l_widget_class;

  l_object_class= G_OBJECT_CLASS(i_klass);
  (*l_object_class).dispose= ruler_dispose;
  (*l_object_class).set_property= ruler_set_property;
  (*l_object_class).get_property= ruler_get_property;
  (*l_object_class).finalize= ruler_finalize;

  ruler_install_class_property(l_object_class);  
  ruler_install_class_signals(l_object_class);

  l_widget_class= GTK_WIDGET_CLASS(i_klass);

  (*l_widget_class).size_allocate= ruler_size_allocate;
  (*l_widget_class).map= ruler_map;
  (*l_widget_class).unmap= ruler_unmap;
  (*l_widget_class).destroy= ruler_destroy;
  (*l_widget_class).realize= ruler_realize;
  (*l_widget_class).unrealize= ruler_unrealize;
  (*l_widget_class).get_preferred_width= ruler_get_preferred_width;
  (*l_widget_class).get_preferred_height= ruler_get_preferred_height;
  (*l_widget_class).motion_notify_event= ruler_motion_notify;
  (*l_widget_class).button_press_event= ruler_button_press;
  (*l_widget_class).button_release_event= ruler_button_release;
  (*l_widget_class).draw= ruler_draw;

  return;
}

static void
ruler_init(
  Ruler *                               io_ruler)
{
  RulerPrivate*                         l_priv; 

  RULER(io_ruler)->m_priv= ruler_get_instance_private(io_ruler);
  l_priv= RULER(io_ruler)->m_priv;
  memset(l_priv, 0, sizeof(*l_priv));
  (*l_priv).m_orientation= GTK_ORIENTATION_HORIZONTAL;
  (*l_priv).m_paper_length= 8.5;
  (*l_priv).m_margin.m_h.m_left= 0.50;
  (*l_priv).m_margin.m_h.m_right= 0.50;
  (*l_priv).m_scale= 1.0;

  gtk_widget_set_can_focus(GTK_WIDGET(io_ruler), TRUE);
  gtk_widget_set_has_window(GTK_WIDGET(io_ruler), FALSE);

  return;
}

extern void
ruler_get_margin(
  Ruler*const                           i_ruler,
  gdouble*const                         o_left_or_top,
  gdouble*const                         o_right_or_bottom)
{
  RulerPrivate*                         l_priv; 

  l_priv= (*i_ruler).m_priv;

  (*o_left_or_top)= (*l_priv).m_margin.m_h.m_left;
  (*o_right_or_bottom)= (*l_priv).m_margin.m_h.m_right;

  return;
}

extern void
ruler_set_adjustment(
  Ruler*const                           i_ruler,
  gdouble const                         i_adjustment)
{

  (*i_ruler).m_priv->m_adjustment= -1.0 * i_adjustment;

  gtk_widget_queue_draw(GTK_WIDGET(i_ruler));

  return;
}

extern void
ruler_set_margin(
  Ruler*const                           i_ruler,
  gdouble const                         i_left_or_top,
  gdouble const                         i_right_or_bottom)
{
  RulerPrivate*                         l_priv; 

  l_priv= (*i_ruler).m_priv;

  (*l_priv).m_margin.m_h.m_left= i_left_or_top;
  (*l_priv).m_margin.m_h.m_right= i_right_or_bottom;

  gtk_widget_queue_draw(GTK_WIDGET(i_ruler));

  return;
}

extern void
ruler_set_paper_length(
  Ruler*const                           i_ruler,
  gdouble const                         i_length)
{
  RulerPrivate*                         l_priv; 

  l_priv= (*i_ruler).m_priv;

  (*l_priv).m_paper_length= i_length;

  gtk_widget_queue_draw(GTK_WIDGET(i_ruler));

  return;
}

extern void
ruler_set_scale(
  Ruler*const                           i_ruler,
  gdouble const                         i_scale)
{
  RulerPrivate*                         l_priv; 

  l_priv= (*i_ruler).m_priv;

  (*l_priv).m_scale= i_scale;

  gtk_widget_queue_draw(GTK_WIDGET(i_ruler));

  return;
}
