/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "query.h"

extern void
query_assign(
  struct query*                         o_query)
{

  memset(o_query, 0, sizeof(*o_query));

  return;
}

extern void
query_clear(
  struct query*                         io_query)
{
  struct query_node*                    l_node;
  struct query_node*                    l_next;

  l_node= (*io_query).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_next= (*l_node).m_next;

    query_node_discharge(l_node);

    g_free(l_node);
    l_node= l_next;

  }while(1);

  memset(io_query, 0, sizeof(*io_query));

  return;
}

extern void
query_count(
  unsigned*const                        o_count,
  struct query const*const              i_query)
{
  unsigned                              l_count;
  struct query_node*                    l_node;

  l_count= 0;
  l_node= (*i_query).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_count++;
    l_node= (*l_node).m_next;

  }while(1);

  (*o_count)= l_count;

  return;
}

extern void
query_delete(
  struct query*                         io_query,
  struct query_node*                    io_node)
{

  if (0 == (*io_node).m_prev)
  {
    (*io_query).m_head= (*io_node).m_next;
  }
  else
  {
    (*io_node).m_prev->m_next= (*io_node).m_next;
  }

  if (0 == (*io_node).m_next)
  {
    (*io_query).m_tail= (*io_node).m_prev;
  }
  else
  {
    (*io_node).m_next->m_prev= (*io_node).m_prev;
  }

  query_node_discharge(io_node);
  g_free(io_node);

  return;
}

extern void
query_discharge(
  struct query*                         io_query)
{

  query_clear(io_query);

  return;
}

extern void
query_first_new(
  struct query_node**                   o_node,
  struct query*                         io_query)
{
  struct query_node*                    l_node;

  l_node= (struct query_node*)g_malloc0(sizeof(*l_node));
  (*o_node)= l_node;

  do
  {

    if (0 == (*io_query).m_head)
    {
      (*io_query).m_head= l_node; 
      (*io_query).m_tail= l_node;
      break;
    }

    (*io_query).m_head->m_prev= l_node;
    (*l_node).m_next= (*io_query).m_head;
    (*io_query).m_head= l_node;

  }while(0);

  return;
}

extern void
query_insert_after(
  struct query*const                     io_query,
  struct query_node*const                io_node,
  struct query_node*const                io_node_new)
{

  if (0 == (*io_node).m_next)
  {
    (*io_query).m_tail= io_node_new;
  }
  else
  {
    (*io_node).m_next->m_prev= io_node_new;
  }

  (*io_node_new).m_next= (*io_node).m_next;
  (*io_node_new).m_prev= io_node;
  (*io_node).m_next= io_node_new;

  return;
}

extern void
query_insert_before(
  struct query*const                     io_query,
  struct query_node*const                io_node,
  struct query_node*const                io_node_new)
{

  if (0 == (*io_node).m_prev)
  {
    (*io_query).m_head= io_node_new;
  }
  else
  {
    (*io_node).m_prev->m_next= io_node_new;
  }

  (*io_node_new).m_next= io_node;
  (*io_node_new).m_prev= (*io_node).m_prev;
  (*io_node).m_prev= io_node_new;

  return;
}

extern void
query_last_new(
  struct query_node**                   o_node,
  struct query*                         io_query)
{
  struct query_node*                    l_node;

  l_node= (struct query_node*)g_malloc0(sizeof(*l_node));
  (*o_node)= l_node;

  do
  {

    if (0 == (*io_query).m_head)
    {
      (*io_query).m_head= l_node; 
      (*io_query).m_tail= l_node;
      break;
    }

    (*io_query).m_tail->m_next= l_node;
    (*l_node).m_prev= (*io_query).m_tail;
    (*io_query).m_tail= l_node;

  }while(0);

  return;
}

extern void
query_node_assign(
  struct query_node*                    o_node)
{

  memset(o_node, 0, sizeof(*o_node));

  return;
}

extern void
query_node_discharge(
  struct query_node*                    io_node)
{


  if ((*io_node).m_text)
  {
    g_free((*io_node).m_text);
  }

  memset(io_node, 0, sizeof(*io_node));

  return;
}

static int
query_read_post_check(
  GError**                              o_error,
  struct query*const                    io_query)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_rc;

  l_error= 0;
  l_exit= 0;

  do
  {

    if (0 == (*io_query).m_head)
    {
      break;
    }

    l_rc= strcmp("primary", (*io_query).m_head->m_tag);

    if (0 == l_rc)
    {
      break;
    }

    l_error= g_error_new(
      QUERY,
      QUERY_XML_MISSING_PRIMARY,
      "'primary' query missing or out of order");
    _error_log(l_error);
    l_exit= -1;

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static int
query_read_statement(
  GError**                              o_error,
  char const*                           i_tag,
  struct query*const                    io_query,
  struct xml *const                     i_xml)
{
  unsigned                              l_count;
  GError*                               l_error;
  int                                   l_exit;
  GList*                                l_node;
  struct query_node*                    l_query;
  int                                   l_rc;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_count= g_list_length(l_node);

    if (1 < l_count)
    {
      l_error= g_error_new(
        QUERY,
        QUERY_XML_TOO_MANY_STATEMENTS,
        "Too many 'sql' statements");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_xml= (struct xml*)(*l_node).data;
    l_rc= strcmp("statement", (*l_xml).m_tag);

    if (l_rc)
    {
      l_error= g_error_new(
        QUERY,
        QUERY_XML_UNKNOWN_ELEMENT,
        "Unkown element ('%s')\n",
        (*l_xml).m_tag);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    if (l_exit)
    {
      break;
    }

    if (strcmp("primary", i_tag))
    {
      query_last_new(&l_query, io_query);
    }
    else
    {
      query_first_new(&l_query, io_query);
    }

    g_strlcpy((*l_query).m_tag, i_tag, sizeof((*l_query).m_tag));
    (*l_query).m_text= (char*)g_malloc0((*l_xml).m_string->len+1);
    memcpy((*l_query).m_text, (*l_xml).m_string->str, (*l_xml).m_string->len);
    (*l_query).m_size= (*l_xml).m_string->len;

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
query_read(
  GError**                              o_error,
  struct query*const                    io_query,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  GList*                                l_node;
  struct query_node*                    l_query;
  int                                   l_rc;
  char const*                           l_tag;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (struct xml*)(*l_node).data;
    l_rc= strcmp("sql", (*l_xml).m_tag);

    if (l_rc)
    {
      l_error= g_error_new(
        QUERY,
        QUERY_XML_UNKNOWN_ELEMENT,
        "Unkown element ('%s')\n",
        (*l_xml).m_tag);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_tag= xml_lookup_attribute_value(l_xml, "name");

    if (0 == l_tag)
    {
      l_error= g_error_new(
        QUERY,
        QUERY_XML_MISSING_ATTRIBUTE,
        "Missing SQL 'name' attribute");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_query= (*io_query).m_head;

    do
    {

      if (0 == l_query)
      {
        break;
      }

      l_rc= strncmp((*l_query).m_tag, l_tag, sizeof((*l_query).m_tag));

      if (0 == l_rc)
      {
        l_error= g_error_new(
          QUERY,
          QUERY_XML_DUPLICATE,
          "Duplicate SQL query ('%s')",
          l_tag);
        _error_log(l_error);
        l_exit= -1;
        break;
      }

      l_query= (*l_query).m_next;

    }while(0);

    if (l_exit)
    {
      break;
    }

    l_exit= query_read_statement(&l_error, l_tag, io_query, l_xml);

    if (l_exit)
    {
      break;
    }

    l_node= (*l_node).next;

  }while(1);

  if (0 == l_exit)
  {
    l_exit= query_read_post_check(&l_error, io_query);
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
query_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct query const*const              i_query)
{
  GError*                               l_error;
  int                                   l_exit;
  struct query_node const*              l_node;

  l_error= 0;
  l_exit= 0;
  l_node= (*i_query).m_head;

  do
  {

    fprintf(io_fp, "\t<query>\n");

    do
    {

      if (0 == l_node)
      {
        break;
      }

      fprintf(io_fp, "\t\t<sql name=\"%s\">\n", (*l_node).m_tag);
      fprintf(io_fp, "\t\t\t<statement>%s</statement>\n", (*l_node).m_text);
      fprintf(io_fp, "\t\t</sql>\n");

      l_node= (*l_node).m_next;

    }while(1);

    fprintf(io_fp, "\t</query>\n");

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
query_unlink(
  struct query_node*                    io_node,
  struct query*const                    io_query)
{

  if (0 == (*io_node).m_prev)
  {
    (*io_query).m_head= (*io_node).m_next;
  } 
  else
  {
    (*io_node).m_prev->m_next= (*io_node).m_next;
  } 
  
  if (0 == (*io_node).m_next)
  {
    (*io_query).m_tail= (*io_node).m_prev;
  } 
  else
  {
    (*io_node).m_next->m_prev= (*io_node).m_prev;
  }
  
}
