/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_selection.h"
#include "handles.h"

extern int
item_selection_get_bbox(
  struct geom_rect*const                o_bbox,
  cairo_t*                              io_cr,
  double const                          i_scale,
  struct item const*const               i_item)
{
  struct geom_rect                      l_bbox;
  unsigned                              l_count;
  struct item_node*                     l_node;
  double                                l_pos;
  rectvect_t                            l_rect;

  memset(o_bbox, 0, sizeof(*o_bbox));
  memset(l_rect, 0, sizeof(l_rect));
  l_count= 0;

  l_node= (*i_item).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if ((*l_node).m_selected)
    {

      (*(*l_node).m_method.m_get_bbox)(&l_bbox, io_cr, i_scale, &(*l_node).m_object);

      if (l_count)
      {
        if (l_rect[top_left_x] > l_bbox.m_pos_x)
        {
          l_rect[top_left_x]= l_bbox.m_pos_x;
          l_rect[bottom_left_x]= l_bbox.m_pos_x;
        }

        l_pos= l_bbox.m_pos_x + l_bbox.m_len_x;
        if (l_rect[top_right_x] < l_pos)
        {
          l_rect[top_right_x]= l_pos;
          l_rect[bottom_right_x]= l_pos;
        }

        if (l_rect[top_left_y] > l_bbox.m_pos_y)
        {
          l_rect[top_left_y]= l_bbox.m_pos_y;
          l_rect[top_right_y]= l_bbox.m_pos_y;
        }

        l_pos= l_bbox.m_pos_y + l_bbox.m_len_y;
        if (l_rect[bottom_left_y] < l_pos)
        {
          l_rect[bottom_left_y]= l_pos;
          l_rect[bottom_right_y]= l_pos;
        }
      }
      else
      {
        l_rect[top_left_x]= l_bbox.m_pos_x;
        l_rect[top_left_y]= l_bbox.m_pos_y;
        l_rect[top_right_x]= l_bbox.m_pos_x + l_bbox.m_len_x;
        l_rect[top_right_y]= l_rect[top_left_x];
        l_rect[bottom_right_x]= l_rect[top_right_x];
        l_rect[bottom_right_y]= l_bbox.m_pos_y + l_bbox.m_len_y;
        l_rect[bottom_left_x]= l_rect[top_left_x];
        l_rect[bottom_left_y]= l_rect[bottom_right_y];
      }

      l_count++;

    }

    l_node= (*l_node).m_next;
  
  }while(1);

  (*o_bbox).m_pos_x= l_rect[top_left_x];
  (*o_bbox).m_pos_y= l_rect[top_left_y];
  (*o_bbox).m_len_x= l_rect[top_right_x] - l_rect[top_left_x];
  (*o_bbox).m_len_y= l_rect[bottom_left_y] - l_rect[top_left_y];

  if (0 > (int)(*o_bbox).m_len_x)
  {
    (*o_bbox).m_pos_x+= (*o_bbox).m_len_x;
    (*o_bbox).m_len_x= -1.0 * (*o_bbox).m_len_x;
  }

  if (0 > (int)(*o_bbox).m_len_y)
  {
    (*o_bbox).m_pos_y+= (*o_bbox).m_len_y;
    (*o_bbox).m_len_y= -1.0 * (*o_bbox).m_len_y;
  }

  return l_count;
}

static void
item_selection_draw_grips(
  cairo_t *const                        io_cr,
  struct geom_rect const*const          i_box,
  double const                          i_scale,
  enum item_selection_drag_type const   i_drag,
  enum event_type const                 i_type)
{
  enum handle_type                      l_handle;
  int                                   l_len_x;
  int                                   l_len_y;
  GdkPixbuf*                            l_pixbuf;
  int                                   l_pos_x;
  int                                   l_pos_y;

  l_pos_x= (*i_box).m_pos_x;
  l_pos_y= (*i_box).m_pos_y;
  l_len_x= (*i_box).m_len_x;
  l_len_y= (*i_box).m_len_y;

  /* top,left (north west)*/
  l_handle= (event_resize_north_west == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_nw_se[l_handle];
  else
    l_pixbuf= g_pixbuf_rotate_nw[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x-13.0, l_pos_y-13.0);
  cairo_paint(io_cr);

  /* top, middle (north) */
  l_handle= (event_resize_north == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_n_s[l_handle];
  else
    l_pixbuf= g_pixbuf_shear_ns[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x+(l_len_x/2)-6.5, l_pos_y-13.0);
  cairo_paint(io_cr);

  /* top, right (north east) */
  l_handle= (event_resize_north_east == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_ne_sw[l_handle];
  else
    l_pixbuf= g_pixbuf_rotate_ne[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x+l_len_x, l_pos_y-13.0);
  cairo_paint(io_cr);

  /* right side, middle (east) */
  l_handle= (event_resize_east == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_e_w[l_handle];
  else
    l_pixbuf= g_pixbuf_shear_ew[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x+l_len_x, l_pos_y+(l_len_y/2.0)-6.5);
  cairo_paint(io_cr);

  /* bottom, right (south east) */
  l_handle= (event_resize_south_east == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_nw_se[l_handle];
  else
    l_pixbuf= g_pixbuf_rotate_se[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x+l_len_x, l_pos_y+l_len_y);
  cairo_paint(io_cr);

  /* bottom, middle (south) */
  l_handle= (event_resize_south == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_n_s[l_handle];
  else
    l_pixbuf= g_pixbuf_shear_ns[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x+(l_len_x/2)-6.5, l_pos_y+l_len_y);
  cairo_paint(io_cr);

  /* bottom, left (south west) */
  l_handle= (event_resize_south_west == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_ne_sw[l_handle];
  else
    l_pixbuf= g_pixbuf_rotate_sw[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x-13.0, l_pos_y+l_len_y);
  cairo_paint(io_cr);

  /* left side, middle (west) */
  l_handle= (event_resize_west == i_type) ? handle_highlight : handle_normal;

  if (drag_rotate_shear != i_drag)
    l_pixbuf= g_pixbuf_resize_e_w[l_handle];
  else
    l_pixbuf= g_pixbuf_shear_ew[l_handle];

  gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, l_pos_x-13.0, l_pos_y+(l_len_y/2.0)-6.5);
  cairo_paint(io_cr);

  return;
}

static int
item_selection_in_event_grips(
  enum event_type*const                 o_type,
  struct item_in_event const*const      i_ctx,
  struct geom_rect const*const          i_box)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {

    /* right side, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x,
      (*i_box).m_pos_y+((*i_box).m_len_y/2)-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_east;
      break;
    }

    /* left side, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-8.0,
      (*i_box).m_pos_y+((*i_box).m_len_y/2)-4.0,
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_west;
      break;
    }

    /* top,left */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-8.0,
      (*i_box).m_pos_y-8.0,
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north_west;
      break;
    }

    /* top, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+((*i_box).m_len_x/2)-4.0, 
      (*i_box).m_pos_y-8.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north;
      break;
    }

    /* top, right */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x,
      (*i_box).m_pos_y-8.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north_east;
      break;
    }

    /* bottom, right */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x, 
      (*i_box).m_pos_y+(*i_box).m_len_y, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south_east;
      break;
    }

    /* bottom, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+((*i_box).m_len_x/2)-4.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south;
      break;
    }

    /* bottom, left */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-8.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y,
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south_west;
      break;
    }

  }while(0);

  return l_hit;
}

extern void
item_selection_draw(
  cairo_t*                              io_cr,
  struct item const*const               i_item,
  double const                          i_scale,
  enum item_selection_drag_type const   i_drag,
  int const                             i_is_dragging,
  enum event_type const                 i_type)
{
  struct geom_rect                      l_bbox;
  unsigned                              l_count;

  do
  {

    l_count= item_selection_get_bbox(&l_bbox, io_cr, i_scale, i_item);

    if (0 == l_count)
    {
      break;
    }
    
    cairo_new_path(io_cr);
    cairo_save(io_cr);
    cairo_set_line_width(io_cr, 1.0);
    cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0);   
    item_common_cairo_dash_set(io_cr, line_style_dash);
    cairo_rectangle(io_cr, l_bbox.m_pos_x, l_bbox.m_pos_y, l_bbox.m_len_x, l_bbox.m_len_y);
    cairo_stroke(io_cr);
    cairo_restore(io_cr);

    if (0 == i_is_dragging)
    {
      item_selection_draw_grips(io_cr, &l_bbox, i_scale, i_drag, i_type);
    }

  }while(0);

  return;
}

extern int
item_selection_in_event(
  enum event_type *const                o_type,
  cairo_t*                              io_cr,
  struct item const*const               i_item,
  struct geom_point const*const         i_event,
  double const                          i_scale)
{
  struct geom_rect                      l_bbox;
  int                                   l_count;
  struct item_in_event                  l_ctx;
  int                                   l_hit;

  memset(&l_ctx, 0, sizeof(l_ctx));
  (*o_type)= event_none;
  l_hit= 0;

  do
  {

    l_count= item_selection_get_bbox(&l_bbox, io_cr, i_scale, i_item);

    if (0 == l_count)
    {
      break;
    }

    l_ctx.m_event= (*i_event);
    l_ctx.m_scale= 1.0;
    l_hit= item_common_event_inside(
      &l_ctx, 
      l_bbox.m_pos_x - 8.0,
      l_bbox.m_pos_y - 8.0,
      l_bbox.m_len_x + 16.0,
      l_bbox.m_len_y + 16.0);

    if (0 == l_hit)
    {
      break;
    }

    l_hit= item_selection_in_event_grips(o_type, &l_ctx, &l_bbox);

    if (l_hit)
    {
      break;
    }

    l_hit= item_common_event_inside(
      &l_ctx, 
      l_bbox.m_pos_x,
      l_bbox.m_pos_y,
      l_bbox.m_len_x,
      l_bbox.m_len_y);

    if (0 == l_hit)
    {
      break;
    }

    (*o_type)= event_move;

  }while(0);

  return l_hit;
}
