/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_common_box.h"

struct item_box_event
{
  enum event_type                       m_type;
  double                                m_mult_x;
  double                                m_mult_y;
};

extern void
item_common_box_get_bbox(
  struct geom_rect*const                o_bbox,
  cairo_t*                              io_cr,
  struct geom_rect const*const          i_box,
  struct item_transform const*const     i_transform,
  double const                          i_scale)
{
  double                                l_center_x;
  double                                l_center_y;
  cairo_matrix_t                        l_matrix;
  double                                l_shear_x;
  double                                l_shear_y;
  double                                l_x1;
  double                                l_x2;
  double                                l_y1;
  double                                l_y2;

  l_center_x= (*i_box).m_pos_x + ((*i_box).m_len_x / 2.0);
  l_center_y= (*i_box).m_pos_y + ((*i_box).m_len_y / 2.0);
  l_shear_x = tan(geom_deg2rad((*i_transform).m_shear_x));
  l_shear_y = tan(geom_deg2rad((*i_transform).m_shear_y));

  cairo_new_path(io_cr);
  cairo_save(io_cr);
  cairo_scale(io_cr, i_scale, i_scale);
  cairo_set_line_width(io_cr, 1.0);
  cairo_translate(io_cr, l_center_x, l_center_y);
  cairo_rotate(io_cr, geom_deg2rad((*i_transform).m_rotation));
  cairo_translate(io_cr, -l_center_x, -l_center_y);
  cairo_matrix_init(
      &l_matrix,
      1.0, l_shear_y,
      l_shear_x, 1.0,
      (*i_box).m_pos_x, (*i_box).m_pos_y);
  cairo_transform(io_cr, &l_matrix);
  cairo_rectangle(io_cr, 0.0, 0.0, (*i_box).m_len_x, (*i_box).m_len_y);
  cairo_restore(io_cr);

  cairo_stroke_extents(io_cr, &l_x1, &l_y1, &l_x2, &l_y2);
  (*o_bbox).m_pos_x= l_x1;
  (*o_bbox).m_pos_y= l_y1;
  (*o_bbox).m_len_x= l_x2 - l_x1;
  (*o_bbox).m_len_y= l_y2 - l_y1;

  return;
}

extern void
item_common_box_normalize(
  struct geom_rect *const               io_box)
{

  if (0.0 > (*io_box).m_len_x)
  {
    (*io_box).m_pos_x+= (*io_box).m_len_x;
    (*io_box).m_len_x= -1.0 * (*io_box).m_len_x;
  }

  if (0.0 > (*io_box).m_len_y)
  {
    (*io_box).m_pos_y+= (*io_box).m_len_y;
    (*io_box).m_len_y= -1.0 * (*io_box).m_len_y;
  }

  return;
}

static void
item_common_box_resize_delta(
  struct geom_rect *const               io_box,
  enum event_type const                 i_type,
  double const                          i_delta_x,
  double const                          i_delta_y)
{

  switch(i_type)
  {
    case event_resize_east:
      (*io_box).m_len_x+= i_delta_x;
      break;
    case event_resize_north:
      (*io_box).m_pos_y+= i_delta_y;
      (*io_box).m_len_y-= i_delta_y;
      break;
    case event_resize_north_east:
      (*io_box).m_pos_y+= i_delta_y;
      (*io_box).m_len_y-= i_delta_y;
      (*io_box).m_len_x+= i_delta_x;
      break;
    case event_resize_north_west:
      (*io_box).m_pos_y+= i_delta_y;
      (*io_box).m_len_y-= i_delta_y;
      (*io_box).m_pos_x+= i_delta_x;
      (*io_box).m_len_x-= i_delta_x;
      break;
    case event_resize_south:
      (*io_box).m_len_y+= i_delta_y;
      break;
    case event_resize_south_east:
      (*io_box).m_len_y+= i_delta_y;
      (*io_box).m_len_x+= i_delta_x;
      break;
    case event_resize_south_west:
      (*io_box).m_len_y+= i_delta_y;
      (*io_box).m_pos_x+= i_delta_x;
      (*io_box).m_len_x-= i_delta_x;
      break;
    case event_resize_west:
      (*io_box).m_pos_x+= i_delta_x;
      (*io_box).m_len_x-= i_delta_x;
      break;
    default:
      break;
  }

  return;
}

static void
item_common_box_transform_event(
  struct item_box_event*const           o_event,
  enum event_type const                 i_type,
  double const                          i_rotation)
{
  enum event_index
  {
    index_north= 0,
    index_north_east,
    index_east,
    index_south_east,
    index_south,
    index_south_west,
    index_west,
    index_north_west
  };
  static struct item_box_event const    l_axis[8]=
  {
    /*0*/{event_resize_north, 1.0, -1.0},
    /*1*/{event_resize_north_east, -1.0, -1.0},
    /*2*/{event_resize_east, -1.0, 1.0},
    /*3*/{event_resize_south_east, -1.0, 1.0},
    /*4*/{event_resize_south, -1.0, 1.0},
    /*5*/{event_resize_south_west, 1.0, 1.0},
    /*6*/{event_resize_west, 1.0, -1.0},
    /*7*/{event_resize_north_west, 1.0, -1.0}
  };
  unsigned                              l_index;

  l_index= 0;

  if (45.0 > i_rotation)
  {
quad1:
    switch(i_type)
    {
      case event_resize_north:
        l_index= index_north;
        break;
      case event_resize_north_east:
        l_index= index_north_east;
        break;
      case event_resize_east:
        l_index= index_east;
        break;
      case event_resize_south_east:
        l_index= index_south_east;
        break;
      case event_resize_south:
        l_index= index_south;
        break;
      case event_resize_south_west:
        l_index= index_south_west;
        break;
      case event_resize_west:
        l_index= index_west;
        break;
      case event_resize_north_west:
        l_index= index_north_west;
        break;
      default:
        break;
    }
  }
  else if (90.0 > i_rotation)
  {
quad2:
    switch(i_type)
    {
      case event_resize_north:
        l_index= index_east;
        break;
      case event_resize_north_east:
        l_index= index_south_east;
        break;
      case event_resize_east:
        l_index= index_south;
        break;
      case event_resize_south_east:
        l_index= index_south_west;
        break;
      case event_resize_south:
        l_index= index_west;
        break;
      case event_resize_south_west:
        l_index= index_north_west;
        break;
      case event_resize_west:
        l_index= index_north;
        break;
      case event_resize_north_west:
        l_index= index_north_east;
        break;
      default:
        break;
    }
  }
  else if (135.0 > i_rotation)
  {
    goto quad2;
  }
  else if (180.0 > i_rotation)
  {
quad3:
    switch(i_type)
    {
      case event_resize_north:
        l_index= index_south;
        break;
      case event_resize_north_east:
        l_index= index_south_west;
        break;
      case event_resize_east:
        l_index= index_west;
        break;
      case event_resize_south_east:
        l_index= index_north_west;
        break;
      case event_resize_south:
        l_index= index_north;
        break;
      case event_resize_south_west:
        l_index= index_north_east;
        break;
      case event_resize_west:
        l_index= index_east;
        break;
      case event_resize_north_west:
        l_index= index_south_east;
        break;
      default:
        break;
    }
  }
  else if (225.0 > i_rotation)
  {
    goto quad3;
  }
  else if (270.0 > i_rotation)
  {
quad4:
    switch(i_type)
    {
      case event_resize_north:
        l_index= index_west;
        break;
      case event_resize_north_east:
        l_index= index_north_west;
        break;
      case event_resize_east:
        l_index= index_north;
        break;
      case event_resize_south_east:
        l_index= index_north_east;
        break;
      case event_resize_south:
        l_index= index_east;
        break;
      case event_resize_south_west:
        l_index= index_south_east;
        break;
      case event_resize_west:
        l_index= index_south;
        break;
      case event_resize_north_west:
        l_index= index_south_west;
        break;
      default:
        break;
    }
  }
  else if (315.0 > i_rotation)
  {
    goto quad4;
  }
  else
  {
    goto quad1;
  }
  
  (*o_event)= l_axis[l_index];

  return;
}

extern void
item_common_box_resize(
  struct geom_rect *const               io_box,
  struct item_resize_event const*const  i_ctx,
  struct item_transform const*const     i_transform)
{
  double                                l_center_x;
  double                                l_center_y;
  double                                l_delta_x;
  double                                l_delta_y;
  double                                l_diff1_x;
  double                                l_diff1_y;
  double                                l_diff2_x;
  double                                l_diff2_y;
  struct item_box_event                 l_event;
  double                                l_len_x;
  double                                l_len_y;
  cairo_matrix_t                        l_matrix;
  double                                l_new_x;
  double                                l_new_y;

  item_common_box_transform_event(&l_event, (*i_ctx).m_type, (*i_transform).m_rotation); 

  l_len_x= (*io_box).m_len_x;
  l_len_y= (*io_box).m_len_y;

  cairo_matrix_init_rotate(&l_matrix, geom_deg2rad((*i_transform).m_rotation));
  l_delta_x= (*i_ctx).m_delta_x;
  l_delta_y= (*i_ctx).m_delta_y;
  cairo_matrix_transform_point(&l_matrix, &l_delta_x, &l_delta_y);
  item_common_box_resize_delta(io_box, l_event.m_type, l_delta_x, l_delta_y);

  l_delta_x= l_len_x - (*io_box).m_len_x;
  l_delta_y= l_len_y - (*io_box).m_len_y;

  l_center_x= (l_event.m_mult_x * l_len_x) / 2.0;
  l_center_y= (l_event.m_mult_y * l_len_y) / 2.0;
  l_new_x= l_center_x;
  l_new_y= l_center_y;
  cairo_matrix_transform_point(&l_matrix, &l_new_x, &l_new_y);
  l_diff1_x= l_new_x - l_center_x;
  l_diff1_y= l_new_y - l_center_y;

  l_len_x+= l_delta_x;
  l_len_y+= l_delta_y;

  l_center_x= (l_event.m_mult_x * l_len_x) / 2.0;
  l_center_y= (l_event.m_mult_y * l_len_y) / 2.0;
  l_new_x= l_center_x;
  l_new_y= l_center_y;
  cairo_matrix_transform_point(&l_matrix, &l_new_x, &l_new_y);
  l_diff2_x= l_new_x - l_center_x;
  l_diff2_y= l_new_y - l_center_y;

  (*io_box).m_pos_x+= l_diff2_x - l_diff1_x; 
  (*io_box).m_pos_y-= l_diff2_y - l_diff1_y;

  return;
}

/* _mjo deprecated - see item_selection_in_event_grips */
extern int
item_common_box_in_event_grips(
  enum event_type*const                 o_type,
  struct item_in_event const*const      i_ctx,
  struct geom_rect const*const          i_box)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {
    /* top,left */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-4.0,
      (*i_box).m_pos_y-4.0,
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north_west;
      break;
    }

    /* top, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+((*i_box).m_len_x/2)-4.0, 
      (*i_box).m_pos_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north;
      break;
    }

    /* top, right */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x-4.0,
      (*i_box).m_pos_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_north_east;
      break;
    }

    /* bottom, right */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x-4.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south_east;
      break;
    }

    /* bottom, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+((*i_box).m_len_x/2)-4.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south;
      break;
    }

    /* bottom, left */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-4.0, 
      (*i_box).m_pos_y+(*i_box).m_len_y-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_south_west;
      break;
    }

    /* right side, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x+(*i_box).m_len_x-4.0, 
      (*i_box).m_pos_y+((*i_box).m_len_y/2)-4.0, 
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_east;
      break;
    }

    /* left side, middle */
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_box).m_pos_x-4.0,
      (*i_box).m_pos_y+((*i_box).m_len_y/2)-4.0,
      8.0,
      8.0);

    if (l_hit) 
    {
      (*o_type)= event_resize_west;
      break;
    }

  }while(0);

  return l_hit;
}

extern int
item_common_box_in_event(
  cairo_t*                              io_cr,
  struct geom_rect const*const          i_box,
  struct item_transform const*const     i_transform,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;
  double                                l_center_x;
  double                                l_center_y;
  cairo_matrix_t                        l_matrix;
  double                                l_shear_x;
  double                                l_shear_y;

  l_center_x= (*i_box).m_pos_x + ((*i_box).m_len_x / 2.0);
  l_center_y= (*i_box).m_pos_y + ((*i_box).m_len_y / 2.0);
  l_shear_x = tan(geom_deg2rad((*i_transform).m_shear_x));
  l_shear_y = tan(geom_deg2rad((*i_transform).m_shear_y));

  cairo_new_path(io_cr);
  cairo_save(io_cr);
  cairo_set_line_width(io_cr, 1.0);
  cairo_scale(io_cr, (*i_ctx).m_scale, (*i_ctx).m_scale);
  cairo_translate(io_cr, l_center_x, l_center_y);
  cairo_rotate(io_cr, geom_deg2rad((*i_transform).m_rotation));
  cairo_translate(io_cr, -l_center_x, -l_center_y);
  cairo_matrix_init(
      &l_matrix,
      1.0, l_shear_y,
      l_shear_x, 1.0,
      (*i_box).m_pos_x, (*i_box).m_pos_y);
  cairo_transform(io_cr, &l_matrix);
  cairo_rectangle(io_cr, 0.0, 0.0, (*i_box).m_len_x, (*i_box).m_len_y);
  cairo_restore(io_cr);
  l_hit = cairo_in_fill(io_cr, (*i_ctx).m_event.m_pos_x, (*i_ctx).m_event.m_pos_y);

  return l_hit;
}

/* _mjo deprecated; see item_selection_draw_grips */
extern void
item_common_box_draw_grips(
  cairo_t *const                        io_cr,
  struct geom_rect const*const          i_box)
{
  int                                   l_pos_x;
  int                                   l_pos_y;
  int                                   l_len_x;
  int                                   l_len_y;

  l_pos_x= (*i_box).m_pos_x;
  l_pos_y= (*i_box).m_pos_y;
  l_len_x= (*i_box).m_len_x;
  l_len_y= (*i_box).m_len_y;

  item_common_cairo_dash_set(io_cr, line_style_solid);
  cairo_set_line_width(io_cr, 1.0);
  cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0); 
  cairo_rectangle(io_cr, l_pos_x, l_pos_y, l_len_x, l_len_y);
  cairo_stroke(io_cr);

  cairo_set_source_rgb(io_cr, 0.0, 1.0, 0.0);

  /* top,left */
  cairo_rectangle(
    io_cr,
    l_pos_x-4.0,
    l_pos_y-4.0,
    8.0,
    8.0); 
  cairo_fill(io_cr);

  /* top, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x+(l_len_x/2)-4.0,
    l_pos_y-4.0,
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* top, right */
  cairo_rectangle(
    io_cr,
    l_pos_x+l_len_x-4.0,
    l_pos_y-4.0,
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* bottom, right */
  cairo_rectangle(
    io_cr,
    l_pos_x+l_len_x-4.0,
    l_pos_y+l_len_y-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* bottom, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x+(l_len_x/2)-4.0,
    l_pos_y+l_len_y-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* bottom, left */
  cairo_rectangle(
    io_cr,
    l_pos_x-4.0,
    l_pos_y+l_len_y-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* right side, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x+l_len_x-4.0,
    l_pos_y+(l_len_y/2)-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  /* left side, middle */
  cairo_rectangle(
    io_cr,
    l_pos_x-4.0,
    l_pos_y+(l_len_y/2)-4.0, 
    8.0,
    8.0);
  cairo_fill(io_cr);

  return;
}

/* _mjo deprecated - object should draw their own hightlight */
extern void
item_common_box_draw_hightlight(
  cairo_t *const                        io_cr,
  struct geom_rect const*const          i_box,
  double const                          i_width)
{
  double                                l_width;

  if (1.00 > i_width)
  {
    l_width= 1.00;
  }
  else
  {
    l_width= i_width;
  }

  item_common_cairo_dash_set(io_cr, line_style_solid);
  cairo_set_line_width(io_cr, l_width);
  cairo_set_source_rgb(io_cr, 0.50, 0.50, 0.00);
  cairo_rectangle(
    io_cr, 
    (*i_box).m_pos_x,
    (*i_box).m_pos_y,
    (*i_box).m_len_x,
    (*i_box).m_len_y);
  cairo_stroke(io_cr);

  return;
}

extern void
item_common_box_move(
  struct geom_rect*const                io_box,
  struct geom_point const*const         i_event)
{

  (*io_box).m_pos_x+= (*i_event).m_pos_x;
  (*io_box).m_pos_y+= (*i_event).m_pos_y;

  return;
}

extern int
item_common_box_read(
  GError**                              o_error,
  struct geom_rect*const                io_box,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_exit= 0;
  l_error= 0;
  
  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_pos_x:
          (*io_box).m_pos_x= xml_convert_double(l_value);
          break;
        case element_tag_pos_y:
          (*io_box).m_pos_y= xml_convert_double(l_value);
          break;
        case element_tag_len_x:
          (*io_box).m_len_x= xml_convert_double(l_value);
          break;
        case element_tag_len_y:
          (*io_box).m_len_y= xml_convert_double(l_value);
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;

}

extern int
item_common_box_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct geom_rect const*const          i_box)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit= 0;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t\t<rect>\n");
  fprintf(io_fp, "\t\t\t\t\t<pos_x>%s</pos_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_box).m_pos_x));
  fprintf(io_fp, "\t\t\t\t\t<pos_y>%s</pos_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_box).m_pos_y));
  fprintf(io_fp, "\t\t\t\t\t<len_x>%s</len_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_box).m_len_x));
  fprintf(io_fp, "\t\t\t\t\t<len_y>%s</len_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_box).m_len_y));
  fprintf(io_fp, "\t\t\t\t</rect>\n");

  return l_exit;
}
