/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "global_common.h"

char*                                   g_builder_dir= 0;
char*                                   g_database_config_filename= 0;
char*                                   g_driver_config_filename= 0;
GtkWindow*                              g_main_window= 0;
char*                                   g_module_filename= 0;

static int
global_common_builder_dir_new(
  GError**                              o_error,
  char const*                           i_testfile)
{
  char*                                 l_base;
  char const*const*                     l_env;
  GError*                               l_error;
  int                                   l_exit;
  char*                                 l_file;
  gboolean                              l_rc;
  unsigned                              l_slot;

  l_error= 0;
  l_exit= 0;
  l_base= 0;
  l_file= 0;

  do
  {

    l_base= g_path_get_dirname(g_module_filename);
    l_file= g_strdup_printf("%s/%s", l_base, i_testfile);
    l_rc= g_file_test(l_file, G_FILE_TEST_IS_REGULAR);

    if (l_rc)
    {
      g_builder_dir= l_base;
      break;
    }

    g_free(l_base);
    l_base= 0;

    l_env= g_get_system_data_dirs();
    l_slot= 0;

    do
    {

      if (0 == l_env[l_slot])
      {
        l_error= g_error_new(
          GLOBAL,
          GLOBAL_GLADE_NOTFOUND,
          "Unable to find glade files");
        l_exit= -1;
        break;
      }

      l_file= g_strdup_printf("%s/iwrite/ui/%s", l_env[l_slot], i_testfile);

      l_rc= g_file_test(l_file, G_FILE_TEST_IS_REGULAR);

      g_free(l_file);

      if (l_rc)
      {
        g_builder_dir= g_strdup_printf("%s/iwrite/ui", l_env[l_slot]);
        break;
      }

      l_slot++;

    }while(1);

  }while(0);

  if (l_error)
  {
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static int
global_common_database_config_filename_new(
  GError**                              o_error)
{
  gchar const*                          l_base_dir;
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  gboolean                              l_rc;
  unsigned                              l_size;
  gchar*                                l_tail;

  l_error= 0;
  l_exit= 0;

  do
  {

    l_base_dir= g_get_user_config_dir();
    g_mkdir(l_base_dir, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);

    l_size= strlen(l_base_dir);
    g_database_config_filename= (char*)g_malloc0(30+l_size);
    memcpy(g_database_config_filename, l_base_dir, l_size);

    l_tail= &g_database_config_filename[l_size];
    (*l_tail++)= G_DIR_SEPARATOR;
    l_tail= g_stpcpy(l_tail, "iwrite");

    g_mkdir(g_database_config_filename, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
    (*l_tail++)= G_DIR_SEPARATOR;
    memcpy(l_tail, "database.ini", sizeof("database.ini"));

    l_fp= g_fopen(g_database_config_filename, "a");
    if (l_fp) fclose(l_fp);

    l_rc= g_file_test(g_database_config_filename, G_FILE_TEST_EXISTS);

    if (l_rc)
    {
      break;
    }

    l_error= g_error_new(
      GLOBAL, 
      GLOBAL_DATABASE_CONFIG_NOTFOUND,
      "Unable to find database config file ('%s')",
      g_database_config_filename);

    l_exit= -1; 

  }while(0);

  if (l_error)
  {
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static int
global_common_driver_config_filename_new(
  GError**                              o_error)
{
  GError*                               l_error;
  int                                   l_exit;
  gboolean                              l_rc;
  gchar const*const*                    l_env;
  unsigned                              l_slot;
  char*                                 l_file;

  l_env= g_get_system_config_dirs();
  l_error= 0;
  l_exit= 0;
  l_slot= 0;

  do
  {

    if (0 == l_env[l_slot])
    {
      l_error= g_error_new(
        GLOBAL,
        GLOBAL_DRIVER_CONFIG_NOTFOUND,
        "Unable to find driver config 'iwrite_driver.cfg'");
      l_exit= -1;
      break;
    }

    l_file= g_build_path(
      G_DIR_SEPARATOR_S,
      l_env[l_slot],
      "iwrite_driver.cfg",
      (char*)0);

    l_rc= g_file_test(l_file, G_FILE_TEST_IS_REGULAR);

    if (l_rc)
    {
      g_driver_config_filename= l_file;
      break;
    }

    g_free(l_file);

    l_slot++;

  }while(1);

  if (l_error)
  {
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

enum create_type
{
  create_for_console,
  create_for_gui
};

static int
global_common_create(
  GError **                             o_error,
  int                                   i_argc, 
  char *                                i_argv[],
  enum create_type const                i_create_for)
{
  GError*                               l_error;
  int                                   l_exit;

  l_error= 0;

  do
  {

#ifdef WIN32

#else
    g_module_filename=g_strdup(i_argv[0]);
#endif

    if (create_for_gui == i_create_for)
    {
      l_exit= global_common_builder_dir_new(&l_error, "mainwindow.glade");

      if (l_exit)
      {
        break;
      }
    }

    l_exit= global_common_database_config_filename_new(&l_error);

    if (l_exit)
    {
      break;
    }

    l_exit= global_common_driver_config_filename_new(&l_error);

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
global_common_create_for_console(
  GError **                             o_error,
  int                                   i_argc, 
  char *                                i_argv[])
{
  return global_common_create(o_error, i_argc, i_argv, create_for_console);
}

extern int
global_common_create_for_gui(
  GError **                             o_error,
  int                                   i_argc, 
  char *                                i_argv[])
{
  return global_common_create(o_error, i_argc, i_argv, create_for_gui);
}

extern void
global_common_destroy()
{

  g_free(g_module_filename);
  g_free(g_database_config_filename);

  return;
}

extern void
utf8_scrub_kludge(
  char*                                 io_buf)
{
  gboolean                              l_bool;
  gchar*                                l_head;
  gchar const*                          l_end;

  l_head= io_buf;

  do
  {

    l_bool= g_utf8_validate(l_head, -1, &l_end);

    if (l_bool)
    {
      break;
    }

    l_head= (gchar*)l_end;

    (*l_head)= '?';

  }while(1);

  return;
}

extern guint
wrap_gtk_builder_add_from_file(
  GtkBuilder *                          builder,
  const gchar *                         filename,
  GError **                             error)
{
  GError*                               l_error;
  guint                                 l_exit;
  gchar*                                l_src;

  l_error= 0;
  l_exit= 0;

  l_src= g_build_path(G_DIR_SEPARATOR_S, g_builder_dir, filename, (char*)0);
  l_exit= gtk_builder_add_from_file(builder, l_src, &l_error);
  g_free(l_src);

  if (l_error)
  {
    _error_log(l_error);
    g_propagate_error(error, l_error);
    l_exit= -1;
  }

  return l_exit;
}
