/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "export_csv.h"

static int                              g_csv_cancel= 0;

struct run_dialog
{
  GtkProgressBar*                       m_progressbar;
  GtkLabel*                             m_file_label;
  GtkButton*                            m_cancel_button;
  size_t                                m_tuple;
};

static gboolean
csv_dialog_end(
  gpointer                              i_data)
{
  struct run_dialog*                    l_rd;

  l_rd= (struct run_dialog*)i_data;
  gtk_button_set_label((*l_rd).m_cancel_button, "Ok");
  g_csv_cancel= 1;

  return G_SOURCE_REMOVE;
}

static gboolean
csv_dialog_update(
  gpointer                              i_data)
{
  char                                  l_text[1+31];
  struct run_dialog*                    l_rd;

  if (0 == g_csv_cancel)
  {
    l_rd= (struct run_dialog*)i_data;
    gtk_progress_bar_pulse((*l_rd).m_progressbar);
    snprintf(l_text, sizeof(l_text), "%ld", (*l_rd).m_tuple);
    gtk_label_set_text((*l_rd).m_file_label, l_text);
  }

  return G_SOURCE_REMOVE;
}

static int
csv_run_property_bind(
  struct run_dialog*const               io_rd,
  GtkBuilder*const                      i_builder)
{
  int                                   l_exit;

  l_exit= 0;

  do
  {

    (*io_rd).m_progressbar= GTK_PROGRESS_BAR(
      gtk_builder_get_object(i_builder, "progressbar1"));

    (*io_rd).m_file_label= GTK_LABEL(
      gtk_builder_get_object(i_builder, "file_label"));

    (*io_rd).m_cancel_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "cancel_button"));

  }while(0);

  return l_exit;
}

static int
print_detail_cell(
  FILE*const                            io_fp,
  struct iwsql_attr const*const         i_attr)
{
  int                                   l_exit;
  char*                                 l_ptr;
  
  l_exit= 0;
  
  do
  {

    if (0 == (*i_attr).m_value)
    {
      break;
    }

    fprintf(io_fp, "\"");

    for (l_ptr= (*i_attr).m_value; *l_ptr; l_ptr++)
    {
      if ('"' == *l_ptr)
      {
        fprintf(io_fp, "\"\"");
      }
      else
      {
        fprintf(io_fp, "%c", *l_ptr);
      }
    }
    
    fprintf(io_fp, "\"");

  }while(0);

  return l_exit;
}

static int
csv_print_detail(
  FILE*const                            io_fp,
  struct query_exec const*const         i_exec)
{
  int                                   l_exit;
  unsigned                              l_slot_sql;
  unsigned                              l_slot_attr;
  struct query_exec_node const*         l_node;
  
  l_exit= 0;
  for (l_node= &(*i_exec).m_sql[0], l_slot_sql= 0;
    (*i_exec).m_count > l_slot_sql;
    l_slot_sql++,
    l_node++)
  {

    l_slot_attr= 0; 

    do
    {

      if ((*l_node).m_attr_count <= l_slot_attr)
      {
        break;
      }

      print_detail_cell(io_fp, &(*l_node).m_attr[l_slot_attr]);

      l_slot_attr++;

      if ((*l_node).m_attr_count > l_slot_attr)
      {
        fprintf(io_fp, ",");
      }

    }while(1);  

    fprintf(io_fp, "\r\n");

  }

  return l_exit;
}


static int
csv_print_header(
  FILE*const                            io_fp,
  struct query_exec const*const         i_exec)
{
  int                                   l_exit;
  unsigned                              l_slot_sql;
  unsigned                              l_slot_attr;
  struct query_exec_node const*         l_node;
  
  l_exit= 0;
  for (l_node= &(*i_exec).m_sql[0], l_slot_sql= 0;
    (*i_exec).m_count > l_slot_sql;
    l_slot_sql++,
    l_node++)
  {

    l_slot_attr= 0; 

    do
    {

      if ((*l_node).m_attr_count <= l_slot_attr)
      {
        break;
      }

      fprintf(io_fp, "\"%s\"", (*l_node).m_attr[l_slot_attr].m_tag);

      l_slot_attr++;

      if ((*l_node).m_attr_count > l_slot_attr)
      {
        fprintf(io_fp, ",");
      }

    }while(1);  

  }

  if (l_slot_attr)
  {
    fprintf(io_fp, "\r\n");
  }

  return l_exit;
}

static int
csv_run_loop(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct run_dialog*const               io_dialog,
  struct query_exec *const              io_exec)
{
  int                                   l_exit;
  int                                   l_eof;
  gint64                                l_time1;
  gint64                                l_time2;

  l_exit= 0;
  l_time1= g_get_monotonic_time();
  (*io_dialog).m_tuple= 1;

  do
  {

    csv_print_detail(io_fp, io_exec);

    if (g_csv_cancel)
    {
      break;
    }

    l_exit= query_exec_next(o_error, &l_eof, io_exec);

    if (l_exit)
    {
      break;
    }

    if (l_eof)
    {
      break;
    }

    (*io_dialog).m_tuple++;

    if ((*io_dialog).m_progressbar)
    {
      l_time2= g_get_monotonic_time();
      if (100000 < (l_time2 - l_time1))
      {
        l_time1= l_time2;
        gdk_threads_add_idle(csv_dialog_update, io_dialog);
      }
    }

  }while(1);

  if ((*io_dialog).m_progressbar)
  {
    csv_dialog_update(io_dialog);
  }

  return l_exit;
}

static int
csv_run_exec(
  struct run_dialog*const               io_dialog,
  struct exports_spec *const            io_export,
  struct goop const*const               i_goop)
{ 
  int                                   l_eof;
  GError*                               l_error;
  struct query_exec                     l_exec;
  int                                   l_exit;
  FILE*                                 l_fp;

  l_exit= 0;
  l_error= 0;
  query_exec_assign(&l_exec);
  l_fp= 0;

  do
  {
    
    l_fp= g_fopen((*i_goop).m_outputfile, "w+");

    if (0 == l_fp)
    {
      l_error= g_error_new(
        CSV,
        CSV_CREATE_FILE,
        "Unable to create file ('%s'); errno(%d)='%s'",
        (*i_goop).m_outputfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_exit= query_exec_connect_prepare_first(
      &l_error,
      &l_eof,
      &l_exec,
      &(*io_export).m_query,
      &(*io_export).m_aspect,
      (*i_goop).m_prompt,
      (*i_goop).m_interactive);

    if (l_exit)
    {
      break;
    }

    if (l_eof)
    {
      break;
    }

    if ((*io_export).m_config.m_print_header)
    {
      csv_print_header(l_fp, &l_exec);
    }

    csv_run_loop(&l_error, l_fp, io_dialog, &l_exec);

  }while(0);

  if (l_error)
  {
    if ((*i_goop).m_interactive)
    {
      _error_display_prompt(l_error);
    }
  }

  g_clear_error(&l_error);

  if (l_fp)
  {
    fclose(l_fp);
  }

  query_exec_discharge(&l_exec);

  return l_exit;
}

struct csv_thread_data
{
  struct run_dialog*                    m_dialog;
  struct exports_spec*                  m_export;
  struct goop const*                    m_goop;
};

static gpointer
csv_run_thread(
  gpointer                              i_data)
{
  int                                   l_exit;
  struct csv_thread_data*               l_info;

  l_info= (struct csv_thread_data*)i_data;

  l_exit= csv_run_exec(
    (*l_info).m_dialog,
    (*l_info).m_export,
    (*l_info).m_goop);

  if (0 == g_csv_cancel)
  {
    gdk_threads_add_idle(csv_dialog_end, (*l_info).m_dialog);
    while(0 == g_csv_cancel);
  }

  g_thread_exit((gpointer)(ptrdiff_t)l_exit);

  return (gpointer)(ptrdiff_t)l_exit;
}

extern int
export_csv_run(
  struct exports_spec *const            io_export,
  struct goop const*const               i_goop)
{
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  struct csv_thread_data                l_info;
  int                                   l_rc;
  struct run_dialog                     l_rd;
  GThread*                              l_thread;

  l_builder= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;
  l_thread= 0;
  memset(&l_rd, 0, sizeof(l_rd));
  memset(&l_info, 0, sizeof(l_info));

  do
  {

    g_csv_cancel= 0;

    if ((*i_goop).m_interactive)
    {
      database_aspect_password_fill(&g_csv_cancel, &(*io_export).m_aspect);

      if (g_csv_cancel)
      {
        break;
      }
    }
    else
    {
      l_exit= csv_run_exec(&l_rd, io_export, i_goop);
      break;
    }

    l_info.m_dialog= 0;
    l_info.m_export= io_export;
    l_info.m_goop= i_goop;

    l_builder= gtk_builder_new();
    l_rc= wrap_gtk_builder_add_from_file(l_builder, "runreport.glade", &l_error);

    if (0 == l_rc)
    {
      _error_log(l_error);
      g_warning("Unable to build dialog: %s\n", l_error->message);
      l_exit= -1;
      break;
    }

    l_dialog= (GtkDialog*)gtk_builder_get_object(l_builder, "run_dialog");

    if (0 == l_dialog)
    {
      g_warning("Unable to find dialog object: 'run_dialog'");
      l_exit= -1;
      break;
    }

    csv_run_property_bind(&l_rd, l_builder);

    gtk_window_set_transient_for(GTK_WINDOW(l_dialog), get_main_window());
    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);

    gtk_widget_grab_focus(gtk_dialog_get_widget_for_response(
      GTK_DIALOG(l_dialog), GTK_RESPONSE_CANCEL));

    l_info.m_dialog= &l_rd;
    l_thread= g_thread_new("csv", csv_run_thread, &l_info);

    gtk_dialog_run(l_dialog);

    g_csv_cancel= 1;
    g_thread_join(l_thread);

  }while(0);

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  g_clear_error(&l_error);

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  if (l_thread)
  {
    g_thread_unref(l_thread);
  }

  return l_exit;
}
