/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "cmd.h"

static void
cmd_group_clear(
  struct cmd_group*const                io_group)
{
  unsigned                              l_slot;
  struct item_node*                     l_node;

  if (cmd_create == (*io_group).m_type)
  {
    for (l_slot= 0; (*io_group).m_slots > l_slot; l_slot++)
    {
      if (0 == (*io_group).m_item[l_slot].m_object.m_create.m_is_ref)
      {
        l_node= (*io_group).m_item[l_slot].m_node;
        (*(*l_node).m_method.m_discharge)(&(*l_node).m_object);
        g_free(l_node);
      }
    }
  }
  else if (cmd_delete == (*io_group).m_type)
  {
    for (l_slot= 0; (*io_group).m_slots > l_slot; l_slot++)
    {
      if (0 == (*io_group).m_item[l_slot].m_object.m_delete.m_is_ref)
      {
        l_node= (*io_group).m_item[l_slot].m_node;
        (*(*l_node).m_method.m_discharge)(&(*l_node).m_object);
        g_free(l_node);
      }
    }
  }
  else if (cmd_paste == (*io_group).m_type)
  {
    for (l_slot= 0; (*io_group).m_slots > l_slot; l_slot++)
    {
      if (0 == (*io_group).m_item[l_slot].m_object.m_paste.m_is_ref)
      {
        l_node= (*io_group).m_item[l_slot].m_node;
        (*(*l_node).m_method.m_discharge)(&(*l_node).m_object);
        g_free(l_node);
      }
    }
  }

  g_free((*io_group).m_item);
  memset(io_group, 0, sizeof(*io_group));

  return;
}

static void
cmd_stack_clear(
  struct cmd_stack*const                io_stack)
{
  unsigned                              l_slot;
  struct cmd_group*                     l_group;

  for (l_slot= 0; (*io_stack).m_slots_used > l_slot; l_slot++)  
  {
    l_group= &(*io_stack).m_group[l_slot];
    cmd_group_clear(l_group);
  }

  g_free((*io_stack).m_group);

  memset(io_stack, 0, sizeof(*io_stack));

  return;
}

static void
cmd_stack_grow(
  struct cmd_stack*const                io_stack)
{
  unsigned                              l_bytes;

  if ((*io_stack).m_slots_allocated == (*io_stack).m_slots_used)
  {
    l_bytes= CMD_GROWBY + (*io_stack).m_slots_allocated;
    l_bytes*= sizeof(struct cmd_group);
    (*io_stack).m_group= (struct cmd_group*)realloc((*io_stack).m_group, l_bytes);
    (*io_stack).m_slots_allocated+= CMD_GROWBY;
  }

  return;
}

extern void
cmd_assign(
  struct cmd*const                      o_cmd)
{

  memset(o_cmd, 0, sizeof(*o_cmd));

  return;
}

extern void
cmd_clear(
  struct cmd*const                      io_cmd)
{

  cmd_stack_clear(&(*io_cmd).m_undo);
  cmd_stack_clear(&(*io_cmd).m_redo);
  memset(io_cmd, 0, sizeof(*io_cmd));

  return;
}

extern void
cmd_discharge(
  struct cmd*const                      io_cmd)
{

  cmd_clear(io_cmd);

  return;
}

extern char const*
cmd_get_label(
  enum cmd_type const                   i_type)
{
  char const*                           l_label;

  switch(i_type)
  {
    case cmd_create:
      l_label= "Insert";
      break;
    case cmd_delete:
      l_label= "Delete";
      break;
    case cmd_move:
      l_label= "Move";
      break;
    case cmd_order:
      l_label= "Order";
      break;
    case cmd_paste:
      l_label= "Paste";
      break;
    case cmd_resize:
      l_label= "Resize";
      break;
    case cmd_rotate:
      l_label= "Rotate";
      break;
    case cmd_shear:
      l_label= "Shear";
      break;
    default:
      l_label= 0;
      break;
  }

  return l_label;
}

extern char const*
cmd_get_redo_label(
  struct cmd const*const                i_cmd)
{
  struct cmd_group const*               l_group;
  char const*                           l_label;

  l_label= 0;

  if ((*i_cmd).m_redo.m_slots_used)
  {
    l_group= &(*i_cmd).m_redo.m_group[(*i_cmd).m_redo.m_slots_used-1];
    l_label= cmd_get_label((*l_group).m_type);
  }

  return l_label;
}

extern char const*
cmd_get_undo_label(
  struct cmd const*const                i_cmd)
{
  struct cmd_group const*               l_group;
  char const*                           l_label;

  l_label= 0;

  if ((*i_cmd).m_undo.m_slots_used)
  {
    l_group= &(*i_cmd).m_undo.m_group[(*i_cmd).m_undo.m_slots_used-1];
    l_label= cmd_get_label((*l_group).m_type);
  }

  return l_label;
}

static void
cmd_store_order(
  struct cmd_order*                     io_order,
  struct item*const                     io_item,
  struct item_node*const                io_node)
{
  unsigned                              l_slot;
  struct item_node*                     l_node;

  l_slot= 0;
  l_node= (*io_item).m_head;

  do
  {

    if (io_node == l_node)
    {
      (*io_order).m_slot= l_slot;
      break;
    }

    l_node= (*l_node).m_next;
    l_slot++;

  }while(1);

  return;
}

static void
cmd_store_item(
  struct cmd_item*const                 o_cmd_item,
  enum cmd_type const                   i_type,
  struct item*const                     io_item,
  struct item_node*const                io_node)
{
  struct cmd_move*                      l_move;
  struct cmd_order*                     l_order;
  struct cmd_resize*                    l_resize;
  struct cmd_rotate*                    l_rotate;
  struct cmd_shear*                     l_shear;

  memset(o_cmd_item, 0, sizeof(*o_cmd_item));
  (*o_cmd_item).m_node= io_node;

  switch(i_type)
  {
    case cmd_move:
      l_move= &(*o_cmd_item).m_object.m_move;
      (*(*io_node).m_method.m_get_position)(
        &(*l_move).m_pos_x, 
        &(*l_move).m_pos_y,
        &(*io_node).m_object);
      break;
    case cmd_order:
      l_order= &(*o_cmd_item).m_object.m_order;
      cmd_store_order(l_order, io_item, io_node);
      break;
    case cmd_resize:
      l_resize= &(*o_cmd_item).m_object.m_resize;
      (*(*io_node).m_method.m_get_size)(
        &(*l_resize).m_len_x, 
        &(*l_resize).m_len_y,
        &(*io_node).m_object);
      break;
    case cmd_rotate:
      l_rotate= &(*o_cmd_item).m_object.m_rotate;
      (*(*io_node).m_method.m_get_rotation)(
        &(*l_rotate).m_rotation, 
        &(*io_node).m_object);
      break;
    case cmd_shear:
      l_shear= &(*o_cmd_item).m_object.m_shear;
      (*(*io_node).m_method.m_get_shear)(
        &(*l_shear).m_shear_x, 
        &(*l_shear).m_shear_y,
        &(*io_node).m_object);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

static void
cmd_stack_push(
  struct cmd_stack*const                io_stack,
  IwrPage*const                         io_page,
  enum cmd_type const                   i_type)
{
  unsigned                              l_bytes;
  struct cmd_item*                      l_cmd_item;
  unsigned                              l_count;
  struct cmd_group*                     l_group;
  struct item *                         l_item;
  struct item_node *                    l_node;
  struct item_node *                    l_next;
  unsigned                              l_slot;

  do
  {

    iwr_page_get_item(&l_item, io_page);
    l_node= (*l_item).m_head;
    l_count= 0;

    for (; l_node; l_node= (*l_node).m_next)
    {
      if ((*l_node).m_selected)
      {
        l_count++;
      }
    }

    if (0 == l_count)
    {
      break;
    }

    cmd_stack_grow(io_stack);
    l_group= &(*io_stack).m_group[(*io_stack).m_slots_used];
    (*io_stack).m_slots_used++;

    (*l_group).m_type= i_type;
    (*l_group).m_page= io_page;
    (*l_group).m_slots= l_count;
    l_bytes= sizeof(struct cmd_item) * l_count;
    (*l_group).m_item= (struct cmd_item*)malloc(l_bytes);
    l_cmd_item= &(*l_group).m_item[0];

    l_node= (*l_item).m_head;
    l_slot= 0;

    do
    {

      if (0 == l_node)
      {
        break;
      }

      l_next= (*l_node).m_next;

      if ((*l_node).m_selected)
      {
        switch(i_type)
        {
          case cmd_create:
            memset(l_cmd_item, 0, sizeof(*l_cmd_item));
            (*l_cmd_item).m_node= l_node;
            (*l_cmd_item).m_object.m_create.m_is_ref= 1;
            break;
          case cmd_delete:
            memset(l_cmd_item, 0, sizeof(*l_cmd_item));
            (*l_cmd_item).m_node= l_node;
            (*l_cmd_item).m_object.m_delete.m_slot= l_slot;
            item_node_unlink(l_item, l_node);
            break;
          case cmd_paste:
            memset(l_cmd_item, 0, sizeof(*l_cmd_item));
            (*l_cmd_item).m_node= l_node;
            (*l_cmd_item).m_object.m_paste.m_is_ref= 1;
            break;
          default:
            cmd_store_item(l_cmd_item, i_type, l_item, l_node);
            break;
        }
        l_cmd_item++;
      }

      l_node= l_next;
      l_slot++;

    }while(1);

  }while(0);

  return;
}

extern void
cmd_push(
  struct cmd*const                      io_cmd,
  IwrPage*const                         io_page,
  enum cmd_type const                   i_type)
{

  cmd_stack_push(&(*io_cmd).m_undo, io_page, i_type);

  return;
}

static void
cmd_undo_order(
  struct cmd_order*const                io_order,
  struct item*const                     io_item,
  struct item_node *                    io_node)
{
  unsigned                              l_slot;
  struct item_node*                     l_node;
  struct item_node*                     l_anchor;

  l_slot= 0;
  l_node= (*io_item).m_head;

  do
  {

    if (io_node == l_node)
    {
      if (l_slot != (*io_order).m_slot)
      {
        item_node_get_nth(&l_anchor, io_item, (*io_order).m_slot);
        item_node_unlink(io_item, io_node);
        if (l_slot > (*io_order).m_slot)
        {
          item_node_insert_before(io_item, l_anchor, l_node);
        }
        else
        {
          item_node_insert_after(io_item, l_anchor, l_node);
        }
        (*io_order).m_slot= l_slot;
      }
      break;
    }

    l_node= (*l_node).m_next;
    l_slot++;

  }while(1);

  return;
}

static void
cmd_undo_item(
  struct cmd_item*const                 o_cmd_item,
  enum cmd_type const                   i_type,
  struct item*const                     io_item)
{
  struct cmd_move*                      l_move;
  struct item_node *                    l_node;
  struct cmd_order*                     l_order;
  struct cmd_resize*                    l_resize;
  struct cmd_rotate*                    l_rotate;
  struct cmd_shear*                     l_shear;

  l_node= (*o_cmd_item).m_node;

  switch(i_type)
  {
    case cmd_move:
      l_move= &(*o_cmd_item).m_object.m_move;
      (*(*l_node).m_method.m_set_position)(
        &(*l_node).m_object,
        (*l_move).m_pos_x, 
        (*l_move).m_pos_y);
      break;
    case cmd_order:
      l_order= &(*o_cmd_item).m_object.m_order;
      cmd_undo_order(l_order, io_item, l_node);
      break;
    case cmd_resize:
      l_resize= &(*o_cmd_item).m_object.m_resize;
      (*(*l_node).m_method.m_set_size)(
        &(*l_node).m_object,
        (*l_resize).m_len_x, 
        (*l_resize).m_len_y);
      break;
    case cmd_rotate:
      l_rotate= &(*o_cmd_item).m_object.m_rotate;
      (*(*l_node).m_method.m_set_rotation)(
        &(*l_node).m_object,
        (*l_rotate).m_rotation);
      break;
    case cmd_shear:
      l_shear= &(*o_cmd_item).m_object.m_shear;
      (*(*l_node).m_method.m_set_shear)(
        &(*l_node).m_object,
        (*l_shear).m_shear_x, 
        (*l_shear).m_shear_y);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

static void
cmd_pop(
  struct cmd_stack*const                io_undo,
  struct cmd_stack*const                io_redo)
{
  unsigned                              l_bytes;
  struct cmd_item*                      l_cmd_item;
  struct cmd_group*                     l_group;
  struct cmd_group*                     l_group_redo;
  struct item *                         l_item;
  unsigned                              l_slot;
  struct item_node*                     l_anchor;

  do
  {

    if (0 == (*io_undo).m_slots_used)
    {
      break;
    }

    (*io_undo).m_slots_used--;
    l_group= &(*io_undo).m_group[(*io_undo).m_slots_used];
    iwr_page_get_item(&l_item, (*l_group).m_page);

    cmd_stack_grow(io_redo);
    l_group_redo= &(*io_redo).m_group[(*io_redo).m_slots_used];
    (*l_group_redo).m_type= (*l_group).m_type;
    (*l_group_redo).m_page= (*l_group).m_page;
    (*l_group_redo).m_slots= (*l_group).m_slots;
    l_bytes= sizeof(struct cmd_item) * (*l_group).m_slots;
    (*l_group_redo).m_item= (struct cmd_item*)malloc(l_bytes);
    (*io_redo).m_slots_used++;

    for (l_slot= 0; (*l_group).m_slots > l_slot; l_slot++)
    {
      l_cmd_item= &(*l_group).m_item[l_slot];
      if (cmd_create == (*l_group).m_type)
      {
        (*l_group_redo).m_item[l_slot]= (*l_cmd_item);
        if ((*l_cmd_item).m_object.m_create.m_is_ref)
        {
          (*l_group_redo).m_item[l_slot].m_object.m_create.m_is_ref= 0; 
          item_node_unlink(l_item, (*l_cmd_item).m_node);
        }
        else
        {
          (*l_group_redo).m_item[l_slot].m_object.m_create.m_is_ref= 1; 
          item_node_append(l_item, (*l_cmd_item).m_node);
        }
      }
      else if (cmd_delete == (*l_group).m_type)
      {
        (*l_group_redo).m_item[l_slot]= (*l_cmd_item);
        if ((*l_cmd_item).m_object.m_delete.m_is_ref)
        {
          (*l_group_redo).m_item[l_slot].m_object.m_delete.m_is_ref= 0; 
          item_node_unlink(l_item, (*l_cmd_item).m_node);
        }
        else
        {
          (*l_group_redo).m_item[l_slot].m_object.m_delete.m_is_ref= 1; 
          item_node_get_nth(&l_anchor, l_item, (*l_cmd_item).m_object.m_delete.m_slot);
          if (l_anchor)
          {
            item_node_insert_before(l_item, l_anchor, (*l_cmd_item).m_node);
          }
          else
          {
            item_node_append(l_item, (*l_cmd_item).m_node);
          }
        }
      }
      else if (cmd_paste == (*l_group).m_type)
      {
        (*l_group_redo).m_item[l_slot]= (*l_cmd_item);
        if ((*l_cmd_item).m_object.m_paste.m_is_ref)
        {
          (*l_group_redo).m_item[l_slot].m_object.m_paste.m_is_ref= 0; 
          item_node_unlink(l_item, (*l_cmd_item).m_node);
        }
        else
        {
          (*l_group_redo).m_item[l_slot].m_object.m_paste.m_is_ref= 1; 
          item_node_append(l_item, (*l_cmd_item).m_node);
        }
      }
      else
      {
        cmd_store_item(
          &(*l_group_redo).m_item[l_slot], 
          (*l_group).m_type,
          l_item,
          (*l_cmd_item).m_node);
        cmd_undo_item(l_cmd_item, (*l_group).m_type, l_item);
      }
    }

    g_free((*l_group).m_item);

    gtk_widget_queue_draw(GTK_WIDGET((*l_group).m_page));

  }while(0);

  return;
}

extern void
cmd_redo(
  struct cmd*const                      io_cmd)
{

  cmd_pop(&(*io_cmd).m_redo, &(*io_cmd).m_undo);

  return;
}

extern void
cmd_undo(
  struct cmd*const                      io_cmd)
{

  cmd_pop(&(*io_cmd).m_undo, &(*io_cmd).m_redo);

  return;
}
