/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "barsym.h"

static char const*
barsym_codabar_encode(
  char const                            i_code)
{
  char const*                           l_encode;

  switch(i_code)
  {
    case '0':
      l_encode= "101010011";
      break;
    case '1':
      l_encode= "101011001";
      break;
    case '2':
      l_encode= "101001011";
      break;
    case '3':
      l_encode= "110010101";
      break;
    case '4':
      l_encode= "101101001";
      break;
    case '5':
      l_encode= "110101001";
      break;
    case '6':
      l_encode= "100101011";
      break;
    case '7':
      l_encode= "100101101";
      break;
    case '8':
      l_encode= "100110101";
      break;
    case '9':
      l_encode= "110100101";
      break;
    case '-':
      l_encode= "101001101";
      break;
    case '$':
      l_encode= "101100101";
      break;
    case ':':
      l_encode= "1101011011";
      break;
    case '/':
      l_encode= "1101101011";
      break;
    case '.':
      l_encode= "1101101101";
      break;
    case '+':
      l_encode= "101100110011";
      break;
    case 'A':
      l_encode= "1011001001";
      break;
    case 'B':
      l_encode= "1010010011";
      break;
    case 'C':
      l_encode= "1001001011";
      break;
    case 'D':
      l_encode= "1010011001";
      break;
    default:
      l_encode= 0;
      break;
  }

  return l_encode;
}

extern void
barsym_codabar(
  struct barsym*const                   io_barsym,
  unsigned char const*                  i_buffer,
  unsigned const                        i_length)
{
  double                                l_center;
  char const*                           l_encode;
  cairo_text_extents_t                  l_extent;
  unsigned char*                        l_filter;
  unsigned                              l_position;
  unsigned char const*                  l_ptr;

  l_position= 0;
  l_ptr= i_buffer;

  (*io_barsym).m_pos_x+= (*io_barsym).m_quiet_zone_left;

  do
  {

    if (i_length <= l_position)
    {
      break;
    }

    l_encode= barsym_codabar_encode(l_ptr[0]);
    barsym_write_bar(io_barsym, l_encode, 0);
    (*io_barsym).m_pos_x+= (*io_barsym).m_bar_width;

    l_ptr++;
    l_position++;

  }while(1);

  l_filter= (unsigned char*)malloc(i_length);
  memset(l_filter, 0, i_length);
  memcpy(l_filter, &i_buffer[1], i_length-2);

  cairo_text_extents((*io_barsym).m_cr, (char*)l_filter, &l_extent);
  l_center= ((*io_barsym).m_pos_x / 2.0) - (l_extent.width / 2.0);

  cairo_move_to((*io_barsym).m_cr, l_center, (*io_barsym).m_height);
  cairo_show_text((*io_barsym).m_cr, (char*)l_filter);
  free(l_filter);

  return;

}

extern double
barsym_codabar_get_width(
  struct barsym*const                   io_barsym,
  unsigned char const*                  i_buffer,
  unsigned const                        i_length)
{
  double                                l_width;

  l_width= (*io_barsym).m_quiet_zone_left;
  l_width+= (*io_barsym).m_quiet_zone_right;
  l_width+= (i_length * (*io_barsym).m_bar_width * 12.0);
  l_width+= (i_length * (*io_barsym).m_bar_width);
  
  return l_width;
}

extern int
barsym_codabar_verify(
  unsigned char const*                  i_buffer,
  unsigned const                        i_length)
{
  int                                   l_exit;
  unsigned char const*                  l_ptr;
  unsigned                              l_slot;
  char const*                           l_encode;

  l_exit= 0;
  l_ptr= i_buffer;
  l_slot= 0;

  do
  {

    if (i_length <= l_slot)
    {
      break;
    }

    l_encode= barsym_codabar_encode(*l_ptr);

    if (0 == l_encode)
    {
      l_exit= -1;
      break;
    }

    l_ptr++;
    l_slot++;

  }while(1);

  do
  {

    if (l_exit)
    {
      break;
    }

    if (2 > i_length)
    {
      l_exit= -1;
      break;
    }

    switch(i_buffer[0])
    {
      case 'A':
      case 'B':
      case 'C':
      case 'D':
        switch(i_buffer[i_length-1])
        {
          case 'A':
          case 'B':
          case 'C':
          case 'D':
            break;
          default:
            l_exit= -1;
            break;
        }
        break;
      default:
        l_exit= -1;
        break;
    }

  }while(0);

  return l_exit;
}
