/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "report.h"

extern void
report_assign(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_unassigned;
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_assign_export(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_export;
  export_assign(&(*o_report).m_object.m_export);
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_assign_filename(
  struct report*const                   io_report,
  char const*                           i_reportfile)
{

  if ((*io_report).m_reportfile)
  {
    g_free((*io_report).m_reportfile);
  }

  (*io_report).m_reportfile= report_util_get_report_filename(i_reportfile);

  return;
}

extern void
report_assign_label(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_label;
  label_assign(&(*o_report).m_object.m_label);
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_assign_layout(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_layout;
  layout_assign(&(*o_report).m_object.m_layout);
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_discharge(
  struct report*const                   io_report)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_discharge(&(*io_report).m_object.m_layout);
      break;
    case style_export:
      export_discharge(&(*io_report).m_object.m_export);
      break;
    case style_label:
      label_discharge(&(*io_report).m_object.m_label);
    default:
      break;
  }

  database_aspect_discharge(&(*io_report).m_aspect);
  query_discharge(&(*io_report).m_query);

  if ((*io_report).m_reportfile)
  {
    g_free((*io_report).m_reportfile);
  }

  memset(io_report, 0, sizeof(*io_report));

  return;
}

extern GtkWidget*
report_gui_new(
  struct report*const                   io_report)
{
  GtkWidget*                            l_widget;

  switch((*io_report).m_style)
  {
    case style_layout:
      l_widget= layout_gui_new(&(*io_report).m_object.m_layout);
      break;
    case style_export:
      l_widget= export_gui_new(&(*io_report).m_object.m_export);
      break;
    case style_label:
      l_widget= label_gui_new(&(*io_report).m_object.m_label);
      break;
    default:
      l_widget= 0; /* programming error */
      break;
  }

  return l_widget;
}

extern void
report_highlight_objects(
  struct report*const                   io_report,
  gboolean const                        i_state)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_highlight_objects(&(*io_report).m_object.m_layout, i_state);
      break;
    case style_label:
      label_highlight_objects(&(*io_report).m_object.m_label, i_state);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

extern struct item_node*
report_item_new(
  struct report*const                   io_report,
  enum item_type const                  i_type)
{
  struct item_node*                     l_node;

  l_node= 0;

  switch((*io_report).m_style)
  {
    case style_layout:
      l_node= layout_item_new(&(*io_report).m_object.m_layout, i_type);
      break;
    case style_label:
      l_node= label_item_new(&(*io_report).m_object.m_label, i_type);
      break;
    default:
      /* programming error */
      break;
  }

  return l_node;
}

extern int
report_read_label(
  GError**                              o_error,
  GtkWidget**                           o_widget,
  struct report*const                   io_report,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  struct label_spec                     l_spec;
  struct label*                         l_label;
  struct item*                          l_item;

  (*o_widget)= 0;
  l_error= 0;
  l_exit= 0;
  label_spec_assign(&l_spec);

  do
  {

    l_exit= label_spec_read(&l_error, &l_spec, i_xml);

    if (l_exit)
    {
      break;
    }

    (*io_report).m_style= style_label;

    (*io_report).m_query= l_spec.m_query;
    memset(&l_spec.m_query, 0, sizeof(l_spec.m_query));

    (*io_report).m_aspect= l_spec.m_aspect;
    memset(&l_spec.m_aspect, 0, sizeof(l_spec.m_aspect));

    l_label= &(*io_report).m_object.m_label;

    (*l_label).m_paper= l_spec.m_paper;
    label_paper_convert_to_points(&(*l_label).m_paper);

    (*o_widget)= label_gui_new(l_label);

    iwr_page_get_item(&l_item, (*l_label).m_page);
    (*l_item)= l_spec.m_report.m_item;
    memset(&l_spec.m_report.m_item, 0, sizeof(l_spec.m_report.m_item));

  }while(0);

  label_spec_discharge(&l_spec);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static void
report_layout_division_new_from_layout_spec(
  struct layout_division*const          io_div,
  struct layout_spec_division*const     io_spec,
  struct layout*const                   io_layout,
  unsigned*const                        io_row)
{
  struct item*                          l_item;
  IwrPage*                              l_page;
  struct paper                          l_paper;
  GtkWidget*                            l_widget;

  do
  {

    if (0 == (*io_spec).m_item)
    {
      break;
    }

    l_widget= gtk_button_new_with_label((*io_spec).m_tag);
    gtk_button_set_relief(GTK_BUTTON(l_widget), GTK_RELIEF_NONE);
    gtk_widget_set_sensitive(l_widget, 0);
    l_page= IWR_PAGE(iwr_page_new());
    (*io_div).m_page= l_page;
    g_strlcpy((*io_div).m_tag, (*io_spec).m_tag, sizeof((*io_div).m_tag));
    iwr_page_set_section_type(l_page, (*io_spec).m_type);
    l_paper= (*io_spec).m_paper;
    paper_convert_to_points(&l_paper);
    iwr_page_set_paper(l_page, &l_paper);
    iwr_page_get_item(&l_item, (*io_div).m_page);
    (*l_item)= *(*io_spec).m_item;
    memset((*io_spec).m_item, 0, sizeof(*(*io_spec).m_item));
    gtk_grid_attach((*io_layout).m_grid, l_widget, 0, (*io_row)++, 1, 1);
    gtk_grid_attach((*io_layout).m_grid, GTK_WIDGET(l_page), 0, (*io_row)++, 1, 1);
    g_signal_connect(l_page, "size-allocate", G_CALLBACK(layout_division_size_allocate_cb), io_layout);

  }while(0);

  return;
}

static int
report_read_layout(
  GError**                              o_error,
  GtkWidget**                           o_widget,
  struct report*const                   io_report,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  struct layout*                        l_layout;
  unsigned                              l_row;
  unsigned                              l_slot;
  struct layout_spec                    l_spec;

  (*o_widget)= 0;
  l_error= 0;
  l_exit= 0;
  layout_spec_assign(&l_spec);

  do
  {

    l_exit= layout_spec_read(&l_error, &l_spec, i_xml);

    if (l_exit)
    {
      break;
    }

    (*io_report).m_style= style_layout;

    (*io_report).m_query= l_spec.m_query;
    memset(&l_spec.m_query, 0, sizeof(l_spec.m_query));

    (*io_report).m_aspect= l_spec.m_aspect;
    memset(&l_spec.m_aspect, 0, sizeof(l_spec.m_aspect));

    l_layout= &(*io_report).m_object.m_layout;
    (*l_layout).m_scale= 1.0;

    (*l_layout).m_paper= l_spec.m_paper;
    paper_convert_to_points(&(*l_layout).m_paper);

    (*o_widget)= layout_gui_new(l_layout);

    l_row= 0;

    report_layout_division_new_from_layout_spec(
      &(*l_layout).m_cover.m_header,
      &l_spec.m_cover.m_header,
      l_layout,
      &l_row);

    report_layout_division_new_from_layout_spec(
      &(*l_layout).m_report.m_header,
      &l_spec.m_report.m_header,
      l_layout,
      &l_row);

    if (l_spec.m_group_slots)
    {
      (*l_layout).m_group= (struct layout_division_group*)
        calloc(l_spec.m_group_slots, sizeof(struct layout_division_group));

      (*l_layout).m_group_slots= l_spec.m_group_slots;

      for (l_slot= 0; l_spec.m_group_slots > l_slot; l_slot++)
      {
        memcpy(
          &(*l_layout).m_group[l_slot].m_break,
          &l_spec.m_group[l_slot].m_break,
          sizeof((*l_layout).m_group[l_slot].m_break));
        report_layout_division_new_from_layout_spec(
          &(*l_layout).m_group[l_slot].m_header,
          &l_spec.m_group[l_slot].m_header,
          l_layout,
          &l_row);
      }
    }

    report_layout_division_new_from_layout_spec(
      &(*l_layout).m_report.m_detail,
      &l_spec.m_report.m_detail,
      l_layout,
      &l_row);

    l_slot= l_spec.m_group_slots;

    do
    {

      if (0 ==l_slot)
      {
        break;
      }
      
      l_slot--;

      report_layout_division_new_from_layout_spec(
        &(*l_layout).m_group[l_slot].m_footer,
        &l_spec.m_group[l_slot].m_footer,
        l_layout,
        &l_row);

    }while(1);

    report_layout_division_new_from_layout_spec(
      &(*l_layout).m_report.m_footer,
      &l_spec.m_report.m_footer,
      l_layout,
      &l_row);

    report_layout_division_new_from_layout_spec(
      &(*l_layout).m_cover.m_footer,
      &l_spec.m_cover.m_footer,
      l_layout,
      &l_row);

  }while(0);

  layout_set_layout_size(&(*io_report).m_object.m_layout);
  layout_spec_discharge(&l_spec);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
report_read_export(
  GError**                              o_error,
  GtkWidget**                           o_widget,
  struct report*const                   io_report,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  struct exports*                       l_export;
  struct exports_spec                   l_spec;

  (*o_widget)= 0;
  l_error= 0;
  l_exit= 0;
  export_spec_assign(&l_spec);

  do
  {

    l_exit= export_spec_read(&l_error, &l_spec, i_xml);

    if (l_exit)
    {
      break;
    }

    (*io_report).m_style= style_export;

    (*io_report).m_query= l_spec.m_query;
    memset(&l_spec.m_query, 0, sizeof(l_spec.m_query));

    (*io_report).m_aspect= l_spec.m_aspect;
    memset(&l_spec.m_aspect, 0, sizeof(l_spec.m_aspect));

    l_export= &(*io_report).m_object.m_export;

    (*o_widget)= export_gui_new(l_export);

    export_load_attr_from_array(l_export, l_spec.m_field, l_spec.m_field_slots);
    export_reload(l_export, &(*io_report).m_aspect, &(*io_report).m_query);

  }while(0);

  export_spec_discharge(&l_spec);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
report_read(
  GError**                              o_error,
  GtkWidget**                           o_widget,
  struct report*const                   io_report)
{
  GError*                               l_error;
  int                                   l_exit;
  enum report_style                     l_style;
  struct xml*                           l_xml;

  (*o_widget)= 0;
  l_error= 0;
  l_exit= 0;
  l_xml= 0;

  do
  {

    l_exit= report_util_open_report(
      &l_error,
      &l_style,
      &l_xml,
      (*io_report).m_reportfile);

    if (l_exit)
    {
      break;
    }

    switch(l_style)
    {
      case style_layout:
        l_exit= report_read_layout(&l_error, o_widget, io_report, l_xml);
        break;
      case style_export:
        l_exit= report_read_export(&l_error, o_widget, io_report, l_xml);
        break;
      case style_label:
        l_exit= report_read_label(&l_error, o_widget, io_report, l_xml);
        break;
      default:
        /*caught in report_util_get_section */
        break;
    }

  }while(0);

  xml_free(l_xml);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
report_set_horizontal_ruler_line(
  struct report *const                  io_report,
  gboolean const                        i_state)
{

  switch((*io_report).m_style)
  {
  case style_layout:
    layout_set_horizontal_ruler_line(&(*io_report).m_object.m_layout, i_state);
    break;
  case style_label:
    label_set_horizontal_ruler_line(&(*io_report).m_object.m_label, i_state);
    break;
  default:
    /* programming error */
    break;
  }

  return;
}

extern void
report_set_vertical_ruler_line(
  struct report *const                  io_report,
  gboolean const                        i_state)
{

  switch((*io_report).m_style)
  {
  case style_layout:
    layout_set_vertical_ruler_line(&(*io_report).m_object.m_layout, i_state);
    break;
  case style_label:
    label_set_vertical_ruler_line(&(*io_report).m_object.m_label, i_state);
    break;
  default:
    /* programming error */
    break;
  }

  return;
}

extern void
report_set_scale(
  struct report *const                  io_report,
  double const                          i_scale)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_set_scale(&(*io_report).m_object.m_layout, i_scale);
      break;
    case style_label:
      label_set_scale(&(*io_report).m_object.m_label, i_scale);
    default:
      /* programming error */
      break;
  }

  return;
}

extern void
report_set_scale_zoom_in(
  struct report *const                  io_report)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_set_scale_zoom_in(&(*io_report).m_object.m_layout);
      break;
    case style_label:
      label_set_scale_zoom_in(&(*io_report).m_object.m_label);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

extern void
report_set_scale_zoom_out(
  struct report *const                  io_report)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_set_scale_zoom_out(&(*io_report).m_object.m_layout);
      break;
    case style_label:
      label_set_scale_zoom_out(&(*io_report).m_object.m_label);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

extern int
report_write(
  GError**                              o_error,
  struct report const*const             i_report)
{
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  char const*                           l_text;

  l_error= 0;
  l_exit= 0;
  l_fp= 0;

  do
  {

    switch((*i_report).m_style)
    {
      case style_layout:
        l_text= "layout";
        break;
      case style_export:
        l_text= "export";
        break;
      case style_label:
        l_text= "label";
        break;
      default:
        /* programming error */
        l_exit= -1;
        break;
    }

    if (l_exit)
    {
      break;
    }

    l_fp= g_fopen((*i_report).m_reportfile, "w+b");

    if (0 == l_fp)
    {
      l_error= g_error_new(
        REPORT, 
        REPORT_FILE_CREATE,
        "Unable to create file ('%s'); errno(%d)='%s'",
        (*i_report).m_reportfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    fprintf(l_fp, "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
    fprintf(l_fp, "<iwrite style=\"%s\" version=\"%d.%d.%d.%d (%s.%s.%s-%s)\">\n",
      l_text,
      g_version.m_major,
      g_version.m_minor,
      g_version.m_stage,
      g_version.m_micro,
      g_version.m_stamp.m_year,
      g_version.m_stamp.m_month,
      g_version.m_stamp.m_day,
      version_get_stage_text(g_version.m_stage));

    l_exit= query_write(&l_error, l_fp, &(*i_report).m_query);

    if (l_exit)
    {
      break;
    }

    l_exit= database_aspect_write(&l_error, l_fp, &(*i_report).m_aspect);

    if (l_exit)
    {
      break;
    }

    switch((*i_report).m_style)
    {
      case style_layout:
        layout_write(&l_error, l_fp, &(*i_report).m_object.m_layout);
        break;
      case style_export:
        export_write(&l_error, l_fp, &(*i_report).m_object.m_export);
        break;
      case style_label:
        label_write(&l_error, l_fp, &(*i_report).m_object.m_label);
        break;
      default:
        break;
    }

    fprintf(l_fp, "</iwrite>\n");

  }while(0);

  if (l_fp)
  {
    fclose(l_fp);
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}
