/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "layout_pdf.h"

static int                              g_pdf_cancel= 0;

struct run_dialog
{
  GtkProgressBar*                       m_progressbar;
  GtkLabel*                             m_file_label;
  GtkButton*                            m_cancel_button;
  size_t                                m_tuple;
};

static gboolean
pdf_dialog_end(
  gpointer                              i_data)
{
  struct run_dialog*                    l_rd;

  l_rd= (struct run_dialog*)i_data;
  gtk_button_set_label((*l_rd).m_cancel_button, "Ok");
  g_pdf_cancel= 1;

  return G_SOURCE_REMOVE;
}

static gboolean
pdf_dialog_update(
  gpointer                              i_data)
{
  char                                  l_text[1+31];
  struct run_dialog*                    l_rd;

  if (0 == g_pdf_cancel)
  {
    l_rd= (struct run_dialog*)i_data;
    gtk_progress_bar_pulse((*l_rd).m_progressbar);
    snprintf(l_text, sizeof(l_text), "%ld", (*l_rd).m_tuple);
    gtk_label_set_text((*l_rd).m_file_label, l_text);
  }

  return G_SOURCE_REMOVE;
}

static int
pdf_run_property_bind(
  struct run_dialog*const               io_rd,
  GtkBuilder*const                      i_builder)
{
  int                                   l_exit;

  l_exit= 0;

  do
  {

    (*io_rd).m_progressbar= GTK_PROGRESS_BAR(
      gtk_builder_get_object(i_builder, "progressbar1"));

    (*io_rd).m_file_label= GTK_LABEL(
      gtk_builder_get_object(i_builder, "file_label"));

    (*io_rd).m_cancel_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "cancel_button"));

  }while(0);

  return l_exit;
}

static void
pdf_draw_division(
  struct layout_spec_division*const     io_division,
  struct rw*const                       io_rw)
{

  do
  {

    if (0 == (*io_division).m_item || 0 == (*io_division).m_item->m_head)
    {
      break;
    }

    cairo_save((*io_rw).m_cr);
    cairo_translate((*io_rw).m_cr, 0.0, (*io_rw).m_pos_y);
    item_draw((*io_division).m_item, (*io_rw).m_cr, 0);
    cairo_restore((*io_rw).m_cr);

    (*io_rw).m_pos_y+= (*io_division).m_paper.m_height;

  }while(0);

  return;
}

static void
pdf_draw_footer(
  struct layout_spec*const              io_layout,
  struct rw*const                       io_rw)
{
  struct layout_spec_division*          l_footer;

  do
  {

    l_footer= &(*io_layout).m_report.m_footer;

    if (0 == (*l_footer).m_item || 0 == (*l_footer).m_item->m_head)
    {
      break;
    }

    (*io_rw).m_pos_y= (*io_layout).m_paper.m_height - (*l_footer).m_paper.m_height;

    rw_item_set_value((*l_footer).m_item, io_rw);
    pdf_draw_division(l_footer, io_rw);

  }while(0);

  cairo_show_page((*io_rw).m_cr);
  (*io_rw).m_pos_y= 0.0;
  (*io_rw).m_page_number++;

  return;
}

static void
pdf_draw_group_footer(
  struct layout_spec*const              io_layout,
  struct rw*const                       io_rw)
{
  unsigned                              l_slot;
  struct layout_spec_division_group*    l_group;

  l_slot= (*io_layout).m_group_slots;

  do
  {

    if (0 == l_slot)
    {
      break;
    }

    l_slot--;

    l_group= &(*io_layout).m_group[l_slot];
    pdf_draw_division(&(*l_group).m_footer, io_rw);

  }while(1);

  return;
}

static void
pdf_draw_headers(
  struct layout_spec*const              io_layout,
  struct rw*const                       io_rw)
{
  unsigned                              l_slot;

  pdf_draw_division(&(*io_layout).m_report.m_header, io_rw);

  for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
  {
    pdf_draw_division(&(*io_layout).m_group[l_slot].m_header, io_rw);
  }

  return;
}

static void
pdf_check_group_break(
  struct rw*const                       io_rw,
  struct layout_spec *const             io_layout)
{
  struct iwsql_attr const*              l_attr;
  struct layout_spec_division_group*    l_group;
  int                                   l_rc;
  unsigned                              l_slot;

  l_slot= (*io_layout).m_group_slots;

  do
  {

    if (0 == l_slot)
    {
      break;
    }

    l_slot--;
    l_group= &(*io_layout).m_group[l_slot];

    rw_attr_lookup_by_name(
      &l_attr,
      &(*io_rw).m_exec,
      "primary",
      (*l_group).m_break.m_tag);

    do
    {

      if (0 == l_attr)
      {
        break;
      }

      l_rc= strcmp((*l_group).m_break.m_value, (*l_attr).m_value);

      if (0 == l_rc)
      {
        break;
      }

      if ((*io_rw).m_fence_bottom <= (*io_rw).m_pos_y+(*l_group).m_footer.m_paper.m_height)
      {
        pdf_draw_footer(io_layout, io_rw);
        pdf_draw_headers(io_layout, io_rw);
      }

      pdf_draw_division(&(*l_group).m_footer, io_rw);

      rw_item_clear_accumulator((*l_group).m_footer.m_item);
      g_free((*l_group).m_break.m_value);
      (*l_group).m_break.m_value= g_strdup((*l_attr).m_value);

      if ((*io_rw).m_fence_bottom <= (*io_rw).m_pos_y+(*l_group).m_header.m_paper.m_height)
      {
        pdf_draw_footer(io_layout, io_rw);
        pdf_draw_headers(io_layout, io_rw);
        break;
      }

      rw_item_set_value((*l_group).m_header.m_item, io_rw);
      pdf_draw_division(&(*l_group).m_header, io_rw);

    }while(0);

  }while(1);

  return;
}

static void
pdf_check_page_break(
  int*                                  o_break,
  struct layout_spec*const              io_layout,
  struct rw*const                       io_rw)
{
  struct layout_spec_division_break*    l_break;
  struct iwsql_attr const*              l_attr;
  int                                   l_rc;

  (*o_break)= 0;

  do
  {

    l_break= &(*io_layout).m_page_break;

    if (0 == (*l_break).m_tag[0])
    {
      break;
    }

    rw_attr_lookup_by_name(
      &l_attr,
      &(*io_rw).m_exec,
      "primary",
      (*l_break).m_tag);

    if (0 == l_attr)
    {
      break;
    }

    if ((*l_break).m_value)
    {
      l_rc= strcmp((*l_break).m_value, (*l_attr).m_value);
    }
    else
    {
      l_rc= -1;
    }

    if (l_rc)
    {
      g_free((*l_break).m_value);
      (*l_break).m_value= g_strdup((*l_attr).m_value);
      (*o_break)= 1;
    }

  }while(0);

  return;
}

static void
pdf_check_break(
  struct rw*const                       io_rw,
  struct layout_spec *const             io_layout)
{
  int                                   l_break;

  do
  {

    pdf_check_page_break(&l_break, io_layout, io_rw);

    if (l_break)
    {
      pdf_draw_group_footer(io_layout, io_rw);
      pdf_draw_footer(io_layout, io_rw);
      pdf_draw_headers(io_layout, io_rw);
      break;
    }

    pdf_check_group_break(io_rw, io_layout);

  }while(0);

  return;
}

static int
pdf_run_loop(
  GError**                              o_error,
  struct run_dialog*const               io_dialog,
  struct rw*const                       io_rw,
  struct layout_spec *const             io_layout)
{
  struct layout_spec_division*          l_detail;
  int                                   l_eof;
  int                                   l_exit;
  struct paper                          l_paper_detail;
  gint64                                l_time1;
  gint64                                l_time2;

  l_detail= &(*io_layout).m_report.m_detail;

  if ((*l_detail).m_item && (*l_detail).m_item->m_head)
  {
    l_paper_detail= (*l_detail).m_paper;
  }
  else
  {
    memset(&l_paper_detail, 0, sizeof(l_paper_detail));
  }

  l_exit= 0;
  l_time1= g_get_monotonic_time();
  (*io_dialog).m_tuple= 1;

  do
  {

    if ((*io_dialog).m_progressbar)
    {
      l_time2= g_get_monotonic_time();
      if (100000 < (l_time2 - l_time1))
      {
        l_time1= l_time2;
        gdk_threads_add_idle(pdf_dialog_update, io_dialog);
      }
    }

    rw_item_set_value((*l_detail).m_item, io_rw);
    pdf_draw_division(l_detail, io_rw);

    if (g_pdf_cancel)
    {
      cairo_show_page((*io_rw).m_cr);
      break;
    }

    rw_layout_group_set_value(io_layout, io_rw);

    l_exit= query_exec_next(o_error, &l_eof, &(*io_rw).m_exec);

    if (l_exit)
    {
      break;
    }

    if (l_eof)
    {
      pdf_draw_group_footer(io_layout, io_rw);
      pdf_draw_footer(io_layout, io_rw);
      break;
    }

    pdf_check_break(io_rw, io_layout);

    (*io_dialog).m_tuple++;

    if ((*io_rw).m_fence_bottom <= (*io_rw).m_pos_y+l_paper_detail.m_height)
    {
      pdf_draw_footer(io_layout, io_rw);
      pdf_draw_headers(io_layout, io_rw);
    }

  }while(1);

  if ((*io_dialog).m_progressbar)
  {
    pdf_dialog_update(io_dialog);
  }

  return l_exit;
}

static int
pdf_run_exec(
  struct run_dialog*const               io_dialog,
  struct layout_spec *const             io_layout,
  struct goop const*const               i_goop)
{ 
  int                                   l_eof;
  GError*                               l_error;
  int                                   l_exit;
  struct rw                             l_rw;
  unsigned                              l_slot;

  l_error= 0;
  l_exit= 0;
  rw_assign(&l_rw);

  do
  {

    l_exit= rw_cairo_pdf_surface_create(
      &l_error,
      &l_rw,
      (*io_layout).m_paper.m_width,
      (*io_layout).m_paper.m_height,
      (*i_goop).m_outputfile);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_connect_prepare_first(
      &l_error,
      &l_eof,
      &l_rw.m_exec,
      &(*io_layout).m_query,
      &(*io_layout).m_aspect,
      (*i_goop).m_prompt,
      (*i_goop).m_interactive);

    if (l_exit)
    {
      break;
    }

    rw_layout_clear_accumulator(io_layout);
    rw_layout_set_fence(&l_rw, io_layout);
    pdf_check_page_break(&l_eof, io_layout, &l_rw);
    rw_level_break_set_value(io_layout, &l_rw);
    rw_item_set_value((*io_layout).m_cover.m_header.m_item, &l_rw);
    rw_item_set_value((*io_layout).m_report.m_header.m_item, &l_rw);

    for (l_slot= 0; (*io_layout).m_group_slots > l_slot; l_slot++)
    {
      rw_item_set_value((*io_layout).m_group[l_slot].m_header.m_item, &l_rw);
    }

    pdf_draw_headers(io_layout, &l_rw);

    l_exit= pdf_run_loop(&l_error, io_dialog, &l_rw, io_layout);

  }while(0);

  if (l_error)
  {
    if ((*i_goop).m_interactive)
    {
      _error_display_prompt(l_error);
    }
  }

  g_clear_error(&l_error);

  rw_discharge(&l_rw);

  return l_exit;
}

struct pdf_thread_data
{
  struct run_dialog*                    m_dialog;
  struct layout_spec*                   m_layout;
  struct goop const*                    m_goop;
};

static gpointer
pdf_run_thread(
  gpointer                              i_data)
{
  int                                   l_exit;
  struct pdf_thread_data*               l_info;

  l_info= (struct pdf_thread_data*)i_data;

  l_exit= pdf_run_exec(
    (*l_info).m_dialog,
    (*l_info).m_layout,
    (*l_info).m_goop);

  if (0 == g_pdf_cancel)
  {
    gdk_threads_add_idle(pdf_dialog_end, (*l_info).m_dialog);
    while(0 == g_pdf_cancel);
  }

  g_thread_exit((gpointer)(ptrdiff_t)l_exit);

  return (gpointer)(ptrdiff_t)l_exit;
}

extern int
layout_pdf_run(
  struct layout_spec *const             io_layout,
  struct goop const*const               i_goop)
{
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  struct pdf_thread_data                l_info;
  int                                   l_rc;
  struct run_dialog                     l_rd;
  GThread*                              l_thread;

  l_builder= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;
  l_thread= 0;
  memset(&l_rd, 0, sizeof(l_rd));
  memset(&l_info, 0, sizeof(l_info));

  do
  {

    g_pdf_cancel= 0;

    if ((*i_goop).m_interactive)
    {
      database_aspect_password_fill(&g_pdf_cancel, &(*io_layout).m_aspect);

      if (g_pdf_cancel)
      {
        break;
      }
    }
    else
    {
      l_exit= pdf_run_exec(&l_rd, io_layout, i_goop);
      break;
    }

    l_info.m_dialog= 0;
    l_info.m_layout= io_layout;
    l_info.m_goop= i_goop;

    l_builder= gtk_builder_new();
    l_rc= wrap_gtk_builder_add_from_file(l_builder, "runreport.glade", &l_error);

    if (0 == l_rc)
    {
      g_warning("Unable to build dialog: %s\n", l_error->message);
      l_exit= -1;
      break;
    }

    l_dialog= (GtkDialog*)gtk_builder_get_object(l_builder, "run_dialog");

    if (0 == l_dialog)
    {
      g_warning("Unable to find dialog object: 'run_dialog'");
      l_exit= -1;
      break;
    }

    pdf_run_property_bind(&l_rd, l_builder);

    gtk_window_set_transient_for(GTK_WINDOW(l_dialog), get_main_window());
    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);

    gtk_widget_grab_focus(gtk_dialog_get_widget_for_response(
      GTK_DIALOG(l_dialog), GTK_RESPONSE_CANCEL));

    l_info.m_dialog= &l_rd;
    l_thread= g_thread_new("pdf", pdf_run_thread, &l_info);

    gtk_dialog_run(l_dialog);

    g_pdf_cancel= 1;
    g_thread_join(l_thread);

  }while(0);

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  g_clear_error(&l_error);

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  if (l_thread)
  {
    g_thread_unref(l_thread);
  }

  return l_exit;
}
