/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_line.h"

extern void
item_line_assign(
  struct item_line*const                o_line,
  struct item_method*const              o_method)
{

  memset(o_line, 0, sizeof(*o_line));

  (*o_line).m_pos_x1= 10.0;
  (*o_line).m_pos_y1= 10.0;
  (*o_line).m_pos_x2= 100.0;
  (*o_line).m_pos_y2= 10.0;
  (*o_line).m_style= line_style_solid;
  (*o_line).m_thickness= 1;
  (*o_line).m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, line);

  return;
}

extern void
item_line_discharge(
  struct item_line*const                io_line)
{
  memset(io_line, 0, sizeof(*io_line));
}

extern void
item_line_draw(
  struct item_line*const                i_line,
  struct item_draw_context const*const  i_ctx)
{

  do
  {

    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_line).m_color.red,
      (*i_line).m_color.green,
      (*i_line).m_color.blue,
      (*i_line).m_color.alpha);

    item_common_cairo_dash_set((*i_ctx).m_cr, (*i_line).m_style);
    cairo_set_line_width((*i_ctx).m_cr, (*i_line).m_thickness);

    cairo_move_to((*i_ctx).m_cr, (*i_line).m_pos_x1, (*i_line).m_pos_y1);
    cairo_line_to((*i_ctx).m_cr, (*i_line).m_pos_x2, (*i_line).m_pos_y2);
    cairo_stroke((*i_ctx).m_cr);

    if ((*i_line).m_double_line)
    {
      switch((*i_line).m_type)
      {
        case line_type_horizontal:
          cairo_move_to((*i_ctx).m_cr, (*i_line).m_pos_x1, 3.00+(*i_line).m_pos_y1);
          cairo_line_to((*i_ctx).m_cr, (*i_line).m_pos_x2, 3.00+(*i_line).m_pos_y2);
          cairo_stroke((*i_ctx).m_cr);
          break;
        case line_type_vertical:
          cairo_move_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x1, (*i_line).m_pos_y1);
          cairo_line_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x2, (*i_line).m_pos_y2);
          cairo_stroke((*i_ctx).m_cr);
          break;
        case line_type_diagonal:
          cairo_move_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x1, 3.00+(*i_line).m_pos_y1);
          cairo_line_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x2, 3.00+(*i_line).m_pos_y2);
          cairo_stroke((*i_ctx).m_cr);
        default:
          break;
      }
    }

    if (0 == (*i_ctx).m_selected)
    {
      break;
    }

    cairo_set_source_rgb((*i_ctx).m_cr, 0.0, 1.0, 0.0);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_line).m_pos_x1-4,
      (*i_line).m_pos_y1-4,
      8,
      8);
    cairo_fill((*i_ctx).m_cr);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_line).m_pos_x2-4,
      (*i_line).m_pos_y2-4,
      8,
      8);
    cairo_fill((*i_ctx).m_cr);

  }while(0);

  return;
}

extern void
item_line_get_bbox(
  struct geom_rect*const                o_bbox,
  void const*const                      i_object)
{
  struct item_line const*               l_line;

  l_line= (struct item_line const*)i_object;
  (*o_bbox).m_pos_x= (*l_line).m_pos_x1;
  (*o_bbox).m_pos_y= (*l_line).m_pos_y1;
  (*o_bbox).m_len_x= (*l_line).m_pos_x2 - (*l_line).m_pos_x1;
  (*o_bbox).m_len_y= (int)(*l_line).m_thickness;

  return;
}

static int
item_line_in_event_grips(
  enum event_type*const                 o_type,
  struct item_line const*const          i_line,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {

    l_hit= item_common_event_inside(
      i_ctx,
      (*i_line).m_pos_x1-4,
      (*i_line).m_pos_y1-4,
      8,
      8);

    if (l_hit)
    {
      if (line_type_horizontal == (*i_line).m_type)
      {
        (*o_type)= event_resize_west;
      }
      else if (line_type_vertical == (*i_line).m_type)
      {
        (*o_type)= event_resize_north;
      }
      else
      {
        (*o_type)= event_resize_north_west;
      }
      break;
    }
    
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_line).m_pos_x2-4,
      (*i_line).m_pos_y2-4,
      8,
      8);

    if (l_hit)
    {
      if (line_type_horizontal == (*i_line).m_type)
      {
        (*o_type)= event_resize_east;
      }
      else if (line_type_vertical == (*i_line).m_type)
      {
       (*o_type)= event_resize_south;
      }
      else
      {
        (*o_type)= event_resize_south_east;
      }
      break;
    }

  }while(0);

  return l_hit;
}

extern void
item_line_in_event(
  enum event_type*const                 o_type,
  struct item_line const*const          i_line,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {

    (*o_type)= event_none;

    l_hit= item_common_line_in_event(i_ctx, (struct geom_line*)i_line);

    if (0 == l_hit)
    {
      break;
    }

    if ((*i_ctx).m_selected)
    {
      l_hit= item_line_in_event_grips(o_type, i_line, i_ctx);

      if (l_hit)
      {
        break;
      }
    }

    (*o_type)= event_move;

  }while(0);

  return;
}

extern void
item_line_move(
  struct item_line*const                io_line,
  struct geom_point const*const         i_event)
{

  (*io_line).m_pos_x1+= (*i_event).m_pos_x;
  (*io_line).m_pos_y1+= (*i_event).m_pos_y;
  (*io_line).m_pos_x2+= (*i_event).m_pos_x;
  (*io_line).m_pos_y2+= (*i_event).m_pos_y;

  return;
}

extern void
item_line_normalize(
  struct item_line*const                io_line,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  gdouble                               l_swap;

  do
  {

    if (line_type_diagonal == (*io_line).m_type)
    {
      break;
    }

    if ((*io_line).m_pos_x1 > (*io_line).m_pos_x2)
    {
      l_swap= (*io_line).m_pos_x1;
      (*io_line).m_pos_x1= (*io_line).m_pos_x2;
      (*io_line).m_pos_x2= l_swap;
    }

    if ((*io_line).m_pos_y1 > (*io_line).m_pos_y2)
    {
      l_swap= (*io_line).m_pos_y1;
      (*io_line).m_pos_y1= (*io_line).m_pos_y2;
      (*io_line).m_pos_y2= l_swap;
    }


    if (0 > (int)(*io_line).m_pos_x2)
    {
      l_swap= (*io_line).m_pos_x2 - (*io_line).m_pos_x1;
      (*io_line).m_pos_x2= 2.0;
      (*io_line).m_pos_x1= (*io_line).m_pos_x2 - l_swap;
    }

    if (0 > (int)(*io_line).m_pos_y2)
    {
      l_swap= (*io_line).m_pos_y2 - (*io_line).m_pos_y1;
      (*io_line).m_pos_y1= 2.0 - l_swap;
      (*io_line).m_pos_y2= (*io_line).m_pos_y1 + l_swap;
    }

    if (i_bound_width < (int)(*io_line).m_pos_x1)
    {
      l_swap= (*io_line).m_pos_x2 - (*io_line).m_pos_x1;
      (*io_line).m_pos_x1= i_bound_width - 2.0;
      (*io_line).m_pos_x2= (*io_line).m_pos_x1 + l_swap;
    }

    if (i_bound_height < (int)(*io_line).m_pos_y1)
    {
      l_swap= (*io_line).m_pos_y2 - (*io_line).m_pos_y1;
      (*io_line).m_pos_y1= i_bound_height - 2.0;
      (*io_line).m_pos_y2= (*io_line).m_pos_y1 + l_swap;
    }

  }while(0);

  return;
}

extern int
item_line_read(
  GError**                              o_error,
  struct item_line*const                io_line,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_pos_x1:
          (*io_line).m_pos_x1= xml_convert_double(l_value);
          break;
        case element_tag_pos_y1:
          (*io_line).m_pos_y1= xml_convert_double(l_value);
          break;
        case element_tag_pos_x2:
          (*io_line).m_pos_x2= xml_convert_double(l_value);
          break;
        case element_tag_pos_y2:
          (*io_line).m_pos_y2= xml_convert_double(l_value);
          break;
        case element_tag_style:
          (*io_line).m_style= item_common_get_line_style(l_value);
          break;
        case element_tag_thickness:
          (*io_line).m_thickness= g_ascii_strtod(l_value, 0);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_line).m_color= l_rgba;
          }
          break;
        case element_tag_type:
          (*io_line).m_type= line_type_horizontal;
          do
          {
            l_rc= strcmp("vertical", l_value);
            if (0 == l_rc)
            {
              (*io_line).m_type= line_type_vertical;
              break;
            }
            l_rc= strcmp("diagonal", l_value);
            if (0 == l_rc)
            {
              (*io_line).m_type= line_type_diagonal;
              break;
            }
          }while(0);
          break;
        case element_tag_doubleline:
          l_value= xml_lookup_attribute_value(l_xml, "on");
          (*io_line).m_double_line= xml_convert_boolean(l_value);
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_line_resize(
  struct item_line*const                io_line,
  struct item_resize_event const*const  i_ctx)
{
  
  switch((*i_ctx).m_type)
  {
    case event_resize_east:
      (*io_line).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north:
      (*io_line).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_north_east:
      (*io_line).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north_west:
      (*io_line).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south:
      (*io_line).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_south_east:
      (*io_line).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south_west:
      (*io_line).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_west:
      (*io_line).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    default:
      break;
  }

  return;
}

extern int
item_line_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_line const*const          i_line)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"line\">\n");
  fprintf(io_fp, "\t\t\t\t<pos_x1>%s</pos_x1>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_x1));
  fprintf(io_fp, "\t\t\t\t<pos_y1>%s</pos_y1>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_y1));
  fprintf(io_fp, "\t\t\t\t<pos_x2>%s</pos_x2>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_x2));
  fprintf(io_fp, "\t\t\t\t<pos_y2>%s</pos_y2>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_pos_y2));
  l_text= gdk_rgba_to_string(&(*i_line).m_color);
  fprintf(io_fp, "\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  fprintf(io_fp, "\t\t\t\t<style>%s</style>\n", item_common_get_line_style_text((*i_line).m_style));
  fprintf(io_fp, "\t\t\t\t<thickness>%s</thickness>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_line).m_thickness));

  switch((*i_line).m_type)
  {
    case line_type_horizontal:
      l_text= "horizontal";
      break;
    case line_type_vertical:
      l_text= "vertical";
      break;
    case line_type_diagonal:
      l_text= "diagonal";
      break;
    default:
      l_text= ""; 
  }

  fprintf(io_fp, "\t\t\t\t<type>%s</type>\n", l_text);
  fprintf(io_fp, "\t\t\t\t<double_line on=\"%s\"/>\n", ((*i_line).m_double_line ? "true" : "false"));
  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
