/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_ftext.h"

extern void
item_ftext_assign(
  struct item_ftext*const               o_ftext,
  struct item_method*const              o_method)
{

  memset(o_ftext, 0, sizeof(*o_ftext));

  (*o_ftext).m_len_x= 30.0;
  (*o_ftext).m_len_y= 18.0;
  (*o_ftext).m_font= pango_font_description_new();
  (*o_ftext).m_color.alpha= 1.0;
  (*o_ftext).m_border.m_style= line_style_solid;
  (*o_ftext).m_border.m_thickness= 1.0;
  (*o_ftext).m_border.m_color.alpha= 1.0;
  (*o_ftext).m_fill.m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, ftext);

  return;
}

extern void
item_ftext_discharge(
  struct item_ftext*const                io_ftext)
{
  if ((*io_ftext).m_font)
  {
    pango_font_description_free((*io_ftext).m_font);
  }

  if ((*io_ftext).m_value)
  {
    g_free((*io_ftext).m_value);
  }

  memset(io_ftext, 0, sizeof(*io_ftext));

  return;
}

extern void
item_ftext_draw(
  struct item_ftext*const               i_ftext,
  struct item_draw_context const*const  i_ctx)
{
  PangoFontDescription*                 l_font;
  PangoLayout*                          l_layout;
  char*                                 l_text;

  l_font= 0;
  l_layout= 0;

  do
  {

    if ((*i_ftext).m_fill.m_active)
    {
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_ftext).m_fill.m_color.red,
        (*i_ftext).m_fill.m_color.green,
        (*i_ftext).m_fill.m_color.blue,
        (*i_ftext).m_fill.m_color.alpha);

      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_ftext).m_pos_x,
        (*i_ftext).m_pos_y,
        (*i_ftext).m_len_x,
        (*i_ftext).m_len_y);

      cairo_fill((*i_ctx).m_cr);
    }

    cairo_save((*i_ctx).m_cr);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_ftext).m_pos_x,
      (*i_ftext).m_pos_y,
      (*i_ftext).m_len_x,
      (*i_ftext).m_len_x);

    cairo_clip((*i_ctx).m_cr);

    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_ftext).m_color.red,
      (*i_ftext).m_color.green,
      (*i_ftext).m_color.blue,
      (*i_ftext).m_color.alpha);

    l_layout= pango_cairo_create_layout((*i_ctx).m_cr);
    pango_layout_set_font_description(l_layout, (*i_ftext).m_font); 

    if ((*i_ftext).m_autowrap)
    {
      pango_layout_set_width(l_layout, PANGO_SCALE*(*i_ftext).m_len_x);
      pango_layout_set_wrap(l_layout, (*i_ftext).m_wrapmode);
    }

    cairo_translate((*i_ctx).m_cr, (*i_ftext).m_pos_x, (*i_ftext).m_pos_y);

    if (item_draw_design == (*i_ctx).m_option.m_type)
    {
      l_text= "default";
    }
    else
    {
      l_text= (*i_ftext).m_value;
    }

    if (l_text)
    {
      pango_layout_set_text(l_layout, l_text, -1);
    }

    pango_cairo_update_layout((*i_ctx).m_cr, l_layout);
    pango_cairo_show_layout((*i_ctx).m_cr, l_layout);

    cairo_restore((*i_ctx).m_cr);

    if ((*i_ftext).m_border.m_active)
    {

      item_common_cairo_dash_set((*i_ctx).m_cr, (*i_ftext).m_border.m_style);
      cairo_set_line_width((*i_ctx).m_cr, (*i_ftext).m_border.m_thickness);
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_ftext).m_border.m_color.red,
        (*i_ftext).m_border.m_color.green,
        (*i_ftext).m_border.m_color.blue,
        (*i_ftext).m_border.m_color.alpha);

      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_ftext).m_pos_x,
        (*i_ftext).m_pos_y,
        (*i_ftext).m_len_x,
        (*i_ftext).m_len_y);

      cairo_stroke((*i_ctx).m_cr);
    }

    if (0 == (*i_ctx).m_selected)
    {
      if ((*i_ctx).m_option.m_highlight &&
        (item_draw_design == (*i_ctx).m_option.m_type))
      {
        item_common_box_draw_hightlight(
          (*i_ctx).m_cr, 
          (struct geom_rect*)i_ftext,
          1.0);
      }
      break;
    }

    item_common_box_draw_grips((*i_ctx).m_cr, (struct geom_rect*)i_ftext);

  }while(0);

  if (l_font)
  {
    pango_font_description_free(l_font);
  }

  if (l_layout)
  {
    g_object_unref(l_layout);
  }

  return;
}

extern void
item_ftext_get_bbox(
  struct geom_rect*const                o_bbox,
  void const*const                      i_object)
{
  (*o_bbox)= *(struct geom_rect*)i_object;
  return;
}

extern void
item_ftext_in_event(
  enum event_type*const                 o_type,
  struct item_ftext const*const         i_ftext,
  struct item_in_event const*const      i_ctx)
{

  item_common_box_in_event(o_type, i_ctx, (struct geom_rect*)i_ftext);

  return;
}

extern void
item_ftext_move(
  struct item_ftext*const               io_ftext,
  struct geom_point const*const         i_event)
{
  item_common_box_move((struct geom_rect*)io_ftext, i_event);
  return;
}

extern void
item_ftext_normalize(
  struct item_ftext*const               io_ftext,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  item_common_box_normalize(
    (struct geom_rect*)io_ftext, 
    i_bound_width,
    i_bound_height);
  return;

}

extern int
item_ftext_read(
  GError**                              o_error,
  struct item_ftext*const               io_ftext,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_pos_x:
          (*io_ftext).m_pos_x= xml_convert_double(l_value);
          break;
        case element_tag_pos_y:
          (*io_ftext).m_pos_y= xml_convert_double(l_value);
          break;
        case element_tag_len_x:
          (*io_ftext).m_len_x= xml_convert_double(l_value);
          break;
        case element_tag_len_y:
          (*io_ftext).m_len_y= xml_convert_double(l_value);
          break;
        case element_tag_font:
          (*io_ftext).m_font= pango_font_description_from_string(l_value);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_ftext).m_color= l_rgba;
          }
          break;
        case element_tag_border:
          l_exit= item_common_read_border_attr(&l_error, &(*io_ftext).m_border, l_xml);
          break;
        case element_tag_fill:
          l_exit= item_common_read_fill(&l_error, &(*io_ftext).m_fill, l_xml);
          break;
        case element_tag_autowrap:
          l_value= xml_lookup_attribute_value(l_xml, "on");
          (*io_ftext).m_autowrap= xml_convert_boolean(l_value);
          break;
        case element_tag_wrapmode:
          (*io_ftext).m_wrapmode= item_common_get_wrapmode(l_value);
          break;
        case element_tag_tag:
          g_strlcpy((*io_ftext).m_tag, l_value, sizeof((*io_ftext).m_tag));
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_ftext_resize(
  struct item_ftext*const               io_ftext,
  struct item_resize_event const*const  i_ctx)
{

  item_common_box_resize((struct geom_rect *)io_ftext, i_ctx);

  return;
}

extern int
item_ftext_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_ftext const*const         i_ftext)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"ftext\">\n");
  fprintf(io_fp, "\t\t\t\t<pos_x>%s</pos_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_ftext).m_pos_x));
  fprintf(io_fp, "\t\t\t\t<pos_y>%s</pos_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_ftext).m_pos_y));
  fprintf(io_fp, "\t\t\t\t<len_x>%s</len_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_ftext).m_len_x));
  fprintf(io_fp, "\t\t\t\t<len_y>%s</len_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_ftext).m_len_y));
  l_text= pango_font_description_to_string((*i_ftext).m_font);
  fprintf(io_fp, "\t\t\t\t<font>%s</font>\n", l_text);
  g_free(l_text);
  l_text= gdk_rgba_to_string(&(*i_ftext).m_color);
  fprintf(io_fp, "\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  item_common_write_border_attr(o_error, io_fp, &(*i_ftext).m_border);
  item_common_write_fill(o_error, io_fp, &(*i_ftext).m_fill);
  fprintf(io_fp, "\t\t\t\t<autowrap on=\"%s\"/>\n", ((*i_ftext).m_autowrap ? "true" : "false"));
  fprintf(io_fp, "\t\t\t\t<wrapmode>%s</wrapmode>\n", item_common_get_wrapmode_text((*i_ftext).m_wrapmode));

  if ((*i_ftext).m_tag[0])
  {
    fprintf(io_fp, "\t\t\t\t<tag>%s</tag>\n", (*i_ftext).m_tag);
  }

  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
