/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_fimage.h"

extern void
item_fimage_assign(
  struct item_fimage*const              o_fimage,
  struct item_method*const              o_method)
{

  memset(o_fimage, 0, sizeof(*o_fimage));

  (*o_fimage).m_len_x= 30.0;
  (*o_fimage).m_len_y= 30.0;
  (*o_fimage).m_border.m_thickness= 1.0;
  (*o_fimage).m_border.m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, fimage);

  return;
}

extern void
item_fimage_destroy(
  struct item_fimage*const              io_fimage)
{

  if ((*io_fimage).m_value)
  {
    g_free((*io_fimage).m_value);
  }

  return;
}

extern void
item_fimage_discharge(
  struct item_fimage*const              io_fimage)
{

  item_fimage_destroy(io_fimage);
  memset(io_fimage, 0, sizeof(*io_fimage));

  return;
}

extern void
item_fimage_draw(
  struct item_fimage*const              i_fimage,
  struct item_draw_context const*const  i_ctx)
{
  int                                   l_exit;
  GdkPixbuf *                           l_pixbuf;

  l_pixbuf= 0;

  do
  {

    if ((*i_fimage).m_border.m_active)
    {
      item_common_cairo_dash_set((*i_ctx).m_cr, (*i_fimage).m_border.m_style);
      cairo_set_line_width((*i_ctx).m_cr, (*i_fimage).m_border.m_thickness);
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_fimage).m_border.m_color.red,
        (*i_fimage).m_border.m_color.green,
        (*i_fimage).m_border.m_color.blue,
        (*i_fimage).m_border.m_color.alpha);
      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_fimage).m_pos_x,
        (*i_fimage).m_pos_y,
        (*i_fimage).m_len_x,
        (*i_fimage).m_len_y);
      cairo_stroke((*i_ctx).m_cr);
    }
    else if (item_draw_design == (*i_ctx).m_option.m_type)
    {
      item_common_cairo_dash_set((*i_ctx).m_cr, line_style_dash);
      cairo_set_line_width((*i_ctx).m_cr, 1);
      cairo_set_source_rgb((*i_ctx).m_cr, 0.0, 0.0, 0.0);
      cairo_rectangle(
        (*i_ctx).m_cr,
        (*i_fimage).m_pos_x,
        (*i_fimage).m_pos_y,
        (*i_fimage).m_len_x,
        (*i_fimage).m_len_y);
      cairo_stroke((*i_ctx).m_cr);
    }

    if ((*i_fimage).m_value && (*i_fimage).m_value_size)
    {
      l_exit= pixbuf_new_from_blob(
        0,
        &l_pixbuf,
        (*i_fimage).m_value,
        (*i_fimage).m_value_size);

      if (l_exit || 0 == l_pixbuf)
      {
        break;
      }

      gdk_cairo_set_source_pixbuf(
        (*i_ctx).m_cr, 
        l_pixbuf,
        (*i_fimage).m_pos_x,
        (*i_fimage).m_pos_y);

      cairo_paint((*i_ctx).m_cr);
    }

    if (0 == (*i_ctx).m_selected)
    {
      if ((*i_ctx).m_option.m_highlight &&
        (item_draw_design == (*i_ctx).m_option.m_type))
      {
        item_common_box_draw_hightlight(
          (*i_ctx).m_cr, 
          (struct geom_rect*)i_fimage,
          (*i_fimage).m_border.m_thickness);
      }
      break;
    }

    item_common_box_draw_grips((*i_ctx).m_cr, (struct geom_rect*)i_fimage);

  }while(0);

  if (l_pixbuf)
  {
    g_object_unref(l_pixbuf);
  }

  return;
}

extern void
item_fimage_get_bbox(
  struct geom_rect*const                o_bbox,
  void const*const                      i_object)
{
  (*o_bbox)= *(struct geom_rect*)i_object;
  return;
}

extern void
item_fimage_in_event(
  enum event_type*const                 o_type,
  struct item_fimage const*const        i_fimage,
  struct item_in_event const*const i_ctx)
{

  item_common_box_in_event(o_type, i_ctx, (struct geom_rect*)i_fimage);

  return;
}

extern void
item_fimage_move(
  struct item_fimage*const              io_fimage,
  struct geom_point const*const         i_event)
{
  item_common_box_move((struct geom_rect*)io_fimage, i_event);
  return;
}

extern void
item_fimage_normalize(
  struct item_fimage*const              io_fimage,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  item_common_box_normalize(
    (struct geom_rect*)io_fimage, 
    i_bound_width,
    i_bound_height);
  return;
}

extern int
item_fimage_read(
  GError**                              o_error,
  struct item_fimage*const              io_fimage,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_pos_x:
          (*io_fimage).m_pos_x= xml_convert_double(l_value);
          break;
        case element_tag_pos_y:
          (*io_fimage).m_pos_y= xml_convert_double(l_value);
          break;
        case element_tag_len_x:
          (*io_fimage).m_len_x= xml_convert_double(l_value);
          break;
        case element_tag_len_y:
          (*io_fimage).m_len_y= xml_convert_double(l_value);
          break;
        case element_tag_border:
          l_exit= item_common_read_border_attr(&l_error, &(*io_fimage).m_border, l_xml);
          break;
        case element_tag_tag:
          g_strlcpy((*io_fimage).m_tag, l_value, sizeof((*io_fimage).m_tag));
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_fimage_resize(
  struct item_fimage*const              io_fimage,
  struct item_resize_event const*const  i_ctx)
{

  item_common_box_resize((struct geom_rect *)io_fimage, i_ctx);

  return;
}

extern int
item_fimage_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_fimage const*const        i_fimage)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;

  l_exit= 0;
  fprintf(io_fp, "\t\t\t<item name=\"fimage\">\n");
  fprintf(io_fp, "\t\t\t\t<pos_x>%s</pos_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fimage).m_pos_x));
  fprintf(io_fp, "\t\t\t\t<pos_y>%s</pos_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fimage).m_pos_y));
  fprintf(io_fp, "\t\t\t\t<len_x>%s</len_x>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fimage).m_len_x));
  fprintf(io_fp, "\t\t\t\t<len_y>%s</len_y>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_fimage).m_len_y));
  item_common_write_border_attr(o_error, io_fp, &(*i_fimage).m_border);

  if ((*i_fimage).m_tag[0])
  {
    fprintf(io_fp, "\t\t\t\t<tag>%s</tag>\n", (*i_fimage).m_tag);
  }

  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
