/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "geom_rectvect.h"

static double
geom_max(
  double const                          l_v1,
  double const                          l_v2,
  double const                          l_v3,
  double const                          l_v4)
{
  double                                l_max;

  l_max= l_v1;

  if (l_v2 > l_max) 
  {
    l_max= l_v2;
  }

  if (l_v3 > l_max) 
  {
    l_max= l_v3;
  }

  if (l_v4 > l_max) 
  {
    l_max= l_v4;
  }

  return l_max;
}

static double
geom_min(
  double const                          l_v1,
  double const                          l_v2,
  double const                          l_v3,
  double const                          l_v4)
{
  double                                l_min;

  l_min= l_v1;

  if (l_v2 < l_min) 
  {
    l_min= l_v2;
  }

  if (l_v3 < l_min)
  {
    l_min= l_v3;  
  }

  if (l_v4 < l_min)
  {
    l_min= l_v4;
  }

  return l_min;
}

extern void
geom_rectvect_copy(
  rectvect_t                            o_target,
  rectvect_t const                      i_src)
{

  o_target[0]= i_src[0];
  o_target[1]= i_src[1];
  o_target[2]= i_src[2];
  o_target[3]= i_src[3];
  o_target[4]= i_src[4];
  o_target[5]= i_src[5];
  o_target[6]= i_src[6];
  o_target[7]= i_src[7];

  return;
}

extern void
geom_rectvect_get_bounding_box(
  struct geom_rect*const                o_box,
  rectvect_t const                      i_rect)
{
  double                                l_min_x;
  double                                l_min_y;
  double                                l_max_x;
  double                                l_max_y;

  l_min_x= geom_min(i_rect[0], i_rect[2], i_rect[4], i_rect[6]);
  l_max_x= geom_max(i_rect[0], i_rect[2], i_rect[4], i_rect[6]);
  l_min_y= geom_min(i_rect[1], i_rect[3], i_rect[5], i_rect[7]);
  l_max_y= geom_max(i_rect[1], i_rect[3], i_rect[5], i_rect[7]);
  (*o_box).m_pos_x= l_min_x;
  (*o_box).m_pos_y= l_min_y;
  (*o_box).m_len_x= (l_max_x - l_min_x);
  (*o_box).m_len_y= (l_max_y - l_min_y);

  return;
}

extern double
geom_rectvect_get_rotation(
  rectvect_t const                      i_rect)
{
  double                                l_degrees;
  double                                l_delta_x;
  double                                l_delta_y;
 
  l_delta_x= i_rect[top_right_x] - i_rect[top_left_x];
  l_delta_y= i_rect[bottom_right_y] - i_rect[bottom_left_y];
  l_degrees= atan2(l_delta_y, l_delta_x) * 180 / G_PI;

  if (0 > l_degrees)
  {
    l_degrees+= 360.0;
  }

  return l_degrees;
}

extern void
geom_rectvect_grow(
  rectvect_t                            io_rect,
  double const                          i_delta)
{
  double                                l_center_x;
  double                                l_center_y;
  double                                l_delta_x;
  double                                l_delta_y;

  /*
  quadrant 1: x and y are both positive.
  quadrant 2: x is negative and y is positive.
  quadrant 3: x is negative and y is negative.
  quadrant 4: x is positive and y is negative. 
  */

  l_center_x= (io_rect[top_left_x] + io_rect[bottom_right_x]) / 2;
  l_center_y= (io_rect[top_left_y] + io_rect[bottom_right_y]) / 2;

  l_delta_x= io_rect[top_left_x] - l_center_x;
  l_delta_y= io_rect[top_left_y] - l_center_y;

  if (0.0 <= l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[top_left_x]+= i_delta;
    io_rect[top_left_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[top_left_x]-= i_delta;
    io_rect[top_left_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 > l_delta_y)
  {
    io_rect[top_left_x]-= i_delta;
    io_rect[top_left_y]-= i_delta;
  }
  else
  {
    io_rect[top_left_x]+= i_delta;
    io_rect[top_left_y]-= i_delta;
  }

  l_delta_x= io_rect[top_right_x] - l_center_x;
  l_delta_y= io_rect[top_right_y] - l_center_y;

  if (0.0 <= l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[top_right_x]+= i_delta;
    io_rect[top_right_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[top_right_x]-= i_delta;
    io_rect[top_right_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 > l_delta_y)
  {
    io_rect[top_right_x]-= i_delta;
    io_rect[top_right_y]-= i_delta;
  }
  else
  {
    io_rect[top_right_x]+= i_delta;
    io_rect[top_right_y]-= i_delta;
  }

  l_delta_x= io_rect[bottom_right_x] - l_center_x;
  l_delta_y= io_rect[bottom_right_y] - l_center_y;

  if (0.0 <= l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[bottom_right_x]+= i_delta;
    io_rect[bottom_right_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[bottom_right_x]-= i_delta;
    io_rect[bottom_right_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 > l_delta_y)
  {
    io_rect[bottom_right_x]-= i_delta;
    io_rect[bottom_right_y]-= i_delta;
  }
  else
  {
    io_rect[bottom_right_x]+= i_delta;
    io_rect[bottom_right_y]-= i_delta;
  }

  l_delta_x= io_rect[bottom_left_x] - l_center_x;
  l_delta_y= io_rect[bottom_left_y] - l_center_y;

  if (0.0 <= l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[bottom_left_x]+= i_delta;
    io_rect[bottom_left_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 <= l_delta_y)
  {
    io_rect[bottom_left_x]-= i_delta;
    io_rect[bottom_left_y]+= i_delta;
  }
  else if (0.0 > l_delta_x && 0.0 > l_delta_y)
  {
    io_rect[bottom_left_x]-= i_delta;
    io_rect[bottom_left_y]-= i_delta;
  }
  else
  {
    io_rect[bottom_left_x]+= i_delta;
    io_rect[bottom_left_y]-= i_delta;
  }

  return;
}

extern void
geom_rectvect_rotate(
  rectvect_t                            io_rect,
  double const                          i_degrees)
{
  double                                l_radians;
  double                                l_adjust_x;
  double                                l_adjust_y;
  double                                l_cos;
  rectvect_t                            l_origin;
  rectvect_t                            l_prime;
  double                                l_sin;

  l_adjust_x= io_rect[0];
  l_adjust_y= io_rect[1];

  l_origin[0]= io_rect[0] - l_adjust_x;
  l_origin[1]= io_rect[1] - l_adjust_y;
  l_origin[2]= io_rect[2] - l_adjust_x;
  l_origin[3]= io_rect[3] - l_adjust_y;
  l_origin[4]= io_rect[4] - l_adjust_x;
  l_origin[5]= io_rect[5] - l_adjust_y;
  l_origin[6]= io_rect[6] - l_adjust_x;
  l_origin[7]= io_rect[7] - l_adjust_y;

  l_radians= geom_deg2rad(i_degrees);
  l_cos= cos(l_radians);
  l_sin= sin(l_radians);

  l_prime[0]= (l_origin[0] * l_cos) - (l_origin[1] * l_sin);
  l_prime[1]= (l_origin[0] * l_sin) + (l_origin[1] * l_cos);
  l_prime[2]= (l_origin[2] * l_cos) - (l_origin[3] * l_sin);
  l_prime[3]= (l_origin[2] * l_sin) + (l_origin[3] * l_cos);
  l_prime[4]= (l_origin[4] * l_cos) - (l_origin[5] * l_sin);
  l_prime[5]= (l_origin[4] * l_sin) + (l_origin[5] * l_cos);
  l_prime[6]= (l_origin[6] * l_cos) - (l_origin[7] * l_sin);
  l_prime[7]= (l_origin[6] * l_sin) + (l_origin[7] * l_cos);

  io_rect[0]= l_prime[0] + l_adjust_x;
  io_rect[1]= l_prime[1] + l_adjust_y;
  io_rect[2]= l_prime[2] + l_adjust_x;
  io_rect[3]= l_prime[3] + l_adjust_y;
  io_rect[4]= l_prime[4] + l_adjust_x;
  io_rect[5]= l_prime[5] + l_adjust_y;
  io_rect[6]= l_prime[6] + l_adjust_x;
  io_rect[7]= l_prime[7] + l_adjust_y;

  return;
}

extern void
geom_rectvect_make_square(
  rectvect_t                            i_rect)
{

  i_rect[bottom_left_x]= i_rect[top_left_x];
  i_rect[bottom_right_x]= i_rect[top_right_x];
  i_rect[bottom_left_y]= i_rect[bottom_right_y];
  i_rect[top_right_y]= i_rect[top_left_y];

  return;
}
