/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "barsym_code39.h"

enum ascii_ctl
{
  NUL = 0x00,
  SOH,
  STX,
  ETX,
  EOT,
  ENQ,
  ACK,
  BEL,
  BS,
  HT,
  LF,
  VT,
  FF,
  CR,
  SO,
  SI,
  DLE,
  DC1,
  DC2,
  DC3,
  DC4,
  NAK,
  SYN,
  ETB,
  CAN,
  EM,
  SUB,
  ESC,
  FS,
  GS,
  RS,
  YS,
  DEL=0x7F
};

#define SPLICE(a,b) a b

#define SHIFT_DOLLAR                    "100100100101" /* $ */
#define SHIFT_SLASH                     "100100101001" /* / */
#define SHIFT_PLUS                      "100101001001" /* + */
#define SHIFT_PERCENT                   "101001001001" /* % */

static char const*
barsym_code39_encode_ext(
  unsigned char const                   i_code)
{
  char const*                           l_encode;

  switch(i_code)
  {
  case NUL:
    /* %U */
    l_encode= SPLICE(SHIFT_PERCENT, "110010101011");
    break;
  case SOH:
    /* $A */
    l_encode= SPLICE(SHIFT_DOLLAR, "110101001011");
    break;
  case STX:
    /* $B */
    l_encode= SPLICE(SHIFT_DOLLAR, "101101001011");
    break;
  case ETX:
    /* $C */
    l_encode= SPLICE(SHIFT_DOLLAR, "110110100101");
    break;
  case EOT:
    /* $D */
    l_encode= SPLICE(SHIFT_DOLLAR, "101011001011");
    break;
  case ENQ:
    /* $E */
    l_encode= SPLICE(SHIFT_DOLLAR, "110101100101");
    break;
  case ACK:
    /* $F */
    l_encode= SPLICE(SHIFT_DOLLAR, "101101100101");
    break;
  case BEL:
    /* $G */
    l_encode= SPLICE(SHIFT_DOLLAR, "101010011011");
    break;
  case BS:
    /* $H */
    l_encode= SPLICE(SHIFT_DOLLAR, "110101001101");
    break;
  case HT:
    /* $I */
    l_encode= SPLICE(SHIFT_DOLLAR, "101101001101");
    break;
  case LF:
    /* $J */
    l_encode= SPLICE(SHIFT_DOLLAR, "101011001101");
    break;
  case VT:
    /* $K */
    l_encode= SPLICE(SHIFT_DOLLAR, "110101010011");
    break;
  case FF:
    /* $L */
    l_encode= SPLICE(SHIFT_DOLLAR, "101101010011");
    break;
  case CR:
    /* $M */
    l_encode= SPLICE(SHIFT_DOLLAR, "110110101001");
    break;
  case SO:
    /* $N */
    l_encode= SPLICE(SHIFT_DOLLAR, "101011010011");
    break;
  case SI:
    /* $O */
    l_encode= SPLICE(SHIFT_DOLLAR, "110101101001");
    break;
  case DLE:
    /* $P */
    l_encode= SPLICE(SHIFT_DOLLAR, "101101101001");
    break;
  case DC1:
    /* $Q */
    l_encode= SPLICE(SHIFT_DOLLAR, "101010110011");
    break;
  case DC2:
    /* $R */
    l_encode= SPLICE(SHIFT_DOLLAR, "110101011001");
    break;
  case DC3:
    /* $S */
    l_encode= SPLICE(SHIFT_DOLLAR, "101101011001"); 
    break;
  case DC4:
    /* $T */
    l_encode= SPLICE(SHIFT_DOLLAR, "101011011001");
    break;
  case NAK:
    /* $U */
    l_encode= SPLICE(SHIFT_DOLLAR, "110010101011");
    break;
  case SYN:
    /* $V */
    l_encode= SPLICE(SHIFT_DOLLAR, "100110101011");
    break;
  case ETB:
    /* $W */
    l_encode= SPLICE(SHIFT_DOLLAR, "110011010101");
    break;
  case CAN:
    /* $X */
    l_encode= SPLICE(SHIFT_DOLLAR, "100101101011");
    break;
  case EM:
    /* $Y */
    l_encode= SPLICE(SHIFT_DOLLAR, "110010110101");
    break;
  case SUB:
    /* $Z */
    l_encode= SPLICE(SHIFT_DOLLAR, "100110110101");
    break;
  case ESC:
    /* %A */
    l_encode= SPLICE(SHIFT_PERCENT, "110101001011");
    break;
  case FS:
    /* %B */
    l_encode= SPLICE(SHIFT_PERCENT, "101101001011");
    break;
  case GS:
    /* %C */
    l_encode= SPLICE(SHIFT_PERCENT, "110110100101");
    break;
  case RS:
    /* %D */
    l_encode= SPLICE(SHIFT_PERCENT, "101011001011");
    break;
  case YS:
    /* %E */
    l_encode= SPLICE(SHIFT_PERCENT, "110101100101");
    break;
  case '!':
    /* /A */
    l_encode= SPLICE(SHIFT_SLASH, "110101001011");
    break;
  case '"': 	
    /* /B */
    l_encode= SPLICE(SHIFT_SLASH, "101101001011");
    break;
  case '#':
    /* /C */
    l_encode= SPLICE(SHIFT_SLASH, "110110100101");
    break;
  case '$':
    /* /D */
    l_encode= SPLICE(SHIFT_SLASH, "101011001011");
    break;
  case '%':
    /* /E */
    l_encode= SPLICE(SHIFT_SLASH, "110101100101");
    break;
  case '&':
    /* /F */
    l_encode= SPLICE(SHIFT_SLASH, "101101100101");
    break;
  case '\'':
    /* /G */
    l_encode= SPLICE(SHIFT_SLASH, "101010011011");
    break;
  case '(':
    /* /H */
    l_encode= SPLICE(SHIFT_SLASH, "110101001101");
    break;
  case ')':
    /* /I */
    l_encode= SPLICE(SHIFT_SLASH, "101101001101");
    break;
  case '*':
    /* /J */
    l_encode= SPLICE(SHIFT_SLASH, "101011001101");
    break;
  case '+':
    /* /K */
    l_encode= SPLICE(SHIFT_SLASH, "110101010011");
    break;
  case ',':
    /* /L */
    l_encode= SPLICE(SHIFT_SLASH, "101101010011");
    break;
  case '/':
    /* /O */
    l_encode= SPLICE(SHIFT_SLASH, "110101101001");
    break;
  case ':':
    /* /Z */
    l_encode= SPLICE(SHIFT_SLASH, "100110110101");
    break;
  case ';':
    /* %F */
    l_encode= SPLICE(SHIFT_PERCENT, "101101100101");
    break;
  case '<':
    /* %G */
    l_encode= SPLICE(SHIFT_PERCENT, "101010011011");
    break;
  case '=':
    /* %H */
    l_encode= SPLICE(SHIFT_PERCENT, "110101001101");
    break;
  case '>':
    /* %I */
    l_encode= SPLICE(SHIFT_PERCENT, "101101001101");
    break;
  case '?':
    /* %J */
    l_encode= SPLICE(SHIFT_PERCENT, "101011001101");
    break;
  case '@':
    /* %V */
    l_encode= SPLICE(SHIFT_PERCENT, "100110101011");
    break;
  case '[':
    /* %K */
    l_encode= SPLICE(SHIFT_PERCENT, "110101010011");
    break;
  case '\\':
    /* %L */
    l_encode= SPLICE(SHIFT_PERCENT, "101101010011");
    break;
  case ']':
    /* %M */
    l_encode= SPLICE(SHIFT_PERCENT, "110110101001");
    break;
  case '^':
    /* %N */
    l_encode= SPLICE(SHIFT_PERCENT, "101011010011");
    break;
  case '_':
    /* %O */
    l_encode= SPLICE(SHIFT_PERCENT, "110101101001");
    break;
  case '`':
    /* %W */
    l_encode= SPLICE(SHIFT_PERCENT, "110011010101");
    break;
  case 'a':
    /* +A */
    l_encode= SPLICE(SHIFT_PLUS, "110101001011");
    break;
  case 'b':
    /* +B */
    l_encode= SPLICE(SHIFT_PLUS, "101101001011");
    break;
  case 'c':
    /* +C */
    l_encode= SPLICE(SHIFT_PLUS, "110110100101");
    break;
  case 'd':
    /* +D */
    l_encode= SPLICE(SHIFT_PLUS, "101011001011");
    break;
  case 'e':
    /* +E */
    l_encode= SPLICE(SHIFT_PLUS, "110101100101");
    break;
  case 'f':
    /* +F */
    l_encode= SPLICE(SHIFT_PLUS, "101101100101");
    break;
  case 'g':
    /* +G */
    l_encode= SPLICE(SHIFT_PLUS, "101010011011");
    break;
  case 'h':
    /* +H */
    l_encode= SPLICE(SHIFT_PLUS, "110101001101");
    break;
  case 'i':
    /* +I */
    l_encode= SPLICE(SHIFT_PLUS, "101101001101");
    break;
  case 'j':
    /* +J */
    l_encode= SPLICE(SHIFT_PLUS, "101011001101");
    break;
  case 'k':
    /* +K */
    l_encode= SPLICE(SHIFT_PLUS, "110101010011");
    break;
  case 'l':
    /* +L */
    l_encode= SPLICE(SHIFT_PLUS, "101101010011");
    break;
  case 'm':
    /* +M */
    l_encode= SPLICE(SHIFT_PLUS, "110110101001");
    break;
  case 'n':
    /* +N */
    l_encode= SPLICE(SHIFT_PLUS, "101011010011");
    break;
  case 'o':
    /* +O */
    l_encode= SPLICE(SHIFT_PLUS, "110101101001");
    break;
  case 'p':
    /* +P */
    l_encode= SPLICE(SHIFT_PLUS, "101101101001");
    break;
  case 'q':
    /* +Q */
    l_encode= SPLICE(SHIFT_PLUS, "101010110011");
    break;
  case 'r':
    /* +R */
    l_encode= SPLICE(SHIFT_PLUS, "110101011001");
    break;
  case 's':
    /* +S */
    l_encode= SPLICE(SHIFT_PLUS, "101101011001");
    break;
  case 't':
    /* +T */
    l_encode= SPLICE(SHIFT_PLUS, "101011011001");
    break;
  case 'u':
    /* +U */
    l_encode= SPLICE(SHIFT_PLUS, "110010101011");
    break;
  case 'v':
    /* +V */
    l_encode= SPLICE(SHIFT_PLUS, "100110101011");
    break;
  case 'w':
    /* +W */
    l_encode= SPLICE(SHIFT_PLUS, "110011010101");
    break;
  case 'x':
    /* +X */
    l_encode= SPLICE(SHIFT_PLUS, "100101101011");
    break;
  case 'y':
    /* +Y */
    l_encode= SPLICE(SHIFT_PLUS, "110010110101");
    break;
  case 'z':
    /* +Z */
    l_encode= SPLICE(SHIFT_PLUS, "100110110101");
    break;
  case '{':
    /* %P */
    l_encode= SPLICE(SHIFT_PERCENT, "101101101001");
    break;
  case '|':
    /* %Q */
    l_encode= SPLICE(SHIFT_PERCENT, "101010110011");
    break;
  case '}':
    /* %R */
    l_encode= SPLICE(SHIFT_PERCENT, "110101011001");
    break;
  case '~':
    /* %S */
    l_encode= SPLICE(SHIFT_PERCENT, "101101011001");
    break;
  case DEL:
    /* %T, %X, %Y, %Z */
    l_encode= SPLICE(SHIFT_PERCENT, "101011011001") 
              SPLICE(SHIFT_PERCENT, "100101101011")
              SPLICE(SHIFT_PERCENT, "110010110101")
              SPLICE(SHIFT_PERCENT, "100110110101");
    break;
  default:
    l_encode= 0;
    break;
  }

  return l_encode;
}

extern char const*
barsym_code39_encode(
  unsigned char const                   i_code)
{
  char const*                           l_encode;

  switch(i_code)
  {
    case '0':
      l_encode= "101001101101";
      break;
    case '1':
      l_encode= "110100101011";
      break;
    case '2':
      l_encode= "101100101011";
      break;
    case '3':
      l_encode= "110110010101";
      break;
    case '4':
      l_encode= "101001101011";
      break;
    case '5':
      l_encode= "110100110101";
      break;
    case '6':
      l_encode= "101100110101";
      break;
    case '7':
      l_encode= "101001011011";
      break;
    case '8':
      l_encode= "110100101101";
      break;
    case '9':
      l_encode= "101100101101";
      break;
    case 'A':
      l_encode= "110101001011";
      break;
    case 'B':
      l_encode= "101101001011";
      break;
    case 'C':
      l_encode= "110110100101";
      break;
    case 'D':
      l_encode= "101011001011";
      break;
    case 'E':
      l_encode= "110101100101";
      break;
    case 'F':
      l_encode= "101101100101";
      break;
    case 'G':
      l_encode= "101010011011";
      break;
    case 'H':
      l_encode= "110101001101";
      break;
    case 'I':
      l_encode= "101101001101";
      break;
    case 'J':
      l_encode= "101011001101";
      break;
    case 'K':
      l_encode= "110101010011";
      break;
    case 'L':
      l_encode= "101101010011";
      break;
    case 'M':
      l_encode= "110110101001";
      break;
    case 'N':
      l_encode= "101011010011";
      break;
    case 'O':
      l_encode= "110101101001";
      break;
    case 'P':
      l_encode= "101101101001";
      break;
    case 'Q':
      l_encode= "101010110011";
      break;
    case 'R':
      l_encode= "110101011001";
      break;
    case 'S':
      l_encode= "101101011001";
      break;
    case 'T':
      l_encode= "101011011001";
      break;
    case 'U':
      l_encode= "110010101011";
      break;
    case 'V':
      l_encode= "100110101011";
      break;
    case 'W':
      l_encode= "110011010101";
      break;
    case 'X':
      l_encode= "100101101011";
      break;
    case 'Y':
      l_encode= "110010110101";
      break;
    case 'Z':
      l_encode= "100110110101";
      break;
    case '-':
      l_encode= "100101011011";
      break;
    case '.':
      l_encode= "110010101101";
      break;
    case ' ':
      l_encode= "100110101101";
      break;
    default:
      l_encode= barsym_code39_encode_ext(i_code);
      break;
  }

  return l_encode;
}
