/*  zint.h - definitions for libzint

    libzint - the open source barcode library
    Copyright (C) 2009 Robin Stuart <robin@zint.org.uk>
    Copyright (c) 2016 iwrite authors

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright 
       notice, this list of conditions and the following disclaimer.  
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.  
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission. 

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
    SUCH DAMAGE.
*/

#ifndef ZINT_H
#define ZINT_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <cairo.h>

struct zint_render_line {
	float x, y, length, width;
	struct zint_render_line *next;      /* Pointer to next line */
};

struct zint_render_string {
	float x, y, fsize;
	float width;                        /* Suggested string width, may be 0 if none recommended */
	int length;
	unsigned char *text;
	struct zint_render_string *next;    /* Pointer to next character */
};

struct zint_render_ring {
	float x, y, radius, line_width;
	struct zint_render_ring *next;      /* Pointer to next ring */
};

struct zint_render_hexagon {
	float x, y;
	struct zint_render_hexagon *next;   /* Pointer to next hexagon */
};

struct zint_render {
	float width, height;
	struct zint_render_line *lines;	 	/* Pointer to first line */
	struct zint_render_string *strings;	/* Pointer to first string */
	struct zint_render_ring *rings;         /* Pointer to first ring */
	struct zint_render_hexagon *hexagons;   /* Pointer to first hexagon */
};

enum symbology_type
{
   BARCODE_UNKOWN=0,
   /*Tbarcode 7codes*/
   BARCODE_CODE11=1,
   BARCODE_C25MATRIX=2,
   BARCODE_C25INTER=3,
   BARCODE_C25IATA=4,
   BARCODE_C25LOGIC=6,
   BARCODE_C25IND=7,
   BARCODE_CODE39=8,
   BARCODE_EXCODE39=9,
   BARCODE_EANX=13,
   BARCODE_EAN128=16,
   BARCODE_CODABAR=18,
   BARCODE_CODE128=20,
   BARCODE_DPLEIT=21,
   BARCODE_DPIDENT=22,
   BARCODE_CODE16K=23,
   BARCODE_CODE49=24,
   BARCODE_CODE93=25,
   BARCODE_FLAT=28,
   BARCODE_RSS14=29,
   BARCODE_RSS_LTD=30,
   BARCODE_RSS_EXP=31,
   BARCODE_TELEPEN=32,
   BARCODE_UPCA=34,
   BARCODE_UPCE=37,
   BARCODE_POSTNET=40,
   BARCODE_MSI_PLESSEY=47,
   BARCODE_FIM=49,
   BARCODE_LOGMARS=50,
   BARCODE_PHARMA=51,
   BARCODE_PZN=52,
   BARCODE_PHARMA_TWO=53,
   BARCODE_PDF417=55,
   BARCODE_PDF417TRUNC=56,
   BARCODE_MAXICODE=57,
   BARCODE_QRCODE=58,
   BARCODE_CODE128B=60,
   BARCODE_AUSPOST=63,
   BARCODE_AUSREPLY=66,
   BARCODE_AUSROUTE=67,
   BARCODE_AUSREDIRECT=68,
   BARCODE_ISBNX=69,
   BARCODE_RM4SCC=70,
   BARCODE_DATAMATRIX=71,
   BARCODE_EAN14=72,
   BARCODE_CODABLOCKF=74,
   BARCODE_NVE18=75,
   BARCODE_JAPANPOST=76,
   BARCODE_KOREAPOST=77,
   BARCODE_RSS14STACK=79,
   BARCODE_RSS14STACK_OMNI=80,
   BARCODE_RSS_EXPSTACK=81,
   BARCODE_PLANET=82,
   BARCODE_MICROPDF417=84,
   BARCODE_ONECODE=85,
   BARCODE_PLESSEY=86,
   
   /*Tbarcode 8codes*/
   BARCODE_TELEPEN_NUM=87,
   BARCODE_ITF14=89,
   BARCODE_KIX=90,
   BARCODE_AZTEC=92,
   BARCODE_DAFT=93,
   BARCODE_MICROQR=97,
   
   /*Tbarcode 9codes*/
   BARCODE_HIBC_128=98,
   BARCODE_HIBC_39=99,
   BARCODE_HIBC_DM=102,
   BARCODE_HIBC_QR=104,
   BARCODE_HIBC_PDF=106,
   BARCODE_HIBC_MICPDF=108,
   BARCODE_HIBC_BLOCKF=110,
   BARCODE_HIBC_AZTEC=112,
   
   /*Zint specific*/
   BARCODE_AZRUNE=128,
   BARCODE_CODE32=129,
   BARCODE_EANX_CC=130,
   BARCODE_EAN128_CC=131,
   BARCODE_RSS14_CC=132,
   BARCODE_RSS_LTD_CC=133,
   BARCODE_RSS_EXP_CC=134,
   BARCODE_UPCA_CC=135,
   BARCODE_UPCE_CC=136,
   BARCODE_RSS14STACK_CC=137,
   BARCODE_RSS14_OMNI_CC=138,
   BARCODE_RSS_EXPSTACK_CC=139,
   BARCODE_CHANNEL=140,
   BARCODE_CODEONE=141,
   BARCODE_GRIDMATRIX=142,
};

struct zint_symbol {
	enum symbology_type symbology;
	int height;
	int whitespace_width;
	int border_width;
	int output_options;
	char fgcolour[10];
	char bgcolour[10];
	char outfile[256];
	float scale;
	int option_1;
	int option_2;
	int option_3;
	int show_hrt;
	int input_mode;
	unsigned char text[128];
	int rows;
	int width;
	char primary[128];
	unsigned char encoded_data[178][143];
	int row_height[178]; /* Largest symbol is 177x177 QR Code */
	char errtxt[100];
	char *bitmap;
	int bitmap_width;
	int bitmap_height;
	struct zint_render *rendered;
};

#define BARCODE_NO_ASCII	1
#define BARCODE_BIND		2
#define BARCODE_BOX		4
#define BARCODE_STDOUT		8
#define READER_INIT		16
#define SMALL_TEXT		32

#define DATA_MODE	0
#define UNICODE_MODE	1
#define GS1_MODE	2
#define KANJI_MODE	3
#define SJIS_MODE	4

#define DM_SQUARE	100

#define WARN_INVALID_OPTION	2
#define ERROR_TOO_LONG		5
#define ERROR_INVALID_DATA	6
#define ERROR_INVALID_CHECK	7
#define ERROR_INVALID_OPTION	8
#define ERROR_ENCODING_PROBLEM	9
#define ERROR_FILE_ACCESS	10
#define ERROR_MEMORY		11

#if defined(__WIN32__) || defined(_WIN32) || defined(WIN32) || defined(_MSC_VER)
#  if defined (DLL_EXPORT) || defined(PIC) || defined(_USRDLL)
# 	 define ZINT_EXTERN __declspec(dllexport)
#  elif defined(ZINT_DLL)
#	 define ZINT_EXTERN __declspec(dllimport)
#  else
#    define ZINT_EXTERN extern
#  endif
#else
#  define ZINT_EXTERN extern	
#endif

ZINT_EXTERN struct zint_symbol *ZBarcode_Create(void);
ZINT_EXTERN void ZBarcode_Clear(struct zint_symbol *symbol);
ZINT_EXTERN void ZBarcode_Delete(struct zint_symbol *symbol);

ZINT_EXTERN int ZBarcode_Encode(struct zint_symbol *symbol, unsigned char *input, int length);
ZINT_EXTERN int ZBarcode_Encode_File(struct zint_symbol *symbol, char *filename);
ZINT_EXTERN int ZBarcode_Print(struct zint_symbol *symbol, int rotate_angle);
ZINT_EXTERN int ZBarcode_Encode_and_Print(struct zint_symbol *symbol, unsigned char *input, int length, int rotate_angle);
ZINT_EXTERN int ZBarcode_Encode_File_and_Print(struct zint_symbol *symbol, char *filename, int rotate_angle);

ZINT_EXTERN int ZBarcode_Render(struct zint_symbol *symbol, float width, float height);

ZINT_EXTERN int ZBarcode_Buffer(struct zint_symbol *symbol, int rotate_angle);
ZINT_EXTERN int ZBarcode_Encode_and_Buffer(struct zint_symbol *symbol, unsigned char *input, int length, int rotate_angle);
ZINT_EXTERN int ZBarcode_Encode_File_and_Buffer(struct zint_symbol *symbol, char *filename, int rotate_angle);

ZINT_EXTERN int ZBarcode_ValidID(int symbol_id);

ZINT_EXTERN int ZBarcode_Print_Cairo(struct zint_symbol *symbol, cairo_t *cr);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* ZINT_H */
