/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "symbology.h"

struct symbology_table                  g_symtab[]= {
  {1, "Code 11"},
  {2, "Standard 2of5"},
  {3, "Interleaved 2of5"},
  {4, "IATA 2of5"},
  {6, "Data Logic"},
  {7, "Industrial 2of5"},
  {8, "Code 39"},
  {9, "Extended Code 39"},
  {13, "EAN"},
  {16, "GS1-128"},
  {18, "Codabar"},
  {20, "Code 128"},
  {21, "Leitcode"},
  {22, "Identcode"},
  {23, "Code 16k"},
  {24, "Code 49"},
  {25, "Code 93"},
  {28, "Flattermarken"},
  {29, "Databar-14 "},
  {30, "Databar Limited"},
  {31, "Databar Extended"},
  {32, "Telepen Alpha"},
  {34, "UPC-A"},
  {37, "UPC-E"},
/*
  obsolete
  {40, "Postnet"},
*/
  {47, "MSI Plessey"},
  {49, "FIM"},
  {50, "Logmars"},
  {51, "Pharma One-Track"},
  {52, "PZN"},
  {53, "Pharma Two-Track"},
  {55, "PDF417"},
  {56, "PDF417 Trunc"},
  {57, "Maxicode"},
  {58, "QR Code"},
  {60, "Code 128-B"},
  {63, "AP Standard Customer"},
  {66, "AP Reply Paid"},
  {67, "AP Routing"},
  {68, "AP Redirection"},
  {69, "ISBN"},
  {70, "RM4SCC"},
  {71, "Data Matrix"},
  {72, "EAN-14"},
  {75, "NVE-18"},
  {76, "Japanese Post"},
  {77, "Korea Post"},
  {79, "Databar-14 Stack"},
  {80, "Databar-14 Stack Omni"},
  {81, "Databar Extended Stack"},
/*
obsolete 
  {82, "Planet"},
*/
  {84, "MicroPDF"},
  {85, "USPS OneCode"},
  {86, "UK Plessey"},
  {87, "Telepen Numeric"},
  {89, "ITF-14"},
  {90, "KIX Code"},
  {92, "Aztec Code"},
  {93, "DAFT Code"},
  {97, "Micro QR Code"},
  {98, "HIBC Code 128"},
  {99, "HIBC Code 39"},
  {102, "HIBC Data Matrix"},
  {104, "HIBC QR Code"},
  {106, "HIBC PDF417"},
  {108, "HIBC MicroPDF417"},
  {112, "HIBC Aztec Code"},
  {128, "Aztec Runes"},
  {129, "Code 23"},
/*
iwrite - not supported
  {130, "Comp EAN"},
  {131, "Comp GS1-128"},
  {132, "Comp Databar-14"},
  {133, "Comp Databar Ltd"},
  {134, "Comp Databar Ext"},
  {135, "Comp UPC-A"},
  {136, "Comp UPC-E"},
  {137, "Comp Databar-14 Stack"},
  {138, "Comp Databar Stack Omni"},
  {139, "Comp Databar Ext Stack"},
*/
  {140, "Channel Code"},
  {141, "Code One"},
  {142, "Grid Matrix"},
  {0,  0}
};

static int
symbology_encode_method_png(
  cairo_t*                              io_cr,
  struct symbology const*               i_symbology,
  double const                          i_pos_x,
  double const                          i_pos_y,
  unsigned char const*                  i_buffer,
  unsigned const                        i_length)
{
  int                                   l_exit;
  int                                   l_fd;
  char*                                 l_filename;
  GError*                               l_error;
  struct zint_symbol*                   l_bc;
  GdkPixbuf*                            l_pixbuf;

  l_fd= -1;
  l_error= 0;
  l_filename= 0;
  l_bc= 0;

  do
  {

    l_fd= g_file_open_tmp("iwrite_XXXXXX.png", &l_filename, &l_error);

    if (-1 == l_fd)
    {
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_bc= ZBarcode_Create();
    (*l_bc).symbology= (*i_symbology).m_barcode;
    (*l_bc).scale= (*i_symbology).m_scale;
    strcpy((*l_bc).outfile, l_filename);

    l_exit= ZBarcode_Encode(l_bc, (unsigned char*)i_buffer, i_length);

    if (l_exit)
    {
      break;
    }

    /* ZBarcode_Buffer - bmp is not displaying correctly need to debug */
    /* using png method instead */
    l_exit= ZBarcode_Print(l_bc, 0.0);

    if (l_exit)
    {
      break;
    }

    l_pixbuf= gdk_pixbuf_new_from_file(l_filename, &l_error);

    if (l_error)
    {
      l_exit= -1;
      break;
    }

    gdk_cairo_set_source_pixbuf(io_cr, l_pixbuf, i_pos_x, i_pos_y);
    cairo_paint(io_cr);

  }while(0);

  if (-1 != l_fd)
  {
    close(l_fd);
  }

  if (l_filename)
  {
    g_unlink(l_filename);
    g_free(l_filename);
  }

  if (l_bc)
  {
    ZBarcode_Delete(l_bc);
  }

  if (l_pixbuf)
  {
    g_object_unref(l_pixbuf);
  }

  g_clear_error(&l_error);

  return l_exit;
}

extern int
symbology_encode(
  cairo_t*                              io_cr,
  struct symbology const*               i_symbology,
  double const                          i_pos_x,
  double const                          i_pos_y,
  unsigned char const*                  i_buffer,
  unsigned const                        i_length)
{
  int                                   l_exit;
  struct zint_symbol*                   l_bc;

  do
  {

    if (BARCODE_MAXICODE == (*i_symbology).m_barcode)
    {
      l_exit= symbology_encode_method_png(
        io_cr,
        i_symbology,
        i_pos_x,
        i_pos_y,
        i_buffer,
        i_length);
      break;
    }

    l_bc= ZBarcode_Create();
    (*l_bc).symbology= (*i_symbology).m_barcode;
    (*l_bc).scale= (*i_symbology).m_scale;

    l_exit= ZBarcode_Encode(l_bc, (unsigned char*)i_buffer, i_length);

    if (l_exit)
    {
      break;
    }

    cairo_save(io_cr);
    cairo_translate(io_cr, i_pos_x, i_pos_y);
    l_exit= ZBarcode_Print_Cairo(l_bc, io_cr);
    cairo_restore(io_cr);

  }while(0);

  if (l_bc)
  {
    ZBarcode_Delete(l_bc);
  }

  return l_exit;
}

extern int
symbology_lookup_by_type(
  enum symbology_type                   i_type)
{
  unsigned                              l_slot;
  int                                   l_exit;

  l_exit= -1;

  do
  {

    if (BARCODE_UNKOWN == i_type)
    {
      break;
    }

    l_slot= 0;

    do
    {

      if (0 == g_symtab[l_slot].m_name)
      {
        break;
      }

      if (i_type == g_symtab[l_slot].m_type)
      {
        l_exit= l_slot;
        break;
      }

      l_slot++;

    }while(1);

  }while(0);

  return l_exit;
}

extern int
symbology_lookup_by_name(
  char const*                           i_name)
{
  unsigned                              l_slot;
  int                                   l_exit;

  l_exit= -1;

  l_slot= 0;

  do
  {

    if (0 == g_symtab[l_slot].m_name)
    {
      break;
    }

    if (0 == strcmp(i_name, g_symtab[l_slot].m_name))
    {
      l_exit= l_slot;
      break;
    }

    l_slot++;

  }while(1);

  return l_exit;
}
