/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "report.h"

extern void
report_assign(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_unassigned;
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_assign_export(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_export;
  export_assign(&(*o_report).m_object.m_export);
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_assign_filename(
  struct report*const                   io_report,
  char const*                           i_reportfile)
{
  FILE*                                 l_fp;
  char*                                 l_ptr;
  unsigned                              l_size;
  char*                                 l_sqlfile;

  if ((*io_report).m_reportfile)
  {
    g_free((*io_report).m_reportfile);
  }

  (*io_report).m_reportfile= g_strdup(i_reportfile);

  l_fp= g_fopen((*io_report).m_reportfile, "a+");

  if (l_fp)
  {
    fclose(l_fp);
  }

  l_size= strlen((*io_report).m_reportfile);
  l_sqlfile= (char*)g_malloc0(5+l_size);
  strcpy(l_sqlfile, (*io_report).m_reportfile);

  l_ptr= strrchr(l_sqlfile, '.'); 

  if (0 == l_ptr)
  {
    l_ptr= &l_sqlfile[l_size];
  }

  strcpy(l_ptr, ".sql");

  l_fp= g_fopen(l_sqlfile, "a+");

  if (l_fp)
  {
    fclose(l_fp);
  }

  (*io_report).m_query.m_sqlfile= l_sqlfile;

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_assign_filename(
        &(*io_report).m_object.m_layout, 
        (*io_report).m_reportfile);
      break;
    case style_export:
      export_assign_filename(
        &(*io_report).m_object.m_export, 
        (*io_report).m_reportfile);
      break;
    case style_label:
      label_assign_filename(
        &(*io_report).m_object.m_label, 
        (*io_report).m_reportfile);
    default:
      break;
  }

  return;
}

extern void
report_assign_label(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_label;
  label_assign(&(*o_report).m_object.m_label);
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_assign_layout(
  struct report*const                   o_report)
{

  memset(o_report, 0, sizeof(*o_report));
  (*o_report).m_style= style_layout;
  layout_assign(&(*o_report).m_object.m_layout);
  database_aspect_assign(&(*o_report).m_aspect);
  query_assign(&(*o_report).m_query);

  return;
}

extern void
report_discharge(
  struct report*const                   io_report)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_discharge(&(*io_report).m_object.m_layout);
      break;
    case style_export:
      export_discharge(&(*io_report).m_object.m_export);
      break;
    case style_label:
      label_discharge(&(*io_report).m_object.m_label);
    default:
      break;
  }

  database_aspect_discharge(&(*io_report).m_aspect);
  query_discharge(&(*io_report).m_query);

  if ((*io_report).m_reportfile)
  {
    g_free((*io_report).m_reportfile);
  }

  memset(io_report, 0, sizeof(*io_report));

  return;
}

extern GtkWidget*
report_gui_new(
  struct report*const                   io_report)
{
  GtkWidget*                            l_widget;

  switch((*io_report).m_style)
  {
    case style_layout:
      l_widget= layout_gui_new(&(*io_report).m_object.m_layout);
      break;
    case style_export:
      l_widget= export_gui_new(&(*io_report).m_object.m_export);
      break;
    case style_label:
      l_widget= label_gui_new(&(*io_report).m_object.m_label);
      break;
    default:
      break;
  }

  return l_widget;
}

extern void
report_highlight_objects(
  struct report*const                   io_report,
  gboolean const                        i_state)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_highlight_objects(&(*io_report).m_object.m_layout, i_state);
      break;
    case style_label:
      label_highlight_objects(&(*io_report).m_object.m_label, i_state);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

extern struct item_node*
report_item_new(
  struct report*const                   io_report,
  enum item_type const                  i_type)
{
  struct item_node*                     l_node;

  l_node= 0;

  switch((*io_report).m_style)
  {
    case style_layout:
      l_node= layout_item_new(&(*io_report).m_object.m_layout, i_type);
      break;
    case style_label:
      l_node= label_item_new(&(*io_report).m_object.m_label, i_type);
      break;
    default:
      /* programming error */
      break;
  }

  return l_node;
}

extern int
report_read(
  GError**                              o_error,
  GtkWidget**                           o_widget,
  struct report*const                   io_report)
{
  struct bson_document                  l_document;
  struct bson_node *                    l_element;
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;

  (*o_widget)= 0;
  l_error= 0;
  l_exit= 0;

  bson_document_assign(&l_document);

  do
  {

    l_exit= query_read(o_error, &(*io_report).m_query);

    if (l_exit)
    {
      break;
    }

    l_fp= g_fopen((*io_report).m_reportfile, "rb");
  
    if (0 == l_fp)
    {
      l_error= g_error_new(
        REPORT, 
        REPORT_FILE_OPEN,
        "Unable to open file ('%s'); errno(%d)='%s'",
        (*io_report).m_reportfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    l_exit= bson_document_parse(&l_document, bson_fread, l_fp);

    fclose(l_fp);

    if (l_exit)
    {
      l_error= g_error_new(
        REPORT, 
        REPORT_PARSE,
        "Unable to parse file ('%s')",
        (*io_report).m_reportfile);
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    l_element= l_document.m_head;

    if (0 == l_element)
    {
      break;
    }

    if (bson_type_end == (*l_element).m_type)
    {
      break;
    }

    if (strcmp("iwrite", (*l_element).m_ename))
    {
      break;
    }

    if (bson_type_int32 != (*l_element).m_type)
    {
      break;
    }

    (*io_report).m_style= (*l_element).m_object.m_int32;
    database_aspect_read(&(*io_report).m_aspect, l_element);

    switch((*io_report).m_style)
    {
      case style_layout:
        layout_assign(&(*io_report).m_object.m_layout);
        layout_assign_filename(
          &(*io_report).m_object.m_layout,
          (*io_report).m_reportfile);
        (*o_widget)= report_gui_new(io_report);
        l_exit= layout_read(&l_error, &(*io_report).m_object.m_layout, l_element);
        break;
      case style_export:
        export_assign(&(*io_report).m_object.m_export);
        export_assign_filename(
          &(*io_report).m_object.m_export,
          (*io_report).m_reportfile);
        (*o_widget)= report_gui_new(io_report);
        l_exit= export_read(&l_error, &(*io_report).m_object.m_export, l_element);
        break;
      case style_label:
        label_assign(&(*io_report).m_object.m_label);
        label_assign_filename(
          &(*io_report).m_object.m_label,
          (*io_report).m_reportfile);
        (*o_widget)= report_gui_new(io_report);
        l_exit= label_read(&l_error, &(*io_report).m_object.m_label, l_element);
      default:
        break;
    }

  }while(0);

  bson_document_discharge(&l_document);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
report_set_horizontal_ruler_line(
  struct report *const                  io_report,
  gboolean const                        i_state)
{

  switch((*io_report).m_style)
  {
  case style_layout:
    layout_set_horizontal_ruler_line(&(*io_report).m_object.m_layout, i_state);
    break;
  case style_label:
    label_set_horizontal_ruler_line(&(*io_report).m_object.m_label, i_state);
    break;
  default:
    /* programming error */
    break;
  }

  return;
}

extern void
report_set_vertical_ruler_line(
  struct report *const                  io_report,
  gboolean const                        i_state)
{

  switch((*io_report).m_style)
  {
  case style_layout:
    layout_set_vertical_ruler_line(&(*io_report).m_object.m_layout, i_state);
    break;
  case style_label:
    label_set_vertical_ruler_line(&(*io_report).m_object.m_label, i_state);
    break;
  default:
    /* programming error */
    break;
  }

  return;
}

extern void
report_set_scale(
  struct report *const                  io_report,
  double const                          i_scale)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_set_scale(&(*io_report).m_object.m_layout, i_scale);
      break;
    case style_label:
      label_set_scale(&(*io_report).m_object.m_label, i_scale);
    default:
      /* programming error */
      break;
  }

  return;
}

extern void
report_set_scale_zoom_in(
  struct report *const                  io_report)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_set_scale_zoom_in(&(*io_report).m_object.m_layout);
      break;
    case style_label:
      label_set_scale_zoom_in(&(*io_report).m_object.m_label);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

extern void
report_set_scale_zoom_out(
  struct report *const                  io_report)
{

  switch((*io_report).m_style)
  {
    case style_layout:
      layout_set_scale_zoom_out(&(*io_report).m_object.m_layout);
      break;
    case style_label:
      label_set_scale_zoom_out(&(*io_report).m_object.m_label);
      break;
    default:
      /* programming error */
      break;
  }

  return;
}

extern int
report_write(
  GError**                              o_error,
  struct report const*const             i_report)
{
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  char                                  l_octet;
  int32_t                               l_value;

  l_error= 0;
  l_exit= -1;
  l_fp= 0;

  do
  {

    l_exit= query_write(o_error, &(*i_report).m_query);

    if (l_exit)
    {
      break;
    }

    if (0 == (*i_report).m_reportfile)
    {
      /* programming error*/
      l_exit= -1;
      break;
    }

    l_fp= g_fopen((*i_report).m_reportfile, "w+b");

    if (0 == l_fp)
    {
      l_error= g_error_new(
        REPORT, 
        REPORT_FILE_CREATE,
        "Unable to create file ('%s'); errno(%d)='%s'",
        (*i_report).m_reportfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    /* int32 document size */
    l_value= 0;
    fwrite(&l_value, sizeof(int32_t), 1, l_fp);

    /* iwrite signature - file format version */
    l_octet= bson_type_int32;
    fwrite(&l_octet, 1, 1, l_fp); 
    fprintf(l_fp, "iwrite%c",0);
    l_value= (*i_report).m_style;
    fwrite(&l_value, sizeof(int32_t), 1, l_fp);

    database_aspect_write(l_fp, &(*i_report).m_aspect);

    switch((*i_report).m_style)
    {
      case style_layout:
        layout_write(&l_error, l_fp, &(*i_report).m_object.m_layout);
        break;
      case style_export:
        export_write(&l_error, l_fp, &(*i_report).m_object.m_export);
        break;
      case style_label:
        label_write(&l_error, l_fp, &(*i_report).m_object.m_label);
        break;
      default:
        break;
    }

    /* end of document marker */
    fprintf(l_fp, "%c", 0);

    /* document size*/
    l_value= ftell(l_fp);
    fseek(l_fp, 0, SEEK_SET);
    fwrite(&l_value, sizeof(int32_t), 1, l_fp);
    fclose(l_fp);
    l_fp= 0;

    l_exit= 0;

  }while(0);

  if (l_fp)
  {
    fclose(l_fp);
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}
