/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "query_exec.h"

static void
query_exec_node_assign(
  struct query_exec_node*const          o_query)
{

  memset(o_query, 0, sizeof(*o_query));

  sql_assign(&(*o_query).m_query);
  database_session_assign(&(*o_query).m_session);
  (*o_query).m_hash= g_hash_table_new(g_str_hash, g_str_equal);

  return;
}

static void
query_exec_node_discharge(
  struct query_exec_node*const          io_query)
{

  sql_discharge(&(*io_query).m_query);
  database_session_discharge(&(*io_query).m_session);

  if ((*io_query).m_text)
  {
    g_free((*io_query).m_text);
  }

  if ((*io_query).m_hash)
  {
    g_hash_table_destroy((*io_query).m_hash);
  }

  memset(io_query, 0, sizeof(*io_query));

  return;
}

static void
query_exec_node_hash_attributes(
  struct query_exec_node*const          io_query)
{
  unsigned                              l_slot;

  g_hash_table_remove_all((*io_query).m_hash);

  for (l_slot= 0; (*io_query).m_attr_count > l_slot; l_slot++)
  {
    g_hash_table_insert(
      (*io_query).m_hash,
      (*io_query).m_attr[l_slot].m_tag,
      (gpointer)&(*io_query).m_attr[l_slot]);
  }

  return;
}

#undef query_exec_node_lookup
extern void
query_exec_node_lookup(
  struct iwsql_attr const**             o_attr,
  char const*                           i_tag,
  struct query_exec_node const*const    i_query)
{
  gpointer                              l_ptr;

  l_ptr= g_hash_table_lookup((*i_query).m_hash, i_tag);
  (*o_attr)= (struct iwsql_attr*)l_ptr;

  return;
}

#define PRIMARY                         0

static void
query_exec_parse(
  struct query_exec*                    io_exec,
  struct query const*const              i_query)
{
  unsigned                              l_count;
  struct query_node const*              l_node;
  struct query_exec_node*               l_query;
  unsigned                              l_size;
  unsigned                              l_slot;

  do
  {

    query_count(&l_count, i_query);

    if (0 == l_count)
    {
      break;
    }

    l_size= (((*io_exec).m_count + l_count) * sizeof(struct query_exec_node));
    (*io_exec).m_sql= g_realloc((*io_exec).m_sql, l_size);

    l_slot= 0;
    l_query= &(*io_exec).m_sql[(*io_exec).m_count];
    (*io_exec).m_count+= l_count;

    l_node= (*i_query).m_head;

    do
    {

      if (l_count <= l_slot)
      {
        break;
      }

      query_exec_node_assign(l_query);
      sql_parse(&(*l_query).m_query, l_node);

      l_node= (*l_node).m_next;
      l_query++;
      l_slot++;

    }while(1);

  }while(0);

  return;
}

static void
query_exec_prompt_copy_values(
  struct query_exec*                    io_exec)
{
  struct sql_node*                      l_node;
  struct query_exec_node*               l_query;
  unsigned                              l_slot;
  gchar const*                          l_text;

  l_slot= 0;
  l_query= &(*io_exec).m_sql[PRIMARY];

  do
  {

    if ((*io_exec).m_count <= l_slot)
    {
      break;
    }

    l_node= (*l_query).m_query.m_head;

    do
    {

      if (0 == l_node)
      {
        break;
      }

      if (sql_type_input == (*l_node).m_type)
      {
        l_text= gtk_entry_get_text((*l_node).m_object.m_input.m_entry);
        (*l_node).m_object.m_input.m_answer= g_strdup(l_text);
        (*l_node).m_object.m_input.m_entry= 0;
      }

      l_node= (*l_node).m_next;

    }while(1);

    l_slot++;
    l_query++;

  }while(1);

  return;
}

static int
query_exec_prompt_set_values(
  struct query_exec*                    io_exec,
  GHashTable const*const                i_prompt)
{
  int                                   l_exit;
  struct sql_node*                      l_node;
  struct query_exec_node*               l_query;
  size_t                                l_size;
  unsigned                              l_slot;
  char const*                           l_value;

  l_exit= 0;
  l_slot= 0;
  l_query= &(*io_exec).m_sql[PRIMARY];
  l_value= 0;

  do
  {
      if (0 == i_prompt)
      {
        break;
      }

    do
    {

      if ((*io_exec).m_count <= l_slot)
      {
        break;
      }

      l_node= (*l_query).m_query.m_head;

      do
      {

        if (0 == l_node)
        {
          break;
        }

        do
        {
          if (sql_type_input != (*l_node).m_type)
          {
            break;
          }

          l_value= g_hash_table_lookup(
            (GHashTable*)i_prompt,
            (*l_node).m_object.m_input.m_prompt);

          if ((0 == l_value) || (0 == l_value[0]))
          {
            break;
          }

          l_size= strlen(l_value);
          (*l_node).m_object.m_input.m_answer= (char*)malloc(1+l_size);
          memset((*l_node).m_object.m_input.m_answer, 0, 1+l_size);
          memcpy((*l_node).m_object.m_input.m_answer, l_value, l_size);

        }while(0);

        l_node= (*l_node).m_next;

      }while(1);

      l_slot++;
      l_query++;

    }while(1);

  }while(0);

  return l_exit;
}

static int
query_exec_prompt_resolve(
  struct query_exec*                    io_exec,
  GHashTable const*const                i_prompt)
{
  GtkBox*                               l_box;
  GtkDialog*                            l_dialog;
  GtkEntry*                             l_entry;
  int                                   l_exit;
  GtkGrid*                              l_grid;
  GtkLabel*                             l_label;
  struct sql_node*                      l_node;
  struct query_exec_node*               l_query;
  unsigned                              l_row;
  GtkWidget*                            l_scrollwindow;
  unsigned                              l_slot;
  gpointer                              l_value;
  GtkWidget*                            l_viewport;

  l_exit= 0;

  l_dialog= GTK_DIALOG(gtk_dialog_new_with_buttons(
    "SQL query input",
    mainw(),
    GTK_DIALOG_MODAL,
    "Ok",
    GTK_RESPONSE_OK,
    "Cancel",
    GTK_RESPONSE_CANCEL,
    NULL));

  gtk_dialog_set_default_response(l_dialog, GTK_RESPONSE_OK);

  l_box= GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(l_dialog)));

  l_scrollwindow= gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_hexpand(GTK_WIDGET(l_scrollwindow), 1);
  gtk_widget_set_vexpand(GTK_WIDGET(l_scrollwindow), 1);
  gtk_box_pack_start(l_box, GTK_WIDGET(l_scrollwindow), 1, 1, 0);

  l_viewport= gtk_viewport_new(NULL, NULL);
  gtk_widget_set_hexpand(GTK_WIDGET(l_viewport), 1);
  gtk_widget_set_vexpand(GTK_WIDGET(l_viewport), 1);
  gtk_container_add(GTK_CONTAINER(l_scrollwindow), GTK_WIDGET(l_viewport));

  l_grid= GTK_GRID(gtk_grid_new());
  gtk_container_add(GTK_CONTAINER(l_viewport), GTK_WIDGET(l_grid));

  l_slot= 0;
  l_row= 0;
  l_query= &(*io_exec).m_sql[PRIMARY];

  do
  {

    if ((*io_exec).m_count <= l_slot)
    {
      break;
    }

    l_node= (*l_query).m_query.m_head;

    do
    {

      if (0 == l_node)
      {
        break;
      }

      if (sql_type_input == (*l_node).m_type)
      {
        l_label= GTK_LABEL(gtk_label_new((*l_node).m_object.m_input.m_prompt));
        l_entry= GTK_ENTRY(gtk_entry_new());
        (*l_node).m_object.m_input.m_entry= l_entry;

        if (i_prompt)
        {
          l_value= g_hash_table_lookup(
            (GHashTable*)i_prompt,
            (*l_node).m_object.m_input.m_prompt);

          if (l_value)
          {
            gtk_entry_set_text(l_entry, l_value);
          }
        }

        gtk_grid_attach(l_grid, GTK_WIDGET(l_label), 0, l_row, 1, 1);
        gtk_grid_attach(l_grid, GTK_WIDGET(l_entry), 1, l_row, 1, 1);
        l_row++;
      }

      l_node= (*l_node).m_next;

    }while(1);

    l_slot++;
    l_query++;

  }while(1);

  do
  {

    if (l_exit)
    {
      break;
    }

    if (0 == l_row)
    {
      break;
    }

    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);
    l_exit= gtk_dialog_run(l_dialog);

    if (GTK_RESPONSE_OK != l_exit)
    {
      l_exit= -1;
      break;
    }

    query_exec_prompt_copy_values(io_exec);

    l_exit= 0;

  }while(0);

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  return l_exit;
}

extern void
query_exec_assign(
  struct query_exec*                    o_exec)
{

  memset(o_exec, 0, sizeof(*o_exec));

  (*o_exec).m_hash= g_hash_table_new(g_str_hash, g_str_equal);

  return;
}

extern int
query_exec_connect(
  GError**                              o_error,
  struct query_exec*                    io_exec,
  struct database_aspect const*const    i_aspect)
{
  int                                   l_exit;
  struct query_exec_node*               l_query;
  unsigned                              l_slot;

  l_exit= 0;
  l_slot= 0;
  l_query= &(*io_exec).m_sql[PRIMARY];

  do
  {

    if ((*io_exec).m_count <= l_slot)
    {
      break;
    }

    l_exit= database_session_connect(o_error, &(*l_query).m_session, i_aspect);

    if (l_exit)
    {
      break;
    }

    l_slot++;
    l_query++;

  }while(1);

  return l_exit;
}

extern int
query_exec_connect_prepare_first(
  GError**                              o_error,
  int *const                            o_eof,
  struct query_exec*const               io_exec,
  struct query const*const              i_query,
  struct database_aspect const*const    i_aspect,
  GHashTable const*const                i_prompt,
  int const                             i_interactive)
{
  int                                   l_eof;
  int                                   l_exit;

  (*o_eof)= 0;
  l_eof= 0;
  l_exit= 0;

  do
  {

    l_exit= query_exec_prepare(
      o_error,
      io_exec,
      i_query,
      (GHashTable*)i_prompt,
      i_interactive);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_connect(o_error, io_exec, i_aspect);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_run(o_error, io_exec);

    if (l_exit)
    {
      break;
    }

    l_exit= query_exec_next(o_error, &l_eof, io_exec);

    if (l_exit)
    {
      break;
    }

    if (l_eof)
    {
      break;
    }
  
  }while(0);

  (*o_eof)= l_eof;

  return l_exit;
}

extern void
query_exec_discharge(
  struct query_exec*                    io_exec)
{
  struct query_exec_node*               l_query;
  unsigned                              l_slot;

  if ((*io_exec).m_hash)
  {
    g_hash_table_destroy((*io_exec).m_hash);
  }

  do
  {

    if (0 == (*io_exec).m_sql)
    {
      break;
    }

    l_query= &(*io_exec).m_sql[PRIMARY];
    l_slot= 0;
    
    do
    {

      if ((*io_exec).m_count <= l_slot)
      {
        break;
      }

      query_exec_node_discharge(l_query);

      l_query++;
      l_slot++;

    }while(1);

    g_free((*io_exec).m_sql);

  }while(0);

  memset(io_exec, 0, sizeof(*io_exec));

  return;
}

static void
query_exec_hash_tables(
  struct query_exec*                    io_exec)
{
  unsigned                              l_slot;

  g_hash_table_remove_all((*io_exec).m_hash);

  for(l_slot= 0; (*io_exec).m_count > l_slot; l_slot++)
  {
    g_hash_table_insert(
      (*io_exec).m_hash, 
      (*io_exec).m_sql[l_slot].m_query.m_tag,
      &(*io_exec).m_sql[l_slot]);
  }

  return;
}

#undef query_exec_lookup
extern void
query_exec_lookup(
  struct query_exec_node const**        o_query,
  char const*                           i_tag,
  struct query_exec const*const         i_exec)
{
  gpointer                              l_ptr;

  l_ptr= g_hash_table_lookup((*i_exec).m_hash, i_tag);
  (*o_query)= (struct query_exec_node*)l_ptr;

  return;
}

static int
query_exec_write(
  FILE*                                 io_fp,
  struct query_exec_node const*const    i_sql)
{
  int                                   l_exit;
  struct sql_node*                      l_node;

  l_exit= 0;
  l_node= (*i_sql).m_query.m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if (sql_type_segment == (*l_node).m_type)
    {
      if ((*l_node).m_object.m_segment.m_text)
      {
        fwrite(
          (*l_node).m_object.m_segment.m_text, 
          (*l_node).m_object.m_segment.m_size, 
          1,
          io_fp);
      }
    }
    else if (sql_type_input == (*l_node).m_type)
    {
      if ((*l_node).m_object.m_input.m_answer)
      {
        fwrite(
          (*l_node).m_object.m_input.m_answer,
          strlen((*l_node).m_object.m_input.m_answer),
          1,
          io_fp);
      }
      else
      {
        fwrite("''", 2, 1, io_fp);
      }
    }
    else if (sql_type_parm == (*l_node).m_type)
    {
      if ((*l_node).m_object.m_parm.m_answer)
      {
        fwrite(
          (*l_node).m_object.m_parm.m_answer,
          strlen((*l_node).m_object.m_parm.m_answer),
          1,
          io_fp);
      }
      else
      {
        fwrite("''", 2, 1, io_fp);
      }
    }

    l_node= (*l_node).m_next;

  }while(1);

  return l_exit;
}

extern int
query_exec_prepare(
  GError**                              o_error,
  struct query_exec*const               io_exec,
  struct query const*const              i_query,
  GHashTable const*const                i_prompt,
  int const                             i_interactive)
{
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  struct query_exec_node*               l_query;
  size_t                                l_size;
  char*                                 l_text;

  l_error= 0;
  l_exit= 0;
  l_fp= 0;

  do
  {

    query_exec_parse(io_exec, i_query);

    if (i_interactive)
    {
      l_exit= query_exec_prompt_resolve(io_exec, i_prompt);

      if (l_exit)
      {
        break;
      }
    }
    else
    {
      l_exit= query_exec_prompt_set_values(io_exec, i_prompt);
    }

    l_fp= tmpfile();

    if (0 == l_fp)
    {
      l_error= g_error_new(
        QUERY_EXEC, 
        QUERY_EXEC_FILE_CREATE1,
        "Unable to create temporary file; errno(%d)='%s'",
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    l_query= &(*io_exec).m_sql[PRIMARY];

    if (0 == l_query)
    {
      break;
    }

    query_exec_write(l_fp, l_query);

    fseek(l_fp, 0, SEEK_END);
    l_size= ftell(l_fp);
    fseek(l_fp, 0, SEEK_SET);

    l_text= (char*)g_malloc0(1+l_size);
    memset(l_text, 0, (1+l_size));

    fread(l_text, l_size, 1, l_fp);

    if ((*l_query).m_text)
    {
      g_free((*l_query).m_text);
    }

    (*l_query).m_text= l_text;
    (*l_query).m_text_size= l_size;

  }while(0);

  if (l_fp)
  {
    fclose(l_fp);
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
query_exec_run(
  GError**                              o_error,
  struct query_exec*const               io_exec)
{
  int                                   l_exit;
  struct query_exec_node*               l_query;

  l_exit= 0;

  do
  {

    if (0 == (*io_exec).m_count)
    {
      break;
    }

    query_exec_hash_tables(io_exec);

    l_query= &(*io_exec).m_sql[PRIMARY];

    l_exit= database_session_exec_buffer(
      o_error,
      &(*l_query).m_session,
      (*l_query).m_text);

    if (l_exit)
    {
      break;
    }

    database_session_get_attributes(
      &(*l_query).m_attr,
      &(*l_query).m_attr_count,
      &(*l_query).m_session);

    query_exec_node_hash_attributes(l_query);

  }while(0);

  return l_exit;
}

extern int
query_exec_parm_primary_resolve(
  struct sql_parm *const                i_parm,
  struct query_exec_node const*         i_sql)
{
  struct iwsql_attr const*              l_attr;
  int                                   l_exit;
  int                                   l_rc;
  int                                   l_slot;

  l_exit= -1;
  l_attr= &(*i_sql).m_attr[0];
  l_slot= 0;

  do
  {

    if ((*i_sql).m_attr_count <= l_slot)
    {
      break;
    }

    l_rc= strcmp((*i_parm).m_field, (*l_attr).m_tag);

    if (0 == l_rc)
    {

      if ((*i_parm).m_answer)
      {
        g_free((*i_parm).m_answer);
        (*i_parm).m_answer= 0;
      }

      if ((*l_attr).m_value)
      {
        (*i_parm).m_answer= g_strdup((*l_attr).m_value);
      }

      l_exit= 0;

      break;
    }

    l_attr++;
    l_slot++;

  }while(1);

  return l_exit;
}

extern void
query_exec_parm_primary_lookup(
  unsigned *const                       o_slot,
  struct sql_parm const*const           i_parm,
  struct query_exec const*const         io_exec,
  unsigned const                        i_slot)
{
  unsigned                              l_pri;
  struct query_exec_node*               l_query;
  int                                   l_rc;
  unsigned                              l_slot;

  l_pri= -1;

  do
  {

    if ((0 == (*i_parm).m_tag) || (0 == (*i_parm).m_tag[0]))
    {
      l_pri= 0;
      break;
    }

    l_slot= 0;
    l_query= &(*io_exec).m_sql[PRIMARY];

    do
    {

      if ((*io_exec).m_count <= l_slot)
      {
        break;
      }

      if (i_slot <= l_slot)
      {
        break;
      }

      l_rc= strcmp((*i_parm).m_tag, (*l_query).m_query.m_tag);

      if (0 == l_rc)
      {
        l_pri= l_slot;
        break;
      }

      l_query++;
      l_slot++;

    }while(1);

  }while(0);

  (*o_slot)= l_pri;

  return;
}
  
static int
query_exec_parm_resolve(
  struct query_exec*const               io_exec,
  unsigned const                        i_slot)
{
  struct query_exec_node*               l_resolve;
  unsigned                              l_slot;
  int                                   l_exit;
  struct sql_node*                      l_node;

  l_exit= 0;
  l_resolve= &(*io_exec).m_sql[i_slot];
  l_node= (*l_resolve).m_query.m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    if (sql_type_parm == (*l_node).m_type)
    {
      query_exec_parm_primary_lookup(
        &l_slot,
        &(*l_node).m_object.m_parm,
        io_exec,
        i_slot);

      if (-1 == l_slot)
      {
        /* query not found m_tag */
        l_exit= -1;
        break;
      }

      l_exit= query_exec_parm_primary_resolve(
        &(*l_node).m_object.m_parm,
        &(*io_exec).m_sql[l_slot]);

      if (l_exit)
      {
        /* attr not found */
        break;
      }

    }

    l_node= (*l_node).m_next;

  }while(1);

  return l_exit;
}

extern int
query_exec_next(
  GError**                              o_error,
  int*const                             o_eof,
  struct query_exec*const               io_exec)
{
  int                                   l_eof;
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  struct query_exec_node*               l_query;
  size_t                                l_size;
  unsigned                              l_slot;
  char*                                 l_text;

  l_error= 0;
  l_exit= 0;
  (*o_eof)= 0;

  do
  {

    if (0 == (*io_exec).m_count)
    {
      (*o_eof)= 1;
      break;
    }

    l_query= &(*io_exec).m_sql[PRIMARY];

    l_exit= database_session_next(&l_error, o_eof, &(*l_query).m_session);

    if (l_exit)
    {
      break;
    }

    if ((*o_eof))
    {
      break;
    }

    l_slot= 1;
    l_query= &(*io_exec).m_sql[1];

    do
    {

      if ((*io_exec).m_count <= l_slot)
      {
        break;
      }

      query_exec_parm_resolve(io_exec, l_slot);

      l_fp= tmpfile();

      if (0 == l_fp)
      {
        l_error= g_error_new(
          QUERY_EXEC, 
          QUERY_EXEC_FILE_CREATE2,
          "Unable to create temporary file; errno(%d)='%s'",
          errno,
          strerror(errno));
        _error_log(l_error);
        l_exit= -1;
        break;
      }

      query_exec_write(l_fp, l_query);

      fseek(l_fp, 0, SEEK_END);
      l_size= ftell(l_fp);
      fseek(l_fp, 0, SEEK_SET);

      l_text= (char*)g_malloc0(1+l_size);
      memset(l_text, 0, (1+l_size));

      fread(l_text, l_size, 1, l_fp);
      fclose(l_fp);
      l_fp= 0;

      if ((*l_query).m_text)
      {
        g_free((*l_query).m_text);
      }

      (*l_query).m_text= l_text;
      (*l_query).m_text_size= l_size;

      l_exit= database_session_exec_buffer(
        &l_error,
        &(*l_query).m_session,
        (*l_query).m_text);

      if (l_exit)
      {
        break;
      }

      database_session_get_attributes(
        &(*l_query).m_attr,
        &(*l_query).m_attr_count,
        &(*l_query).m_session);

      query_exec_node_hash_attributes(l_query);

      l_exit= database_session_next(&l_error, &l_eof, &(*l_query).m_session);

      if (l_exit)
      {
        break;
      }

      l_query++;
      l_slot++;

    }while(1);

  }while(0);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}
