/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "query.h"

extern void
query_assign(
  struct query*                         o_query)
{

  memset(o_query, 0, sizeof(*o_query));

  return;
}

extern void
query_clear(
  struct query*                         io_query)
{

  query_list_delete(io_query);

  if ((*io_query).m_sqlfile)
  {
    g_free((*io_query).m_sqlfile);
  }

  memset(io_query, 0, sizeof(*io_query));

  return;
}

extern void
query_count(
  unsigned*const                        o_count,
  struct query const*const              i_query)
{
  unsigned                              l_count;
  struct query_node*                    l_node;

  l_count= 0;
  l_node= (*i_query).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_count++;
    l_node= (*l_node).m_next;

  }while(1);

  (*o_count)= l_count;

  return;
}

extern void
query_delete(
  struct query*                         io_query,
  struct query_node*                    io_node)
{

  if (0 == (*io_node).m_prev)
  {
    (*io_query).m_head= (*io_node).m_next;
  }
  else
  {
    (*io_node).m_prev->m_next= (*io_node).m_next;
  }

  if (0 == (*io_node).m_next)
  {
    (*io_query).m_tail= (*io_node).m_prev;
  }
  else
  {
    (*io_node).m_next->m_prev= (*io_node).m_prev;
  }

  query_node_discharge(io_node);
  g_free(io_node);

  return;
}

extern void
query_discharge(
  struct query*                         io_query)
{

  query_clear(io_query);

  return;
}

extern void
query_first_new(
  struct query_node**                   o_node,
  struct query*                         io_query)
{
  struct query_node*                    l_node;

  l_node= (struct query_node*)g_malloc0(sizeof(*l_node));
  (*o_node)= l_node;

  do
  {

    if (0 == (*io_query).m_head)
    {
      (*io_query).m_head= l_node; 
      (*io_query).m_tail= l_node;
      break;
    }

    (*io_query).m_head->m_prev= l_node;
    (*l_node).m_next= (*io_query).m_head;
    (*io_query).m_head= l_node;

  }while(0);

  return;
}

extern void
query_insert_after(
  struct query*const                     io_query,
  struct query_node*const                i_node,
  struct query_node*const                i_node_new)
{

  if (0 == (*i_node).m_next)
  {
    (*io_query).m_tail= i_node_new;
  }
  else
  {
    (*i_node).m_next->m_prev= i_node_new;
  }

  (*i_node_new).m_next= (*i_node).m_next;
  (*i_node_new).m_prev= i_node;
  (*i_node).m_next= i_node_new;

  return;
}

extern void
query_insert_before(
  struct query*const                     io_query,
  struct query_node*const                i_node,
  struct query_node*const                i_node_new)
{

  if (0 == (*i_node).m_prev)
  {
    (*io_query).m_head= i_node_new;
  }
  else
  {
    (*i_node).m_prev->m_next= i_node_new;
  }

  (*i_node_new).m_next= i_node;
  (*i_node_new).m_prev= (*i_node).m_prev;
  (*i_node).m_prev= i_node_new;

  return;
}

extern void
query_last_new(
  struct query_node**                   o_node,
  struct query*                         io_query)
{
  struct query_node*                    l_node;

  l_node= (struct query_node*)g_malloc0(sizeof(*l_node));
  (*o_node)= l_node;

  do
  {

    if (0 == (*io_query).m_head)
    {
      (*io_query).m_head= l_node; 
      (*io_query).m_tail= l_node;
      break;
    }

    (*io_query).m_tail->m_next= l_node;
    (*l_node).m_prev= (*io_query).m_tail;
    (*io_query).m_tail= l_node;

  }while(0);

  return;
}

extern void
query_list_delete(
  struct query*                         io_query)
{
  struct query_node*                    l_node;
  struct query_node*                    l_next;

  l_node= (*io_query).m_head;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_next= (*l_node).m_next;

    query_node_discharge(l_node);

    g_free(l_node);
    l_node= l_next;

  }while(1);

  (*io_query).m_head= 0;
  (*io_query).m_tail= 0;

  return;
}

extern void
query_node_assign(
  struct query_node*                    io_node)
{

  memset(io_node, 0, sizeof(*io_node));

  return;
}

extern void
query_node_discharge(
  struct query_node*                    io_node)
{


  if ((*io_node).m_text)
  {
    g_free((*io_node).m_text);
  }

  memset(io_node, 0, sizeof(*io_node));

  return;
}

extern int
query_read(
  GError**                              o_error,
  struct query*const                    io_query)
{
  struct bson_document                  l_document;
  struct bson_node*                     l_element;
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  struct query_node *                   l_node;
  struct bson_node*                     l_query;
  int                                   l_rc;

  l_fp= 0;
  l_error= 0;
  l_exit= 0;
  l_node= (*io_query).m_head;
  bson_document_assign(&l_document);

  do
  {

    l_fp= g_fopen((*io_query).m_sqlfile, "r");

    if (0 == l_fp)
    {
      l_error= g_error_new(
        QUERY, 
        QUERY_FILE_OPEN,
        "Unable to open SQL file ('%s'); errno(%d)='%s'",
        (*io_query).m_sqlfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    l_exit= bson_document_parse(&l_document, bson_fread, l_fp);

    fclose(l_fp);

    if (l_exit)
    {
      l_error= g_error_new(
        QUERY, 
        QUERY_SQL_PARSE,
        "Unable to parse SQL file ('%s')",
        (*io_query).m_sqlfile);
      _error_log(l_error);
      break;
    }

    l_element= l_document.m_head;

    do
    {

      if (0 == l_element)
      {
        break;
      }

      if (bson_type_end == (*l_element).m_type)
      {
        break;
      }

      do
      {

        if (bson_type_array != (*l_element).m_type)
        {
          break;
        }
  
        l_query= (*l_element).m_object.m_document.m_head;

        if (0 == l_query)
        {
          break;
        }
      
        if (bson_type_string != (*l_query).m_type)
        {
          break; 
        }

        l_rc= strcmp("query", (*l_query).m_ename);

        if (l_rc)
        {
          break;
        }

        l_rc= !strcmp("primary", (*l_element).m_ename);

        if (l_rc)
        {
          query_first_new(&l_node, io_query);
        }
        else
        {
          query_last_new(&l_node, io_query);
        }

        strncpy(
          (*l_node).m_tag,
          (*l_element).m_ename,
          sizeof((*l_node).m_tag));

        (*l_node).m_size= ((*l_query).m_object.m_string.m_size-1);

        (*l_node).m_text= 
          (char*)g_malloc0((*l_query).m_object.m_string.m_size);

        memcpy(
          (*l_node).m_text,
          (*l_query).m_object.m_string.m_text,
          (*l_query).m_object.m_string.m_size);

      }while(0);

      l_element= (*l_element).m_next;

    }while(1);

  }while(0);

  bson_document_discharge(&l_document);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
query_node_write(
  FILE*const                            io_fp,
  struct query_node const*const         i_node)
{
  int                                   l_exit;
  int32_t                               l_int32;
  char                                  l_octet;
  int32_t                               l_offset;
  int32_t                               l_size;
  int32_t                               l_value;

  l_exit= 0;

  /* array */
  l_octet= bson_type_array;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "%s%c", (*i_node).m_tag, 0);

  /* document */
  l_offset= ftell(io_fp);
  l_size= 0;
  fwrite(&l_size, sizeof(int32_t), 1, io_fp); 

  /* query */
  if ((*i_node).m_size)
  {
    l_octet= bson_type_string;
    fwrite(&l_octet, 1, 1, io_fp); 
    fprintf(io_fp, "query%c",0);
    l_value= 1+(*i_node).m_size;
    fwrite(&l_value, sizeof(int32_t), 1, io_fp); 
    fwrite((*i_node).m_text, 1, l_value, io_fp);
  }

  /* end of document */
  l_octet= 0;
  fwrite(&l_octet, 1, 1, io_fp); 

  /* document */
  l_int32= ftell(io_fp);
  l_size= (l_int32 - l_offset);
  fseek(io_fp, l_offset, SEEK_SET);
  fwrite(&l_size, sizeof(l_size), 1, io_fp); 

  fseek(io_fp, 0, SEEK_END);

  return l_exit;
}

extern int
query_write(
  GError**                              o_error,
  struct query const*const              i_query)
{
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  struct query_node const*              l_node;
  char                                  l_octet;
  int32_t                               l_value;

  l_fp= 0;
  l_error= 0;
  l_exit= 0;
  l_node= (*i_query).m_head;

  do
  {

    l_fp= g_fopen((*i_query).m_sqlfile, "w");

    if (0 == l_fp)
    {
      l_error= g_error_new(
        QUERY, 
        QUERY_FILE_CREATE,
        "Unable to create SQL file ('%s'); errno(%d)='%s'",
        (*i_query).m_sqlfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    /* int32 document size */
    l_value= 0;
    fwrite(&l_value, sizeof(int32_t), 1, l_fp);

    /* iwrite signature - file format version */
    l_octet= bson_type_int32;
    fwrite(&l_octet, 1, 1, l_fp); 
    fprintf(l_fp, "iwsql%c",0);
    l_value= 1;
    fwrite(&l_value, sizeof(int32_t), 1, l_fp);

    do
    {

      if (0 == l_node)
      {
        break;
      }

      query_node_write(l_fp, l_node);

      l_node= (*l_node).m_next;

    }while(1);

    /* end of document marker */
    fprintf(l_fp, "%c", 0);

    /* document size*/
    l_value= ftell(l_fp);
    fseek(l_fp, 0, SEEK_SET);
    fwrite(&l_value, sizeof(int32_t), 1, l_fp);
    fclose(l_fp);
    l_fp= 0;

  }while(0);

  if (l_fp)
  {
    fclose(l_fp);
  }

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
query_unlink(
  struct query_node*                    io_node,
  struct query*const                    io_query)
{

  if (0 == (*io_node).m_prev)
  {
    (*io_query).m_head= (*io_node).m_next;
  } 
  else
  {
    (*io_node).m_prev->m_next= (*io_node).m_next;
  } 
  
  if (0 == (*io_node).m_next)
  {
    (*io_query).m_tail= (*io_node).m_prev;
  } 
  else
  {
    (*io_node).m_next->m_prev= (*io_node).m_prev;
  }
  
}
