/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "label_config.h"

extern void
label_config_assign(
  struct label_config*const             o_config)
{

  memset(o_config, 0, sizeof(*o_config));
  label_paper_set_defaults(&(*o_config).m_paper);

  return;
}

extern void
label_config_assign_filename(
  struct label_config*const             io_config,
  char const*                           i_reportfile)
{
  FILE*                                 l_fp;
  char*                                 l_ptr;
  unsigned                              l_size;
  char*                                 l_configfile;

  if ((*io_config).m_filename)
  {
    g_free((*io_config).m_filename);
  }

  l_size= strlen(i_reportfile);
  l_configfile= (char*)g_malloc0(5+l_size);
  strcpy(l_configfile, i_reportfile);

  l_ptr= strrchr(l_configfile, '.'); 

  if (0 == l_ptr)
  {
    l_ptr= &l_configfile[l_size];
  }

  strcpy(l_ptr, ".cfg");

  l_fp= g_fopen(l_configfile, "a+");

  if (l_fp)
  {
    fclose(l_fp);
  }

  (*io_config).m_filename= l_configfile;

  return;
}

extern void
label_config_discharge(
  struct label_config*const             io_config)
{

  if ((*io_config).m_filename)
  {
    g_free((*io_config).m_filename);
  }

  memset(io_config, 0, sizeof(*io_config));

  return;
}

extern int
label_config_read(
  GError**                              o_error,
  struct label_config*const             io_config)
{
  GError*                               l_error;
  GKeyFile*                             l_key;
  gboolean                              l_bool;

  l_error= 0;

  do
  {

    l_key= g_key_file_new();

    l_bool= g_key_file_load_from_file(
      l_key,
      (*io_config).m_filename, 
      G_KEY_FILE_NONE, 
      &l_error);

    if (0 == l_bool)
    {
      break;
    }

    (*io_config).m_paper.m_page_width= POINTS_PER_INCH*g_key_file_get_double(
      l_key,
      "Page Setup",
      "PageWidth",
      0);

    (*io_config).m_paper.m_page_height= POINTS_PER_INCH*g_key_file_get_double(
      l_key,
      "Page Setup",
      "PageHeight",
      0);

    (*io_config).m_paper.m_horizontal_pitch= POINTS_PER_INCH*g_key_file_get_double(
      l_key,
      "Page Setup",
      "HorizontalPitch",
      0);

    (*io_config).m_paper.m_vertical_pitch= POINTS_PER_INCH*g_key_file_get_double(
      l_key,
      "Page Setup",
      "VerticalPitch",
      0);

    (*io_config).m_paper.m_label_width= POINTS_PER_INCH*g_key_file_get_double(
      l_key,
      "Page Setup",
      "LabelWidth",
      0);

    (*io_config).m_paper.m_label_height= POINTS_PER_INCH*g_key_file_get_double(
      l_key,
      "Page Setup",
      "LabelHeight",
      0);

    (*io_config).m_paper.m_columns= g_key_file_get_integer(
      l_key,
      "Page Setup",
      "Columns",
      0);

    (*io_config).m_paper.m_rows= g_key_file_get_integer(
      l_key,
      "Page Setup",
      "Rows",
      0);

  }while(0);

  if (l_error)
  {
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  if (l_key)
  {
    g_key_file_free(l_key);
  }


  return 0;
}

extern int
label_config_write(
  GError**                              o_error,
  struct label_config const*const       i_config)
{
  GError*                               l_error;
  GKeyFile*                             l_key;
  gboolean                              l_bool;

  l_error= 0;
  l_key= 0;

  do
  {

    l_key= g_key_file_new();

    l_bool= g_key_file_load_from_file(
      l_key,
      (*i_config).m_filename, 
      G_KEY_FILE_NONE, 
      &l_error);

    if (0 == l_bool)
    {
      break;
    }

    g_key_file_set_double(
      l_key,
      "Page Setup",
      "PageWidth",
      (*i_config).m_paper.m_page_width/POINTS_PER_INCH);

    g_key_file_set_double(
      l_key,
      "Page Setup",
      "PageHeight",
      (*i_config).m_paper.m_page_height/POINTS_PER_INCH);

    g_key_file_set_double(
      l_key,
      "Page Setup",
      "HorizontalPitch",
      (*i_config).m_paper.m_horizontal_pitch/POINTS_PER_INCH);

    g_key_file_set_double(
      l_key,
      "Page Setup",
      "VerticalPitch",
      (*i_config).m_paper.m_vertical_pitch/POINTS_PER_INCH);

    g_key_file_set_double(
      l_key,
      "Page Setup",
      "LabelWidth",
      (*i_config).m_paper.m_label_width/POINTS_PER_INCH);

    g_key_file_set_double(
      l_key,
      "Page Setup",
      "LabelHeight",
      (*i_config).m_paper.m_label_height/POINTS_PER_INCH);

    g_key_file_set_integer(
      l_key,
      "Page Setup",
      "Columns",
      (*i_config).m_paper.m_columns);

    g_key_file_set_integer(
      l_key,
      "Page Setup",
      "Rows",
      (*i_config).m_paper.m_rows);

    l_bool= g_key_file_save_to_file(l_key, (*i_config).m_filename, 0);

    if (0 == l_bool)
    {
      break;
    }

  }while(0);

  if (l_error)
  {
    _error_log(l_error);
    g_propagate_error(o_error, l_error);
  }

  if (l_key)
  {
    g_key_file_free(l_key);
  }

  return 0;
}
