/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "label.h"
#include "paper.h"

static double const                     SCALE_MAX= 2.50;
static double const                     SCALE_MIN= 0.25;
static double const                     SCALE_FACTOR= 0.25;

extern void
label_assign(
  struct label*const                    o_label)
{

  memset(o_label, 0, sizeof(*o_label));
  label_config_assign(&(*o_label).m_config);
  (*o_label).m_scale= 1.0;

  return;
}

extern void
label_assign_filename(
  struct label*const                    io_label,
  char const*                           i_filename)
{

  label_config_assign_filename(&(*io_label).m_config, i_filename);

  return;
}

extern void
label_discharge(
  struct label*const                    io_label)
{

  label_config_discharge(&(*io_label).m_config);

  gtk_widget_destroy(GTK_WIDGET((*io_label).m_window));

  memset(io_label, 0, sizeof(*io_label));

  return;
}

extern GtkWidget*
label_gui_new(
  struct label*const                    io_label)
{
  struct paper                          l_paper;

  (*io_label).m_window= GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(0, 0));
  gtk_widget_set_hexpand(GTK_WIDGET((*io_label).m_window), 1);
  gtk_widget_set_vexpand(GTK_WIDGET((*io_label).m_window), 1);

  (*io_label).m_layout= GTK_LAYOUT(gtk_layout_new(0, 0));

  (*io_label).m_page= IWR_PAGE(iwr_page_new());
  iwr_page_set_section_type((*io_label).m_page, section_label);
  memset(&l_paper, 0, sizeof(l_paper));
  l_paper.m_width= (*io_label).m_config.m_paper.m_label_width;
  l_paper.m_height= (*io_label).m_config.m_paper.m_label_height;
  iwr_page_set_paper((*io_label).m_page, &l_paper);
  gtk_layout_set_size((*io_label).m_layout, l_paper.m_width, l_paper.m_height);

  gtk_layout_put(
    (*io_label).m_layout,
    GTK_WIDGET((*io_label).m_page),
    0,
    0);

  gtk_container_add(
    GTK_CONTAINER((*io_label).m_window),
    GTK_WIDGET((*io_label).m_layout));

  return GTK_WIDGET((*io_label).m_window);
}

extern void
label_highlight_objects(
  struct label*const                    io_label,
  gboolean const                        i_state)
{
  
  iwr_page_set_highlight_objects((*io_label).m_page, i_state);

  return;
}

extern struct item_node*
label_item_new(
  struct label*const                    io_label,
  enum item_type const                  i_type)
{
  struct item_node*                     l_node;

  gtk_widget_grab_focus(GTK_WIDGET((*io_label).m_page));
  iwr_page_item_new(&l_node, (*io_label).m_page, i_type);
  
  return l_node;
}

extern void
label_paper_changed(
  struct label*const                    io_label)
{
  struct paper                          l_paper;

  memset(&l_paper, 0, sizeof(l_paper));
  l_paper.m_width= (*io_label).m_config.m_paper.m_label_width;
  l_paper.m_height=  (*io_label).m_config.m_paper.m_label_height;
  iwr_page_set_paper((*io_label).m_page, &l_paper);
  gtk_layout_set_size((*io_label).m_layout, l_paper.m_width, l_paper.m_height);

  return;
}

extern int
label_read(
  GError**                              o_error,
  struct label *const                   io_label,
  struct bson_node const*const          i_element)
{
  struct bson_node const*               l_bson;
  struct bson_node const*               l_bson_section;
  int                                   l_exit;
  int                                   l_rc;

  l_exit= 0;
  l_bson= i_element;

  do
  {

    l_exit= label_config_read(o_error, &(*io_label).m_config);

    if (l_exit)
    {
      break;
    }

    do
    {

      if (0 == l_bson)
      {
        break;
      }

      do
      {

        if (bson_type_document != (*l_bson).m_type)
        {
          break;
        }

        l_rc= strcmp("label", (*l_bson).m_ename);

        if (l_rc) 
        {
          break;
        }

        l_bson_section= (*l_bson).m_object.m_document.m_head;

        if (bson_type_array != (*l_bson_section).m_type)
        {
          break;
        }

        if (0 == (*io_label).m_page)
        {
          (*io_label).m_page= IWR_PAGE(iwr_page_new());
        }
      
        iwr_page_set_section_type((*io_label).m_page, section_label);
        l_exit= iwr_page_read(o_error, (*io_label).m_page, l_bson_section);

      }while(0);

      if (l_exit)
      {
        break;
      }

      l_bson= (*l_bson).m_next;

    }while(1);

  }while(0);

  return l_exit;
}

extern void
label_set_horizontal_ruler_line(
  struct label *const                   io_label,
  gboolean const                        i_state)
{

  iwr_page_set_horizontal_ruler_line((*io_label).m_page, i_state);

  return;
}

extern void
label_set_scale(
  struct label *const                   io_label,
  double const                          i_scale)
{

  iwr_page_set_scale((*io_label).m_page, i_scale);
  (*io_label).m_scale= i_scale;

  return;
}

extern void
label_set_scale_zoom_in(
  struct label *const                   io_label)
{
  double                                l_scale;

  do
  {

    if (SCALE_MIN >= (*io_label).m_scale)
    {
      break;
    }

    l_scale= ((*io_label).m_scale - SCALE_FACTOR);

    label_set_scale(io_label, l_scale);

  }while(0);

  return;
}

extern void
label_set_scale_zoom_out(
  struct label *const                   io_label)
{
  double                                l_scale;

  do
  {

    if (SCALE_MAX <= (*io_label).m_scale)
    {
      break;
    }

    l_scale= ((*io_label).m_scale + SCALE_FACTOR);

    label_set_scale(io_label, l_scale);

  }while(0);

  return;
}

extern void
label_set_vertical_ruler_line(
  struct label *const                   io_label,
  gboolean const                        i_state)
{

  iwr_page_set_vertical_ruler_line((*io_label).m_page, i_state);

  return;
}

extern void
label_size_allocate(
  struct label*const                    io_label,
  GdkRectangle const*const              i_alloc)
{
  guint                                 l_len_x;
  guint                                 l_len_y;
  double                                l_page_center;
  double                                l_pos_x;
  double                                l_pos_y;
  double                                l_window_center;

  gtk_layout_get_size((*io_label).m_layout, &l_len_x, &l_len_y); 

  l_window_center= (*i_alloc).width / 2.0;
  l_page_center= l_len_x / 2.0;
  l_pos_x= (l_window_center - l_page_center);

  if (0 > l_pos_x)
  {
    l_pos_x= 0.0;
  }

  l_window_center= (*i_alloc).height / 2.0;
  l_page_center= l_len_y / 2.0;
  l_pos_y= (l_window_center - l_page_center);

  if (0 > l_pos_y)
  {
    l_pos_y= 0.0;
  }

  gtk_layout_move(
    (*io_label).m_layout, 
    GTK_WIDGET((*io_label).m_page),
    l_pos_x,
    l_pos_y);

  return;
}

extern int
label_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct label const*const              i_label)
{
  int                                   l_exit;
  int32_t                               l_int32;
  char                                  l_octet;
  int32_t                               l_offset;
  int32_t                               l_size;

  l_exit= 0;

  do
  {

    l_exit= label_config_write(o_error, &(*i_label).m_config);

    if (l_exit)
    {
      break;
    }

    l_octet= bson_type_document;
    fwrite(&l_octet, 1, 1, io_fp); 
    fprintf(io_fp, "%s%c", "label", 0);

    /* document */
    l_offset= ftell(io_fp);
    l_int32= 0;
    fwrite(&l_int32, sizeof(int32_t), 1, io_fp); 

    l_exit= iwr_page_write(o_error, io_fp, (*i_label).m_page);

    if (l_exit)
    {
      break;
    }

    /* end of document */
    l_octet= 0;
    fwrite(&l_octet, 1, 1, io_fp); 

    /* document */
    l_int32= ftell(io_fp);
    l_size= (l_int32 - l_offset);
    fseek(io_fp, l_offset, SEEK_SET);
    fwrite(&l_size, sizeof(l_size), 1, io_fp); 

    fseek(io_fp, 0, SEEK_END);

  }while(0);

  return l_exit;
}
