/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "iwr_page.h"
#include "iwr_page_menu.h"

enum key_action_type
{
  key_action_none,
  key_action_move,
  key_action_resize
};

enum iwr_page_drag_type
{
  drag_type_none= 0,
  drag_type_item,
  drag_type_resize,
};

struct _IwrPagePrivate
{
  enum iwr_page_section_type            m_type;
  struct item                           m_item;	
  GdkWindow*				                    m_event_window;
  GtkMenu*                              m_popup;
#if GTK_CHECK_VERSION(3,12,0)
  GtkPopover*                           m_popover;
#endif
  struct paper                          m_paper;
  struct geom_point                     m_event;
  enum event_type                       m_sticky;
  double                                m_scale;
  enum iwr_page_drag_type               m_drag_type;
  int                                   m_horizontal_line_show:1;
  int                                   m_vertical_line_show:1;
  int                                   m_highlight_objects:1; 
};

G_DEFINE_TYPE_WITH_PRIVATE(IwrPage, iwr_page, GTK_TYPE_WIDGET)

extern void
iwr_page_clear(
  IwrPage*const                         io_page)
{

  item_clear(&(*io_page).m_priv->m_item);

  return;
}

extern void
iwr_page_draw_item(
  cairo_t*const                         i_cr,
  IwrPage *const                        io_page,
  struct item_draw_options const*const  i_option)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;
  item_draw(&(*l_priv).m_item, i_cr, i_option);

  return;
}

extern void
iwr_page_item_selected_move(
  IwrPage*const                         io_page,
  struct geom_point const*const         i_delta)
{

  item_selected_move(&(*io_page).m_priv->m_item, i_delta);

  return;
}

extern void
iwr_page_get_scale(
  double *const                          o_scale,
  IwrPage const*const                    i_page)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*i_page).m_priv;
  (*o_scale)= (*l_priv).m_scale;

  return;
}

extern void
iwr_page_get_section_type(
  enum iwr_page_section_type* const     o_type,
  IwrPage const*const                   i_page)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*i_page).m_priv;
  (*o_type)= (*l_priv).m_type;

  return;
}

extern void
iwr_page_get_item(
  struct item**const                    o_item,
  IwrPage const*const                   i_page)
{

  (*o_item)= &(*i_page).m_priv->m_item;

  return;
}

extern void
iwr_page_get_paper(
  struct paper *const                   o_paper,
  IwrPage const*const                   i_page)
{

  (*o_paper)= (*i_page).m_priv->m_paper;

  return;
}

extern void
iwr_page_item_new(
  struct item_node**                    o_node,
  IwrPage*const                         io_page,
  enum item_type const                  i_type)
{

  item_node_new(o_node, &(*io_page).m_priv->m_item, i_type);
  gtk_widget_queue_draw(GTK_WIDGET(io_page));

  return;
}

static gint
iwr_page_key_press(
  GtkWidget*                            i_widget,
  GdkEventKey*                          i_event)
{
  enum key_action_type                  l_action;
  struct geom_point                     l_delta;
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv;
  int                                   l_redraw;
  enum event_type                       l_type;
  struct item_resize_event              l_resize;
 
  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;
  l_delta.m_pos_x= 0.0;
  l_delta.m_pos_y= 0.0;
  l_action= key_action_none;
  l_redraw= 0;
  l_type= event_none;

  do
  {

    if (GDK_KEY_Delete == (*i_event).keyval)
    {
      item_selected_remove(&(*l_page).m_priv->m_item);
      l_redraw= 1;
      break;
    }

    if (GDK_KEY_Left == (*i_event).keyval)
    {

      l_delta.m_pos_x= -1;

      if (0 == (GDK_SHIFT_MASK & (*i_event).state))
      {
        l_action= key_action_move;
      }
      else
      {
 
        l_action= key_action_resize;
        l_type= event_resize_west;
      }

      break;
    }
 
    if (GDK_KEY_Up == (*i_event).keyval)
    {

      l_delta.m_pos_y= -1;

      if (0 == (GDK_SHIFT_MASK & (*i_event).state))
      {
        l_action= key_action_move;
      }
      else
      {
        l_action= key_action_resize;
        l_type= event_resize_north;
      }

      break;

    }
 
    if (GDK_KEY_Right == (*i_event).keyval)
    {

      l_delta.m_pos_x= 1;

      if (0 == (GDK_SHIFT_MASK & (*i_event).state))
      {
        l_action= key_action_move;
      }
      else
      {
        l_action= key_action_resize;
        l_type= event_resize_west;
      }

      break;
    }

    if (GDK_KEY_Down == (*i_event).keyval)
    {

      l_delta.m_pos_y= 1;

      if (0 == (GDK_SHIFT_MASK & (*i_event).state))
      {
        l_action= key_action_move;
      }
      else
      {
        l_action= key_action_resize;
        l_type= event_resize_north;
      }

      break;
    }
 
  }while(0);

  if (key_action_move == l_action)
  {
    item_selected_move(&(*l_priv).m_item, &l_delta);
    l_redraw= 1;
  }
  else if (key_action_resize == l_action)
  {
 
    l_resize.m_event= l_delta;
    l_resize.m_type= l_type;
    item_selected_resize(&(*l_priv).m_item, &l_resize);
    l_redraw= 1;
  }

  if (l_redraw)
  {
      gtk_widget_queue_draw(i_widget);
  }

  return 1;
}

static void
iwr_page_in_resize_event(
  enum event_type*                      o_type,
  GtkWidget*                            i_widget,
  double                                i_event_x,
  double                                i_event_y)
{
  struct item_in_event                  l_ctx;
  int                                   l_hit;
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv; 

  (*o_type)= event_none;
  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;
  memset(&l_ctx, 0, sizeof(l_ctx));

  do
  {

    l_ctx.m_event.m_pos_x= i_event_x;
    l_ctx.m_event.m_pos_y= i_event_y;
    l_ctx.m_scale= (*l_priv).m_scale;

    l_hit= item_common_event_inside(
      &l_ctx,
      0.0,
      (*l_priv).m_paper.m_height-4.0,
      (*l_priv).m_paper.m_width,
      8.0);
    
    if (0 == l_hit)
    {
      break;
    }

    (*o_type)= event_resize_north_south;

  }while(0);

  return;
}

#if GTK_CHECK_VERSION(3,12,0)
static void
iwr_page_set_popover_text(
  IwrPagePrivate*                       l_priv)
{
  double                                l_height;
  GtkLabel*                             l_label;
  char                                  l_text[32];
  double                                l_width;

  l_height= ((*l_priv).m_paper.m_height / POINTS_PER_INCH); 
  l_width= ((*l_priv).m_paper.m_width / POINTS_PER_INCH);
  snprintf(l_text, sizeof(l_text), "%0.2lf x %0.2lf", l_width, l_height);
  l_label= GTK_LABEL(gtk_bin_get_child(GTK_BIN((*l_priv).m_popover)));
  gtk_label_set_text(l_label, l_text);

  return;
}
#endif

static gboolean
iwr_page_button_press2(
  GtkWidget*                            i_widget,
  GdkEventButton*                       i_event)
{
#if GTK_CHECK_VERSION(3,12,0)
  GtkLabel*                             l_label;
#endif
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv; 
  enum event_type                       l_type;

  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;

  do
  {

    iwr_page_in_resize_event(&l_type, i_widget, (*i_event).x, (*i_event).y);

    if (event_none == l_type)
    {
      break;
    }

    (*l_priv).m_drag_type= drag_type_resize;

#if GTK_CHECK_VERSION(3,12,0)
    (*l_priv).m_popover= GTK_POPOVER(gtk_popover_new(i_widget));
    gtk_popover_set_position((*l_priv).m_popover, GTK_POS_BOTTOM);
    l_label= GTK_LABEL(gtk_label_new(0));
    gtk_container_add(GTK_CONTAINER((*l_priv).m_popover), GTK_WIDGET(l_label));
    gtk_popover_set_modal((*l_priv).m_popover, 0);
    iwr_page_set_popover_text(l_priv);
    gtk_widget_show_all(GTK_WIDGET((*l_priv).m_popover));
#endif

  }while(0);

  return 1;
}

static gboolean
iwr_page_button_press(
  GtkWidget*                            i_widget,
  GdkEventButton*                       i_event)
{
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv; 
  struct item_node*                     l_node;
  enum event_type                       l_type;

  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;
  l_node= 0;
  l_type= event_none;

  if (!gtk_widget_has_focus(i_widget))
  {
    gtk_widget_grab_focus(i_widget);
  }

  do
  {

    (*l_priv).m_sticky= event_none;
    (*l_priv).m_event.m_pos_x= (*i_event).x;
    (*l_priv).m_event.m_pos_y= (*i_event).y;
    (*l_priv).m_drag_type= drag_type_none;

    item_node_in_event(
      &l_node,
      &l_type,
      &(*l_priv).m_item,
      &(*l_priv).m_event,
      (*l_priv).m_scale);

    if (0 == l_node)
    {
      item_deselect_all(&(*l_priv).m_item);
      if (section_label != (*l_priv).m_type)
      {
        iwr_page_button_press2(i_widget, i_event);
      }
      break;
    }

    if (0 == (GDK_SHIFT_MASK & (*i_event).state))
    {
      item_deselect_all(&(*l_priv).m_item);
    }

    (*l_node).m_selected= 1;

    if (GDK_2BUTTON_PRESS == (*i_event).type)
    {
      item_node_property(l_node);
      break;
    }

    if (3 /*right button*/ == (*i_event).button)
    {

      if (0 == (*l_priv).m_popup)
      {
        (*l_priv).m_popup= iwr_page_menu_new(&(*l_priv).m_item);
        gtk_widget_show_all(GTK_WIDGET((*l_priv).m_popup));
      }

      gtk_menu_popup(
        (*l_priv).m_popup,
        0, /* parent shell */
        0, /* menu item */
        0, /* user func */
        0, /* user data */
        (*i_event).button,
        (*i_event).time);

      break;
    }

    (*l_priv).m_drag_type= drag_type_item;
    (*l_priv).m_sticky= l_type;

  }while(0);

  gtk_widget_queue_draw(i_widget);

  return 1;
}

static gboolean
iwr_page_button_release(
  GtkWidget*                            i_widget,
  GdkEventButton*                       i_event)
{
  GtkAllocation                         l_alloc;
  GdkCursor*                            l_cursor;
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv; 

  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;

  do
  {

#if GTK_CHECK_VERSION(3,12,0)
    if ((*l_priv).m_popover)
    {
      gtk_widget_destroy(GTK_WIDGET((*l_priv).m_popover));
      (*l_priv).m_popover= 0;
    }
#endif

    if (event_none == (*l_priv).m_sticky)
    {
      break;
    }

    gtk_widget_get_allocation(i_widget, &l_alloc);
    item_selected_normalize(&(*l_priv).m_item, l_alloc.width, l_alloc.height);

    l_cursor= gdk_cursor_new_for_display(gdk_display_get_default(), GDK_ARROW);
    gdk_window_set_cursor((*l_priv).m_event_window, l_cursor);
    g_object_unref(l_cursor);

    gtk_widget_queue_draw(GTK_WIDGET(l_page));

  }while(0);

  (*l_priv).m_drag_type= drag_type_none;
  (*l_priv).m_sticky= event_none;
  (*l_priv).m_event.m_pos_x= (*i_event).x;
  (*l_priv).m_event.m_pos_y= (*i_event).y;

  return 1;
}

static gboolean
iwr_page_motion_notify2(
  GtkWidget*                            i_widget,
  GdkEventMotion*                       i_event)
{
  double                                l_delta_y;
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv; 

  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;

  l_delta_y= (*i_event).y - (*l_priv).m_event.m_pos_y;

  (*l_priv).m_paper.m_height+= (1.0 / (*l_priv).m_scale) * l_delta_y;

  if (0.0 > (*l_priv).m_paper.m_height)
  {
    (*l_priv).m_paper.m_height= 1.0;
  }

#if GTK_CHECK_VERSION(3,12,0)
  iwr_page_set_popover_text(l_priv);
#endif

  gtk_widget_queue_resize(i_widget);

  return 1;
}

static gboolean
iwr_page_motion_notify(
  GtkWidget*                            i_widget,
  GdkEventMotion*                       i_event)
{
  GdkCursor*                            l_cursor;
  struct geom_point                     l_event;
  struct item_node*                     l_node;
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv; 
  struct item_resize_event              l_resize;
  enum event_type                       l_type;

  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;
  l_node= 0;
  l_type= event_none;

  do
  {

    if (drag_type_resize == (*l_priv).m_drag_type)
    {
      iwr_page_motion_notify2(i_widget, i_event);
      break;
    }

    l_event.m_pos_x= (*i_event).x;
    l_event.m_pos_y= (*i_event).y;

    if (drag_type_none == (*l_priv).m_drag_type)
    {

      item_node_in_event(
        &l_node,
        &l_type,
        &(*l_priv).m_item,
        &l_event,
        (*l_priv).m_scale);

      if ((event_none == l_type) && (section_label != (*l_priv).m_type))
      {
        iwr_page_in_resize_event(&l_type, i_widget, (*i_event).x, (*i_event).y);
      }

      if (l_type == (*l_priv).m_sticky)
      {
        break;
      }

      l_cursor= gdk_cursor_new_for_display(
        gdk_display_get_default(),
        (GdkCursorType)l_type);

      gdk_window_set_cursor((*l_priv).m_event_window, l_cursor);
      g_object_unref(l_cursor);

      (*l_priv).m_sticky= l_type;

      break;
    }

    l_event.m_pos_x= ((*i_event).x - (*l_priv).m_event.m_pos_x);
    l_event.m_pos_y= ((*i_event).y - (*l_priv).m_event.m_pos_y);
    l_event.m_pos_x*= (1.00 / (*l_priv).m_scale);
    l_event.m_pos_y*= (1.00 / (*l_priv).m_scale);

    if (event_move == (*l_priv).m_sticky)
    {
      item_selected_move(&(*l_priv).m_item, &l_event);
    }
    else
    {
      l_resize.m_event= l_event;
      l_resize.m_type= (*l_priv).m_sticky;
      item_selected_resize(&(*l_priv).m_item, &l_resize);
    }

    gtk_widget_queue_draw(i_widget);

  }while(0);

  (*l_priv).m_event.m_pos_x= (*i_event).x;
  (*l_priv).m_event.m_pos_y= (*i_event).y;

  return 1;
}

static void
iwr_page_get_preferred_width(
  GtkWidget*                            i_widget,
  gint*                                 o_minimum_size,
  gint*                                 o_natural_size)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= IWR_PAGE(i_widget)->m_priv;

  (*o_minimum_size)= ((*l_priv).m_paper.m_width * (*l_priv).m_scale);
  (*o_natural_size)= (*o_minimum_size);

  return;
}

static void
iwr_page_get_preferred_height(
  GtkWidget *                           i_widget,
  gint *                                o_minimum_size,
  gint *                                o_natural_size)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= IWR_PAGE(i_widget)->m_priv;

  (*o_minimum_size)= ((*l_priv).m_paper.m_height * (*l_priv).m_scale);
  (*o_natural_size)= (*o_minimum_size);

  return;
}

static void
iwr_page_finalize(
  GObject*                              i_object)
{

  G_OBJECT_CLASS(iwr_page_parent_class)->finalize(i_object);

  return;
}

static void
iwr_page_realize(
  GtkWidget*                            i_widget)
{
  IwrPagePrivate*                       l_priv; 
  GdkWindow*                            l_win;
  GdkWindowAttr                         l_attr;
  gint                                  l_attr_mask;
  GtkAllocation                         l_alloc;

  l_priv= IWR_PAGE(i_widget)->m_priv;
  memset(&l_attr, 0, sizeof(l_attr));
  memset(&l_alloc, 0, sizeof(l_alloc));

  GTK_WIDGET_CLASS(iwr_page_parent_class)->realize(i_widget);

  gtk_widget_get_allocation(i_widget, &l_alloc);

  l_attr.x= l_alloc.x;
  l_attr.y= l_alloc.y;
  l_attr.width= l_alloc.width;
  l_attr.height= l_alloc.height;
  l_attr.wclass= GDK_INPUT_ONLY;
  l_attr.window_type= GDK_WINDOW_CHILD;
  l_attr_mask= (GDK_WA_X | GDK_WA_Y);
  l_attr.event_mask= gtk_widget_get_events(i_widget);
  l_attr.event_mask|=
    (GDK_KEY_PRESS_MASK |
    GDK_POINTER_MOTION_MASK |
    GDK_BUTTON_MOTION_MASK |
    GDK_BUTTON_PRESS_MASK |
    GDK_BUTTON_RELEASE_MASK|
    GDK_ENTER_NOTIFY_MASK);

  l_win= gtk_widget_get_window(i_widget),
  (*l_priv).m_event_window= gdk_window_new(l_win, &l_attr, l_attr_mask);
  gtk_widget_register_window(i_widget, (*l_priv).m_event_window);

  return;
}

static void
iwr_page_unrealize(
  GtkWidget*                          i_widget)
{
  IwrPagePrivate *                    l_priv; 

  do
  {

    l_priv= IWR_PAGE(i_widget)->m_priv;

    if (0 ==(*l_priv).m_event_window)
    {
      break;
    }

    gtk_widget_unregister_window(i_widget, (*l_priv).m_event_window);
    gdk_window_destroy((*l_priv).m_event_window);
    (*l_priv).m_event_window= 0;

  }while(0);

  GTK_WIDGET_CLASS(iwr_page_parent_class)->unrealize(i_widget);

  return;
}

static void
iwr_page_destroy(
  GtkWidget*                            i_widget)
{
  IwrPage*                              l_page;

  l_page= IWR_PAGE(i_widget);

  item_discharge(&(*l_page).m_priv->m_item);

  if ((*l_page).m_priv->m_popup)
  {
    gtk_widget_destroy(GTK_WIDGET((*l_page).m_priv->m_popup));
  }

  GTK_WIDGET_CLASS (iwr_page_parent_class)->destroy(i_widget);

  return;
}

extern GtkWidget *
iwr_page_new()
{
  GObject*                              l_gobject;

  l_gobject= g_object_new(IWR_TYPE_PAGE, 0);

  return GTK_WIDGET(l_gobject);
}

extern int
iwr_page_read(
  GError**                              o_error,
  IwrPage*const                         io_page,
  struct bson_node const*const          i_element)
{
  struct bson_node const*               l_element;
  int                                   l_exit;
  int                                   l_found;
  IwrPagePrivate*                       l_priv; 
  enum element_tag_type                 l_type;

  l_element= i_element;
  l_exit= 0;
  l_priv= (*io_page).m_priv;

  do
  {

    if (0 == l_element)
    {
      break;
    }

    do
    {

      if (bson_type_array == (*l_element).m_type)
      {
        l_exit= item_node_read(o_error, &(*l_priv).m_item, l_element);
        break;
      }

      item_common_lookup(&l_found, &l_type, (*l_element).m_ename);

      if (0 == l_found)
      {
        break;
      }

      switch(l_type)
      {
        case element_tag_type:
          if (bson_type_int32 == (*l_element).m_type)
          {
            (*l_priv).m_type= (enum iwr_page_section_type)
              (*l_element).m_object.m_int32;
          }
          break;
        default:
          break;
      }

    }while(0);
    
    l_element= (*l_element).m_next;

  }while(1);

  return l_exit;
}

static void
iwr_page_draw_margin_cover(
  cairo_t*                              io_cr,
  IwrPage const*const                   io_page)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;

  cairo_set_line_width(io_cr, 1.0);
  item_common_cairo_dash_set(io_cr, line_style_dash);
  cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0);
  cairo_move_to(io_cr, (*l_priv).m_paper.m_margin_left, (*l_priv).m_paper.m_margin_top);
  cairo_line_to(io_cr, ((*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right), (*l_priv).m_paper.m_margin_top);
  cairo_line_to(io_cr, ((*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right), ((*l_priv).m_paper.m_height - (*l_priv).m_paper.m_margin_bottom));
  cairo_line_to(io_cr, (*l_priv).m_paper.m_margin_left, ((*l_priv).m_paper.m_height - (*l_priv).m_paper.m_margin_bottom));
  cairo_line_to(io_cr, (*l_priv).m_paper.m_margin_left, (*l_priv).m_paper.m_margin_top);
  cairo_stroke(io_cr);

  return;
}

static void
iwr_page_draw_margin_header(
  cairo_t*                              io_cr,
  IwrPage const*const                   io_page)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;

  cairo_set_line_width(io_cr, 1.0);
  item_common_cairo_dash_set(io_cr, line_style_dash);
  cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0);
  cairo_move_to(io_cr, (*l_priv).m_paper.m_margin_left, (*l_priv).m_paper.m_height);
  cairo_line_to(io_cr, (*l_priv).m_paper.m_margin_left, (*l_priv).m_paper.m_margin_top);
  cairo_line_to(io_cr, ((*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right), (*l_priv).m_paper.m_margin_top);
  cairo_line_to(io_cr, (*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right, (*l_priv).m_paper.m_height);
  cairo_stroke(io_cr);

  return;
}

static void
iwr_page_draw_margin_detail(
  cairo_t*                              io_cr,
  IwrPage const*const                   io_page)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;

  cairo_set_line_width(io_cr, 1.0);
  item_common_cairo_dash_set(io_cr, line_style_dash);
  cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0);

  cairo_move_to(io_cr, (*l_priv).m_paper.m_margin_left, 0);
  cairo_line_to(io_cr, (*l_priv).m_paper.m_margin_left, (*l_priv).m_paper.m_height);
  cairo_move_to(io_cr, ((*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right), 0);
  cairo_line_to(io_cr, ((*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right), (*l_priv).m_paper.m_height);
  cairo_stroke(io_cr);

  return;
}

static void
iwr_page_draw_margin_footer(
  cairo_t*                              io_cr,
  IwrPage *const                        io_page)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;

  cairo_set_line_width(io_cr, 1.0);
  item_common_cairo_dash_set(io_cr, line_style_dash);
  cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0);
  cairo_move_to(io_cr, (*l_priv).m_paper.m_margin_left, 0);
  cairo_line_to(io_cr, (*l_priv).m_paper.m_margin_left, ((*l_priv).m_paper.m_height - (*l_priv).m_paper.m_margin_bottom)); 
  cairo_line_to(io_cr, ((*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right), ((*l_priv).m_paper.m_height - (*l_priv).m_paper.m_margin_bottom)); 
  cairo_line_to(io_cr, ((*l_priv).m_paper.m_width - (*l_priv).m_paper.m_margin_right), 0);
  cairo_stroke(io_cr);

  return;
}

static gboolean
iwr_page_draw(
  GtkWidget*                            i_widget,
  cairo_t*                              io_cr)
{
  double                                l_end_y; 
  double                                l_end_x; 
  double                                l_has_focus;
  double                                l_len_x; 
  double                                l_len_y; 
  struct item_draw_options              l_option;
  double                                l_pos_y; 
  double                                l_pos_x; 
  IwrPage*                              l_page;
  IwrPagePrivate*                       l_priv; 

  memset(&l_option, 0, sizeof(l_option));

  l_page= IWR_PAGE(i_widget);
  l_priv= (*l_page).m_priv;

  l_has_focus= gtk_widget_has_focus(i_widget);

  cairo_save(io_cr);
  cairo_scale(io_cr, (*l_priv).m_scale, (*l_priv).m_scale);

  /* page fill */
  if (l_has_focus)
  {
    cairo_set_source_rgb(io_cr, 1.0, 1.0, 1.0);
  }
  else
  {
    cairo_set_source_rgb(io_cr, 0.8, 0.8, 0.8);
  }
  cairo_rectangle(io_cr, 0, 0, (*l_priv).m_paper.m_width, (*l_priv).m_paper.m_height);
  cairo_fill(io_cr);

  /* page border */
  cairo_set_line_width(io_cr, 1.0);
  cairo_set_source_rgb(io_cr, 0.0, 0.0, 0.0);
  cairo_rectangle(io_cr, 0, 0, (*l_priv).m_paper.m_width, (*l_priv).m_paper.m_height);
  cairo_stroke(io_cr);

  switch((*l_priv).m_type)
  {
    case section_cover_header:
      iwr_page_draw_margin_cover(io_cr, l_page);
      break;
    case section_cover_footer:
      iwr_page_draw_margin_cover(io_cr, l_page);
      break;
    case section_report_header:
      iwr_page_draw_margin_header(io_cr, l_page);
      break;
    case section_report_detail:
      iwr_page_draw_margin_detail(io_cr, l_page);
      break;
    case section_report_footer:
      iwr_page_draw_margin_footer(io_cr, l_page);
      break;
    case section_group_header:
      iwr_page_draw_margin_detail(io_cr, l_page);
      break;
    case section_group_footer:
      iwr_page_draw_margin_detail(io_cr, l_page);
      break;
    default:
      break;
  }

  if ((*l_priv).m_horizontal_line_show)
  {
    l_len_x= (*l_priv).m_paper.m_width; 
    l_end_y= (*l_priv).m_paper.m_height;
    l_pos_y= 0.0;
    do
    {    

      l_pos_y+= 18.0;

      if (l_end_y <= l_pos_y)
      {
        break;
      }

      cairo_set_line_width(io_cr, 1.0);
      cairo_set_source_rgb(io_cr, 0.447, 0.623, 0.811);
      cairo_move_to(io_cr, 0.0, l_pos_y);
      cairo_line_to(io_cr, l_len_x, l_pos_y);
      cairo_stroke(io_cr);

    }while(1);
  }

  if ((*l_priv).m_vertical_line_show)
  {
    l_len_y= (*l_priv).m_paper.m_height; 
    l_end_x= (*l_priv).m_paper.m_width;
    l_pos_x= 0.0;
    do
    {    

      l_pos_x+= 18.0;

      if (l_end_x <= l_pos_x)
      {
        break;
      }

      cairo_set_line_width(io_cr, 1.0);
      cairo_set_source_rgb(io_cr, 0.447, 0.623, 0.811);
      cairo_move_to(io_cr, l_pos_x, 0.0);
      cairo_line_to(io_cr, l_pos_x, l_len_y);
      cairo_stroke(io_cr);

    }while(1);
  }

  l_option.m_type= item_draw_design;
  l_option.m_highlight= (*l_priv).m_highlight_objects;
  item_draw(&(*l_priv).m_item, io_cr, &l_option);

  cairo_restore(io_cr);

  return 0;
}

static void
iwr_page_map(
  GtkWidget*                              i_widget)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= IWR_PAGE(i_widget)->m_priv;

  GTK_WIDGET_CLASS(iwr_page_parent_class)->map(i_widget);

  gdk_window_show(l_priv->m_event_window);

  return;
}

static void
iwr_page_unmap(
  GtkWidget*                              i_widget)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= IWR_PAGE(i_widget)->m_priv;

  gdk_window_hide(l_priv->m_event_window);

  GTK_WIDGET_CLASS(iwr_page_parent_class)->unmap(i_widget);

  return;
}

static void
iwr_page_size_allocate(
  GtkWidget*                            i_widget,
  GtkAllocation*                        i_allocation)
{
  IwrPagePrivate*                       l_priv; 
  int                                   l_rc;

  l_priv= IWR_PAGE(i_widget)->m_priv;

  gtk_widget_set_allocation(i_widget, i_allocation);

  do
  {

    l_rc= gtk_widget_get_realized(i_widget);

    if (0 ==l_rc)
    {
      break;
    }

    gdk_window_move_resize (
      (*l_priv).m_event_window,
      i_allocation->x, 
      i_allocation->y,
      i_allocation->width, 
      i_allocation->height);

  }while(0);

  return;
}

static void
iwr_page_class_init (
  IwrPageClass *                        i_klass)
{
  GObjectClass*                         l_gobject_class;
  GtkWidgetClass*                       l_widget_class;

  l_gobject_class= G_OBJECT_CLASS(i_klass);
  (*l_gobject_class).finalize= iwr_page_finalize;

  l_widget_class= GTK_WIDGET_CLASS(i_klass);

  (*l_widget_class).size_allocate= iwr_page_size_allocate;
  (*l_widget_class).map= iwr_page_map;
  (*l_widget_class).unmap= iwr_page_unmap;
  (*l_widget_class).destroy= iwr_page_destroy;
  (*l_widget_class).realize= iwr_page_realize;
  (*l_widget_class).unrealize= iwr_page_unrealize;
  (*l_widget_class).get_preferred_width= iwr_page_get_preferred_width;
  (*l_widget_class).get_preferred_height= iwr_page_get_preferred_height;
  (*l_widget_class).motion_notify_event= iwr_page_motion_notify;
  (*l_widget_class).button_press_event= iwr_page_button_press;
  (*l_widget_class).button_release_event= iwr_page_button_release;
  (*l_widget_class).key_press_event= iwr_page_key_press;
  (*l_widget_class).draw= iwr_page_draw;

  return;
}

static void
iwr_page_init (
  IwrPage *                             o_page)
{
  IwrPagePrivate*                       l_priv; 

  IWR_PAGE(o_page)->m_priv= iwr_page_get_instance_private(o_page);
  l_priv= IWR_PAGE(o_page)->m_priv;
  memset(l_priv, 0, sizeof(*l_priv));

  gtk_widget_set_can_focus(GTK_WIDGET(o_page), TRUE);
  gtk_widget_set_has_window(GTK_WIDGET(o_page), FALSE);

  (*l_priv).m_paper.m_width= POINTS_PER_INCH * PAPER_DEFAULT_WIDTH;
  (*l_priv).m_paper.m_height= POINTS_PER_INCH * PAPER_DEFAULT_HEIGHT;
  (*l_priv).m_paper.m_margin_top= POINTS_PER_INCH * PAPER_DEFAULT_MARGIN;
  (*l_priv).m_paper.m_margin_bottom= POINTS_PER_INCH * PAPER_DEFAULT_MARGIN;
  (*l_priv).m_paper.m_margin_left= POINTS_PER_INCH * PAPER_DEFAULT_MARGIN;
  (*l_priv).m_paper.m_margin_right= POINTS_PER_INCH * PAPER_DEFAULT_MARGIN;

  item_assign(&(*l_priv).m_item);
  (*l_priv).m_event.m_pos_x= 0.00;
  (*l_priv).m_event.m_pos_y= 0.00;
  (*l_priv).m_sticky= event_none;
  (*l_priv).m_scale= 1.00;

  return;
}

extern void
iwr_page_set_highlight_objects(
  IwrPage*const                         io_page,
  gboolean const                        i_state)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;
  (*l_priv).m_highlight_objects= i_state;
  gtk_widget_queue_draw(GTK_WIDGET(io_page));

  return;
}

extern void
iwr_page_set_paper(
  IwrPage*const                         io_page,
  struct paper const*const              i_paper)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;

  if ((*l_priv).m_paper.m_width > (*i_paper).m_width ||
    (*l_priv).m_paper.m_height > (*i_paper).m_height)
  {
    item_normalize(&(*l_priv).m_item, (*i_paper).m_width, (*i_paper).m_height);
  }

  memcpy(&(*l_priv).m_paper, i_paper, sizeof(*i_paper));

  gtk_widget_queue_resize(GTK_WIDGET(io_page));

  return;
}

extern void
iwr_page_set_vertical_ruler_line(
  IwrPage*const                         io_page,
  gboolean const                        i_state)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;
  (*l_priv).m_vertical_line_show= i_state;
  gtk_widget_queue_draw(GTK_WIDGET(io_page));

  return;
}

extern void
iwr_page_set_horizontal_ruler_line(
  IwrPage*const                         io_page,
  gboolean const                        i_state)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;
  (*l_priv).m_horizontal_line_show= i_state;
  gtk_widget_queue_draw(GTK_WIDGET(io_page));

  return;
}

extern void
iwr_page_set_scale(
  IwrPage*const                         io_page,
  double const                          i_scale)
{
  IwrPagePrivate*                       l_priv; 

  l_priv= (*io_page).m_priv;
  (*l_priv).m_scale= i_scale;
  gtk_widget_queue_resize(GTK_WIDGET(io_page));

  return;
}

extern void
iwr_page_set_section_type(
  IwrPage*const                         io_page,
  enum iwr_page_section_type const      i_type)
{

  (*io_page).m_priv->m_type= i_type;
  gtk_widget_queue_resize(GTK_WIDGET(io_page));

  return;
}

extern int
iwr_page_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  IwrPage const*const                   i_page)
{
  int                                   l_exit;

  l_exit= item_node_write(o_error, io_fp, &(*i_page).m_priv->m_item);

  return l_exit;
}
