/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_line.h"

struct line_dialog
{
  GtkEntry*                             m_color_entry;
  GtkButton*                            m_color_button;
  GtkComboBox*                          m_style_comboboxtext;
  GtkSpinButton*                        m_thickness_spinbutton;
  GtkRadioButton*                       m_horizontal_radiobutton;
  GtkRadioButton*                       m_vertical_radiobutton;
  GtkRadioButton*                       m_diagonal_radiobutton;
  GtkCheckButton*                       m_double_line_checkbutton;
};

static void
item_line_color_button(
  GtkWidget *                           i_widget,
  gpointer                              i_data)
{
  GdkRGBA                               l_color;
  GtkWidget*                            l_dialog;
  int                                   l_exit;
  struct line_dialog*                  l_line;
  gchar*                                l_rgba;
  gchar const*                          l_text;

  l_dialog= 0;
  memset(&l_color, 0, sizeof(l_color));

  do
  {

    l_line= (struct line_dialog*)i_data;

    l_dialog= gtk_color_chooser_dialog_new("Border Color", mainw());

    l_text= gtk_entry_get_text((*l_line).m_color_entry);
    gdk_rgba_parse(&l_color, l_text);
    gtk_color_chooser_set_rgba(GTK_COLOR_CHOOSER(l_dialog), &l_color);

    gtk_window_set_modal((GtkWindow*)l_dialog, 1);

    l_exit= gtk_dialog_run(GTK_DIALOG(l_dialog));

    if (GTK_RESPONSE_OK != l_exit)
    {
      break;
    }

    gtk_color_chooser_get_rgba(GTK_COLOR_CHOOSER(l_dialog), &l_color);

    l_rgba= gdk_rgba_to_string(&l_color);
    gtk_entry_set_text((*l_line).m_color_entry, l_rgba);
    g_free(l_rgba);

  }while(0);

  if (l_dialog)
  {
    gtk_widget_destroy((GtkWidget*)l_dialog);
  }
  
  return;
}

static void
item_line_bind(
  struct line_dialog*const              o_line,
  GtkBuilder*const                      i_builder)
{

  memset(o_line, 0, sizeof(*o_line));

  do
  {

    (*o_line).m_color_entry=
      GTK_ENTRY(gtk_builder_get_object(i_builder, "color_entry"));

    (*o_line).m_color_button=
      GTK_BUTTON(gtk_builder_get_object(i_builder, "color_button"));

    g_signal_connect(
      (*o_line).m_color_button,
      "clicked",
      G_CALLBACK(item_line_color_button),
      o_line);

    (*o_line).m_style_comboboxtext=
      GTK_COMBO_BOX(gtk_builder_get_object(i_builder, "style_comboboxtext"));

    (*o_line).m_thickness_spinbutton=
      GTK_SPIN_BUTTON(gtk_builder_get_object(i_builder, "thickness_spinbutton"));

    gtk_spin_button_set_range((*o_line).m_thickness_spinbutton, 1.0, 30.0);

    (*o_line).m_horizontal_radiobutton=
      GTK_RADIO_BUTTON(gtk_builder_get_object(i_builder, "horizontal_radiobutton"));

    (*o_line).m_vertical_radiobutton=
      GTK_RADIO_BUTTON(gtk_builder_get_object(i_builder, "vertical_radiobutton"));

    (*o_line).m_diagonal_radiobutton=
      GTK_RADIO_BUTTON(gtk_builder_get_object(i_builder, "diagonal_radiobutton"));

    (*o_line).m_double_line_checkbutton=
      GTK_CHECK_BUTTON(gtk_builder_get_object(i_builder, "double_line_checkbutton"));

  }while(0);

  return;
}

static void
item_line_set_values(
  struct line_dialog*const             i_lined,
  struct item_line const*const         i_line)
{
  gchar *                               l_color;

  l_color= gdk_rgba_to_string(&(*i_line).m_color);
  gtk_entry_set_text((*i_lined).m_color_entry, l_color);
  g_free(l_color);
  l_color= 0;

  gtk_combo_box_set_active(
      GTK_COMBO_BOX((*i_lined).m_style_comboboxtext), 
      (*i_line).m_style);

  gtk_spin_button_set_value(
    GTK_SPIN_BUTTON((*i_lined).m_thickness_spinbutton),
    (*i_line).m_thickness);

  if (line_type_horizontal == (*i_line).m_type)
  {
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON((*i_lined).m_horizontal_radiobutton),1);
  }
  else if (line_type_vertical == (*i_line).m_type)
  {
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON((*i_lined).m_vertical_radiobutton), 1);
  }
  else
  {
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON((*i_lined).m_diagonal_radiobutton), 1);
  }

  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON((*i_lined).m_double_line_checkbutton),
    (*i_line).m_double_line);

  return;
}

static void
item_line_copy_values(
  struct item_line *const              io_line,
  struct line_dialog const*const       i_lined)
{
  gint                                  l_rc;
  int                                   l_stride;
  gchar const*                          l_text;

  l_text= gtk_entry_get_text(GTK_ENTRY((*i_lined).m_color_entry));
  gdk_rgba_parse(&(*io_line).m_color, l_text);

  l_rc = gtk_combo_box_get_active(
    GTK_COMBO_BOX((*i_lined).m_style_comboboxtext));

  if ((0 <= l_rc) && (LINE_STYLE_SLOTS > l_rc))
  {
    (*io_line).m_style= (enum line_style_type)l_rc;
  }
  else
  {
    (*io_line).m_style= line_style_solid;
  }

  (*io_line).m_thickness= gtk_spin_button_get_value_as_int(
    GTK_SPIN_BUTTON((*i_lined).m_thickness_spinbutton));

  l_rc= gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON((*i_lined).m_horizontal_radiobutton));

  if (l_rc)
  {
    if (line_type_horizontal != (*io_line).m_type)
    {
      (*io_line).m_type= line_type_horizontal;
      l_stride= (*io_line).m_pos_y2 - (*io_line).m_pos_y1;
      (*io_line).m_pos_x1= (*io_line).m_pos_x2 - l_stride;
      (*io_line).m_pos_y1= (*io_line).m_pos_y2;
    }
  }
  else
  {
    l_rc= gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON((*i_lined).m_vertical_radiobutton));
    if (l_rc)
    {
      if (line_type_vertical != (*io_line).m_type)
      {
        (*io_line).m_type= line_type_vertical;
        l_stride= (*io_line).m_pos_y2 - (*io_line).m_pos_y1;
        (*io_line).m_pos_y1= (*io_line).m_pos_y2 - l_stride;
        (*io_line).m_pos_x1= (*io_line).m_pos_x2;
      }
    }
    else
    {
      (*io_line).m_type= line_type_diagonal;
    }
  }

  (*io_line).m_double_line= gtk_toggle_button_get_active(
    GTK_TOGGLE_BUTTON((*i_lined).m_double_line_checkbutton));

  return;
}

extern int
item_line_property(
  struct item_line*const               io_line)
{
  GtkBox*                               l_box;
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GtkGrid*                              l_grid;
  GError*                               l_error;
  int                                   l_exit;
  struct line_dialog                    l_line;
  int                                   l_rc;

  l_builder= 0;
  l_dialog= 0;
  l_grid= 0;
  l_error= 0;
  l_exit= 0;

  do
  {

    l_builder= gtk_builder_new();

    l_rc= wrap_gtk_builder_add_from_file(l_builder, "line.glade", &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      break;
    }
  
    l_grid= GTK_GRID(gtk_builder_get_object(l_builder, "line_grid"));

    if (0 == l_grid)
    {
      l_error= g_error_new(
        GENERAL,
        GENERIC,
        "Unable to find dialog object: 'line_grid'");
      l_exit= -1;
      break;
    }

    item_line_bind(&l_line, l_builder);

    item_line_set_values(&l_line, io_line);

    l_dialog= GTK_DIALOG(gtk_dialog_new_with_buttons(
      "Line Edit",
      mainw(),
      GTK_DIALOG_MODAL,
      "Ok",
      GTK_RESPONSE_OK,
      "Cancel",
      GTK_RESPONSE_CANCEL,
      NULL));

    l_box= GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(l_dialog)));

    gtk_box_pack_start(GTK_BOX(l_box), GTK_WIDGET(l_grid), 1, 1,0);

    gtk_widget_show(GTK_WIDGET(l_grid));

    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);
    l_exit= gtk_dialog_run(l_dialog);

    if (GTK_RESPONSE_OK != l_exit)
    {
      break;
    }

    item_line_copy_values(io_line, &l_line);

  }while(0);

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  if (l_error)
  {
    _error_log(l_error);
    _error_display_prompt(l_error);
  }

  g_clear_error(&l_error);

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  return l_exit;
}
