/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_line.h"

extern void
item_line_assign(
  struct item_line*const                o_line,
  struct item_method*const              o_method)
{

  memset(o_line, 0, sizeof(*o_line));

  (*o_line).m_pos_x1= 10.0;
  (*o_line).m_pos_y1= 10.0;
  (*o_line).m_pos_x2= 100.0;
  (*o_line).m_pos_y2= 10.0;
  (*o_line).m_style= line_style_solid;
  (*o_line).m_thickness= 1;

  ITEM_METHOD_ASSIGN(o_method, line);

  return;
}

extern void
item_line_discharge(
  struct item_line*const                io_line)
{
  memset(io_line, 0, sizeof(*io_line));
}

extern void
item_line_draw(
  struct item_line*const                i_line,
  struct item_draw_context const*const  i_ctx)
{

  do
  {

    cairo_set_source_rgb(
      (*i_ctx).m_cr,
      (*i_line).m_color.red,
      (*i_line).m_color.green,
      (*i_line).m_color.blue);

    item_common_cairo_dash_set((*i_ctx).m_cr, (*i_line).m_style);
    cairo_set_line_width((*i_ctx).m_cr, (*i_line).m_thickness);

    cairo_move_to((*i_ctx).m_cr, (*i_line).m_pos_x1, (*i_line).m_pos_y1);
    cairo_line_to((*i_ctx).m_cr, (*i_line).m_pos_x2, (*i_line).m_pos_y2);
    cairo_stroke((*i_ctx).m_cr);

    if ((*i_line).m_double_line)
    {
      switch((*i_line).m_type)
      {
        case line_type_horizontal:
          cairo_move_to((*i_ctx).m_cr, (*i_line).m_pos_x1, 3.00+(*i_line).m_pos_y1);
          cairo_line_to((*i_ctx).m_cr, (*i_line).m_pos_x2, 3.00+(*i_line).m_pos_y2);
          cairo_stroke((*i_ctx).m_cr);
          break;
        case line_type_vertical:
          cairo_move_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x1, (*i_line).m_pos_y1);
          cairo_line_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x2, (*i_line).m_pos_y2);
          cairo_stroke((*i_ctx).m_cr);
          break;
        case line_type_diagonal:
          cairo_move_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x1, 3.00+(*i_line).m_pos_y1);
          cairo_line_to((*i_ctx).m_cr, 3.00+(*i_line).m_pos_x2, 3.00+(*i_line).m_pos_y2);
          cairo_stroke((*i_ctx).m_cr);
        default:
          break;
      }
    }

    if (0 == (*i_ctx).m_selected)
    {
      break;
    }

    cairo_set_source_rgb((*i_ctx).m_cr, 0.0, 1.0, 0.0);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_line).m_pos_x1-4,
      (*i_line).m_pos_y1-4,
      8,
      8);
    cairo_fill((*i_ctx).m_cr);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_line).m_pos_x2-4,
      (*i_line).m_pos_y2-4,
      8,
      8);
    cairo_fill((*i_ctx).m_cr);

  }while(0);

  return;
}

extern void
item_line_get_bbox(
  struct geom_rect*const                o_bbox,
  int*const                             o_overlay,
  void const*const                      i_object)
{
  struct item_line const*               l_line;

  l_line= (struct item_line const*)i_object;
  (*o_bbox).m_pos_x= (*l_line).m_pos_x1;
  (*o_bbox).m_pos_y= (*l_line).m_pos_y1;
  (*o_bbox).m_len_x= (*l_line).m_pos_x2 - (*l_line).m_pos_x1;
  (*o_bbox).m_len_y= (int)(*l_line).m_thickness;
  (*o_overlay)= 0;

  return;
}

static int
item_line_in_event_grips(
  enum event_type*const                 o_type,
  struct item_line const*const          i_line,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {

    l_hit= item_common_event_inside(
      i_ctx,
      (*i_line).m_pos_x1-4,
      (*i_line).m_pos_y1-4,
      8,
      8);

    if (l_hit)
    {
      if (line_type_horizontal == (*i_line).m_type)
      {
        (*o_type)= event_resize_west;
      }
      else if (line_type_vertical == (*i_line).m_type)
      {
        (*o_type)= event_resize_north;
      }
      else
      {
        (*o_type)= event_resize_north_west;
      }
      break;
    }
    
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_line).m_pos_x2-4,
      (*i_line).m_pos_y2-4,
      8,
      8);

    if (l_hit)
    {
      if (line_type_horizontal == (*i_line).m_type)
      {
        (*o_type)= event_resize_east;
      }
      else if (line_type_vertical == (*i_line).m_type)
      {
       (*o_type)= event_resize_south;
      }
      else
      {
        (*o_type)= event_resize_south_east;
      }
      break;
    }

  }while(0);

  return l_hit;
}

extern void
item_line_in_event(
  enum event_type*const                 o_type,
  struct item_line const*const          i_line,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {

    (*o_type)= event_none;

    l_hit= item_common_line_in_event(i_ctx, (struct geom_line*)i_line);

    if (0 == l_hit)
    {
      break;
    }

    if ((*i_ctx).m_selected)
    {
      l_hit= item_line_in_event_grips(o_type, i_line, i_ctx);

      if (l_hit)
      {
        break;
      }
    }

    (*o_type)= event_move;

  }while(0);

  return;
}

extern void
item_line_move(
  struct item_line*const                io_line,
  struct geom_point const*const         i_event)
{

  (*io_line).m_pos_x1+= (*i_event).m_pos_x;
  (*io_line).m_pos_y1+= (*i_event).m_pos_y;
  (*io_line).m_pos_x2+= (*i_event).m_pos_x;
  (*io_line).m_pos_y2+= (*i_event).m_pos_y;

  return;
}

extern void
item_line_normalize(
  struct item_line*const                io_line,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  gdouble                               l_swap;

  do
  {

    if (line_type_diagonal == (*io_line).m_type)
    {
      break;
    }

    if ((*io_line).m_pos_x1 > (*io_line).m_pos_x2)
    {
      l_swap= (*io_line).m_pos_x1;
      (*io_line).m_pos_x1= (*io_line).m_pos_x2;
      (*io_line).m_pos_x2= l_swap;
    }

    if ((*io_line).m_pos_y1 > (*io_line).m_pos_y2)
    {
      l_swap= (*io_line).m_pos_y1;
      (*io_line).m_pos_y1= (*io_line).m_pos_y2;
      (*io_line).m_pos_y2= l_swap;
    }


    if (0 > (int)(*io_line).m_pos_x2)
    {
      l_swap= (*io_line).m_pos_x2 - (*io_line).m_pos_x1;
      (*io_line).m_pos_x2= 2.0;
      (*io_line).m_pos_x1= (*io_line).m_pos_x2 - l_swap;
    }

    if (0 > (int)(*io_line).m_pos_y2)
    {
      l_swap= (*io_line).m_pos_y2 - (*io_line).m_pos_y1;
      (*io_line).m_pos_y1= 2.0 - l_swap;
      (*io_line).m_pos_y2= (*io_line).m_pos_y1 + l_swap;
    }

    if (i_bound_width < (int)(*io_line).m_pos_x1)
    {
      l_swap= (*io_line).m_pos_x2 - (*io_line).m_pos_x1;
      (*io_line).m_pos_x1= i_bound_width - 2.0;
      (*io_line).m_pos_x2= (*io_line).m_pos_x1 + l_swap;
    }

    if (i_bound_height < (int)(*io_line).m_pos_y1)
    {
      l_swap= (*io_line).m_pos_y2 - (*io_line).m_pos_y1;
      (*io_line).m_pos_y1= i_bound_height - 2.0;
      (*io_line).m_pos_y2= (*io_line).m_pos_y1 + l_swap;
    }

  }while(0);

  return;
}

extern int
item_line_read(
  GError**                              o_error,
  struct item_line*const                io_line,
  struct bson_node *const               i_element)
{
  struct bson_node *                    l_element;
  int                                   l_found;
  enum element_tag_type                 l_type;

  l_element= i_element;

  do
  {

    if (0 == l_element)
    {
      break;
    }

    if (bson_type_end == (*l_element).m_type)
    {
      break;
    }

    item_common_lookup(&l_found, &l_type, (*l_element).m_ename);

    if (l_found)
    {
        switch(l_type)
        {
          case element_tag_color_red:
            if (bson_type_float  == (*l_element).m_type)
            {
              (*io_line).m_color.red= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_color_green:
            if (bson_type_float  == (*l_element).m_type)
            {
              (*io_line).m_color.green= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_color_blue:
            if (bson_type_float  == (*l_element).m_type)
            {
              (*io_line).m_color.blue= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_pos_x1:
            if (bson_type_float == (*l_element).m_type)
            {
              (*io_line).m_pos_x1= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_pos_y1:
            if (bson_type_float == (*l_element).m_type)
            {
              (*io_line).m_pos_y1= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_pos_x2:
            if (bson_type_float == (*l_element).m_type)
            {
              (*io_line).m_pos_x2= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_pos_y2:
            if (bson_type_float == (*l_element).m_type)
            {
              (*io_line).m_pos_y2= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_style:
            if (bson_type_int32 == (*l_element).m_type)
            {
              (*io_line).m_style= (*l_element).m_object.m_int32;
            }
            break;
          case element_tag_thickness:
            if (bson_type_float == (*l_element).m_type)
            {
              (*io_line).m_thickness= (*l_element).m_object.m_float;
            }
            break;
          case element_tag_type:
            if (bson_type_int32 == (*l_element).m_type)
            {
              (*io_line).m_type= (*l_element).m_object.m_int32;
            }
            break;
          case element_tag_doubleline:
            if (bson_type_int32 == (*l_element).m_type)
            {
              (*io_line).m_double_line= (*l_element).m_object.m_int32;
            }
            break;
          default:
            break;
        }
    }

    l_element= (*l_element).m_next;

  }while(1);

  return 0;
}

extern void
item_line_resize(
  struct item_line*const                io_line,
  struct item_resize_event const*const  i_ctx)
{
  
  switch((*i_ctx).m_type)
  {
    case event_resize_east:
      (*io_line).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north:
      (*io_line).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_north_east:
      (*io_line).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north_west:
      (*io_line).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south:
      (*io_line).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_south_east:
      (*io_line).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south_west:
      (*io_line).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      (*io_line).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_west:
      (*io_line).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    default:
      break;
  }

  return;
}

extern int
item_line_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_line const*const          i_line)
{
  int                                   l_exit;
  int32_t                               l_int32;
  char                                  l_octet;
  int32_t                               l_offset;
  int32_t                               l_size;

  l_exit= 0;

  /* array */
  l_octet= bson_type_array;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "line%c",0);

  /* document */
  l_offset= ftell(io_fp);
  l_size= 0;
  fwrite(&l_size, sizeof(int32_t), 1, io_fp); 

  /* pos_x1 */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "pos_x1%c",0);
  fwrite(&(*i_line).m_pos_x1, sizeof(double), 1, io_fp);

  /* pos_y1 */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "pos_y1%c",0);
  fwrite(&(*i_line).m_pos_y1, sizeof(double), 1, io_fp);

  /* pos_x2 */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "pos_x2%c",0);
  fwrite(&(*i_line).m_pos_x2, sizeof(double), 1, io_fp);

  /* pos_y2 */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "pos_y2%c",0);
  fwrite(&(*i_line).m_pos_y2, sizeof(double), 1, io_fp);

  /* color - red */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "color_red%c",0);
  fwrite(&(*i_line).m_color.red, sizeof(double), 1, io_fp);

  /* color - green */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "color_green%c",0);
  fwrite(&(*i_line).m_color.green, sizeof(double), 1, io_fp);

  /* color - blue */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "color_blue%c",0);
  fwrite(&(*i_line).m_color.blue, sizeof(double), 1, io_fp);

  /* style */
  l_octet= bson_type_int32;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "style%c",0);
  fwrite(&(*i_line).m_style, sizeof(int32_t), 1, io_fp);

  /* thickness */
  l_octet= bson_type_float;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "thickness%c",0);
  fwrite(&(*i_line).m_thickness, sizeof(double), 1, io_fp);

  /* type */
  l_octet= bson_type_int32;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "type%c",0);
  fwrite(&(*i_line).m_type, sizeof(int32_t), 1, io_fp);

  /* double line */
  l_octet= bson_type_int32;
  fwrite(&l_octet, 1, 1, io_fp); 
  fprintf(io_fp, "double_line%c",0);
  fwrite(&(*i_line).m_double_line, sizeof(int32_t), 1, io_fp);

  /* end of document */
  l_octet= 0;
  fwrite(&l_octet, 1, 1, io_fp); 

  /* document */
  l_int32= ftell(io_fp);
  l_size= (l_int32 - l_offset);
  fseek(io_fp, l_offset, SEEK_SET);
  fwrite(&l_size, sizeof(l_size), 1, io_fp); 


  fseek(io_fp, 0, SEEK_END);

  return l_exit;
}
