/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_image.h"

struct image_dialog
{
  GtkEntry*                             m_color_entry;
  GtkCheckButton*                       m_border_checkbutton;
  GtkComboBoxText*                      m_border_style_combobox;
  GtkSpinButton*                        m_border_thickness_spinbutton;
  GtkCheckButton*                       m_overlay_checkbutton;
  GtkEntry*                             m_filename_entry;
};

static void
item_file_button_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_data)
{
  GtkFileChooser*                       l_chooser;
  GtkWidget*                            l_dialog;
  GtkFileFilter*                        l_filter;
  struct image_dialog*                  l_imaged;
  char*                                 l_imagefile;
  char const*                           l_imagefilec;
  int                                   l_rc;

  l_dialog= 0;
  l_imagefile= 0;
  
  do
  {

    l_imaged= (struct image_dialog*)i_data;

    l_dialog = gtk_file_chooser_dialog_new(
      "Open Report",
      mainw(),
      GTK_FILE_CHOOSER_ACTION_SAVE,
      "_Cancel",
      GTK_RESPONSE_CANCEL,
      "_Save",
      GTK_RESPONSE_ACCEPT,
      NULL);

    l_chooser= GTK_FILE_CHOOSER(l_dialog);

    l_imagefilec= gtk_entry_get_text ((*l_imaged).m_filename_entry);

    if ((l_imagefilec) && (l_imagefilec[0]))
    {
      gtk_file_chooser_set_filename(l_chooser, l_imagefilec);
    }
    else
    {
      gtk_file_chooser_set_current_name(l_chooser, "Untitled");
    }

    l_filter= gtk_file_filter_new();
    gtk_file_filter_add_pattern(l_filter, "*.png");
    gtk_file_filter_add_pattern(l_filter, "*.jpg");
    gtk_file_filter_add_pattern(l_filter, "*.bmp");
    gtk_file_filter_add_pattern(l_filter, "*.tif");
    gtk_file_filter_add_pattern(l_filter, "*.xpm");
    gtk_file_filter_set_name(l_filter, "Image");
    gtk_file_chooser_add_filter(l_chooser, l_filter);

    l_rc= gtk_dialog_run(GTK_DIALOG(l_dialog));

    if (GTK_RESPONSE_ACCEPT != l_rc)
    {
      break;
    }

    l_imagefile= gtk_file_chooser_get_filename(l_chooser);
    gtk_entry_set_text ((*l_imaged).m_filename_entry, l_imagefile);

  }while(0);

  if (l_dialog)
  {
    gtk_widget_destroy(l_dialog); 
  }

  if (l_imagefile)
  {
    g_free(l_imagefile);
  }

  return;
}

static void
item_image_bind(
  struct image_dialog*const             o_image,
  GtkBuilder*const                      i_builder)
{
  GtkButton*                            l_button;

  memset(o_image, 0, sizeof(*o_image));

  do
  {

    (*o_image).m_color_entry=
      GTK_ENTRY(gtk_builder_get_object(i_builder, "color_entry"));

    (*o_image).m_border_checkbutton=
      GTK_CHECK_BUTTON(gtk_builder_get_object(i_builder, "border_checkbutton"));

    (*o_image).m_border_style_combobox=
      GTK_COMBO_BOX_TEXT(gtk_builder_get_object(i_builder, "border_style_comboboxtext"));

    (*o_image).m_border_thickness_spinbutton=
      GTK_SPIN_BUTTON(gtk_builder_get_object(i_builder, "border_thickness_spinbutton"));

    (*o_image).m_overlay_checkbutton=
      GTK_CHECK_BUTTON(gtk_builder_get_object(i_builder, "overlay_checkbutton"));

    (*o_image).m_filename_entry=
      GTK_ENTRY(gtk_builder_get_object(i_builder, "filename_entry"));

    l_button= 
      GTK_BUTTON(gtk_builder_get_object(i_builder, "file_button"));

    g_signal_connect(
      l_button,
      "clicked",
      G_CALLBACK(item_file_button_cb),
      o_image);

  }while(0);

  return;
}

static void
item_image_set_values(
  struct image_dialog*const             i_imaged,
  struct item_image const*const         i_image)
{

  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON((*i_imaged).m_border_checkbutton),
    (*i_image).m_border);

  gtk_combo_box_set_active(
      GTK_COMBO_BOX((*i_imaged).m_border_style_combobox), 
      (*i_image).m_border_style);

  gtk_spin_button_set_value(
    GTK_SPIN_BUTTON((*i_imaged).m_border_thickness_spinbutton),
    (*i_image).m_border_thickness);

  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON((*i_imaged).m_overlay_checkbutton),
    (*i_image).m_overlay);

  return;
}

static void
item_image_copy_values(
  struct item_image *const              io_image,
  struct image_dialog const*const       i_imaged)
{
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  gint                                  l_rc;
  gchar const*                          l_text;

  l_error= 0;
  (*io_image).m_border=
    gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON((*i_imaged).m_border_checkbutton));

  l_rc = gtk_combo_box_get_active(
    GTK_COMBO_BOX((*i_imaged).m_border_style_combobox));

  if ((0 <= l_rc) && (LINE_STYLE_SLOTS > l_rc))
  {
    (*io_image).m_border_style= (enum line_style_type)l_rc;
  }
  else
  {
    (*io_image).m_border_style= line_style_solid;
  }

  (*io_image).m_border_thickness= gtk_spin_button_get_value_as_int(
    GTK_SPIN_BUTTON((*i_imaged).m_border_thickness_spinbutton));

  (*io_image).m_overlay=
    gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON((*i_imaged).m_overlay_checkbutton));

  l_text= gtk_entry_get_text(GTK_ENTRY((*i_imaged).m_filename_entry));

  if (l_text && l_text[0])
  {
    item_image_load(&l_error, io_image, l_text);
  }

  if (l_error)
  {
    l_dialog= GTK_DIALOG(gtk_message_dialog_new(
      mainw(),
      GTK_DIALOG_MODAL,
      GTK_MESSAGE_ERROR,
      GTK_BUTTONS_CLOSE,
      (*l_error).message, 
      0));
    gtk_dialog_run(l_dialog);
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  g_clear_error(&l_error);

  return;
}

extern int
item_image_property(
  struct item_image*const               io_image)
{
  GtkBox*                               l_box;
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  GtkGrid*                              l_grid;
  struct image_dialog                   l_image;
  int                                   l_rc;

  l_builder= 0;
  l_grid= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;

  do
  {

    l_builder= gtk_builder_new();

    l_rc= wrap_gtk_builder_add_from_file(l_builder, "image.glade", &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      break;
    }
  
    l_grid= GTK_GRID(gtk_builder_get_object(l_builder, "image_grid"));

    if (0 == l_grid)
    {
      l_error= g_error_new(
        GENERAL,
        GENERIC,
        "Unable to find dialog object: 'image_grid'");
      l_exit= -1;
      break;
    }

    item_image_bind(&l_image, l_builder);

    item_image_set_values(&l_image, io_image);

    l_dialog= GTK_DIALOG(gtk_dialog_new_with_buttons(
      "Image Edit",
      mainw(),
      GTK_DIALOG_MODAL,
      "Ok",
      GTK_RESPONSE_OK,
      "Cancel",
      GTK_RESPONSE_CANCEL,
      NULL));

    l_box= GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(l_dialog)));

    gtk_box_pack_start(GTK_BOX(l_box), GTK_WIDGET(l_grid), 1, 1,0);

    gtk_widget_show(GTK_WIDGET(l_grid));

    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);
    l_exit= gtk_dialog_run(l_dialog);

    if (GTK_RESPONSE_OK != l_exit)
    {
      break;
    }

    item_image_copy_values(io_image, &l_image);

  }while(0);

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  if (l_error)
  {
    _error_log(l_error);
    _error_display_prompt(l_error);
  }

  g_clear_error(&l_error);

  if (l_dialog)
  {
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  return l_exit;
}
