/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "database_session.h"

extern void
database_session_assign(
  struct database_session*const         o_session)
{

  memset(o_session, 0, sizeof(*o_session));

  iwsql_dll_assign(&(*o_session).m_dll);

  return;
}

extern int
database_session_connect(
  GError**                              o_error,
  struct database_session*const         io_session,
  struct database_aspect const*const    i_aspect)
{
  struct iwsql_aspect                   l_conn;
  struct driver_config                  l_driver;
  GError*                               l_error;
  int                                   l_exit;
  struct driver_module const*           l_module;
  int                                   l_rc;
  unsigned                              l_slot;

  l_exit= 0;
  l_error= 0;

  driver_config_assign(&l_driver);

  do
  {

    if (0 == (*i_aspect).m_driver[0])
    {
      l_error= g_error_new(
        DATABASE_SESSION, 
        DATABASE_SESSION_DRIVER_NAME,
        "Database driver is missing");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    if (0 == (*i_aspect).m_dbname[0])
    {
      l_error= g_error_new(
        DATABASE_SESSION, 
        DATABASE_SESSION_TABLE_NAME,
        "Database is missing");
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_exit= driver_config_load(&l_error, &l_driver);

    if (l_exit)
    {
      break;
    }

    l_module= 0;

    for (l_slot= 0; l_driver.m_count > l_slot; l_slot++)
    {

      l_rc= strcmp(
        l_driver.m_module[l_slot].m_tag,
        (*i_aspect).m_driver);

      if (0 == l_rc)
      {
        l_module= &l_driver.m_module[l_slot];
        break;
      }

    }

    if (0 == l_module)
    {
      l_error= g_error_new(
        DATABASE_SESSION, 
        DATABASE_SESSION_MODULE,
        "Database driver module ('%s') not found",
        (*i_aspect).m_driver);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_exit= iwsql_dll_load(&l_error, &(*io_session).m_dll, l_module);

    if (l_exit) 
    {
      break;
    }

    iwsql_dll_new(&(*io_session).m_dll);
    memset(&l_conn, 0, sizeof(l_conn));
    strcpy(l_conn.m_dbname, (*i_aspect).m_dbname);
    strcpy(l_conn.m_host, (*i_aspect).m_host);
    strcpy(l_conn.m_hostaddr, (*i_aspect).m_hostaddr);
    strcpy(l_conn.m_port, (*i_aspect).m_port);
    strcpy(l_conn.m_username, (*i_aspect).m_username);
    strncpy(
      l_conn.m_password, 
      (((*i_aspect).m_password) ? (*i_aspect).m_password : ""),
      sizeof(l_conn.m_password)-1);

    l_exit= iwsql_dll_connect(&l_error, &(*io_session).m_dll, &l_conn);

    if (l_exit)
    {
      break;
    }

    (*io_session).m_connected= 1;

  }while(0);

  driver_config_discharge(&l_driver);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
database_session_discharge(
  struct database_session*const         io_session)
{

  iwsql_dll_discharge(&(*io_session).m_dll);
  memset(io_session, 0, sizeof(*io_session));

  return;
}

extern int
database_session_exec(
  GError**                              o_error,
  struct database_session*const         io_session,
  char const*                           i_sqlfile)
{
  GError*                               l_error;
  int                                   l_exit;
  FILE*                                 l_fp;
  size_t                                l_size;
  char*                                 l_sql;

  l_error= 0;
  l_exit= 0;
  l_sql= 0;

  do
  {

    l_fp= g_fopen(i_sqlfile, "r");

    if (0 == l_fp)
    {
      l_error= g_error_new(
        DATABASE_SESSION, 
        DATABASE_SESSION_FILE_OPEN,
        "Unable to open SQL file ('%s'); errno(%d)='%s'",
        i_sqlfile,
        errno,
        strerror(errno));
      _error_log(l_error);
      l_exit= -1; 
      break;
    }

    fseek(l_fp, 0, SEEK_END);
    l_size= ftell(l_fp);
    fseek(l_fp, 0, SEEK_SET);

    l_sql= (char*)g_malloc0(1+l_size);
    memset(l_sql, 0, (1+l_size));

    fread(l_sql, l_size, 1, l_fp);
    fclose(l_fp);
    l_fp= 0;

    l_exit= database_session_exec_buffer(&l_error, io_session, l_sql);

  }while(0);

  g_free(l_sql);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern int
database_session_exec_buffer(
  GError**                              o_error,
  struct database_session*const         io_session,
  char const*                           i_sql)
{
  int                                   l_exit;

  l_exit= iwsql_dll_exec(o_error, &(*io_session).m_dll, i_sql);

  return l_exit;
}

extern void
database_session_get_attributes(
  struct iwsql_attr const**             o_attr,
  unsigned*const                        o_slots,
  struct database_session const*const   i_session)
{

  iwsql_dll_result(o_attr, o_slots, &(*i_session).m_dll);

  return;
}

extern int
database_session_next(
  GError**                              o_error,
  int*const                             o_eof,
  struct database_session*const         io_session)
{
  int                                   l_exit;

  l_exit= iwsql_dll_next(o_error, o_eof, &(*io_session).m_dll);

  return l_exit;
}
