/*
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
  Copyright (c) 2015 Dante University Foundation and Contributors
  
  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:
  
     1. Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
  
     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.
  
  THIS SOFTWARE IS PROVIDED BY Dante University Foundation ``AS IS'' AND ANY
  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL Dante University Foundation OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.
  
  The views and conclusions contained in the software and documentation are
  those of the authors and should not be interpreted as representing official
  policies, either expressed or implied, of Dante University Foundation.
  
  Dante University Foundation
  P.O. Box 812158
  Wellesley, MA 02482
  USA
  www.danteuniversity.org
*/
#include "bson.h"

struct dynstring
{
  unsigned                              m_size;
  char*                                 m_text;
  unsigned                              m_pos;
};

static int
bson_document_parse_internal(
  struct bson_document*                 io_doc,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle);

static void
dynstring_assign(
  struct dynstring*                     o_dyn)
{
  memset(o_dyn, 0, sizeof(*o_dyn));
}

static void
dynstring_discharge(
  struct dynstring*                     o_dyn)
{
  if ((*o_dyn).m_text)
  {
    free((*o_dyn).m_text);
  }
  memset(o_dyn, 0, sizeof(*o_dyn));
}

#define DYNSTRING_GROWBY                (2*1024)
static void
dynstring_realloc(
  struct dynstring*                     io_dyn,
  unsigned const                        i_size)
{
  unsigned                              l_size;

  do
  {

    if (i_size <= (*io_dyn).m_size)
    {
      break;
    }

    l_size= (*io_dyn).m_size;

    if (DYNSTRING_GROWBY < i_size)
    {
      l_size+= i_size;
    }
    else
    {
      l_size+= DYNSTRING_GROWBY;
    }

    (*io_dyn).m_text= realloc((*io_dyn).m_text, l_size);
    (*io_dyn).m_size= l_size;

  }while(0);

  memset((*io_dyn).m_text, 0, (*io_dyn).m_size);

  return;
}

static void
dynstring_putc(
  struct dynstring*                     io_dyn,
  char const                            i_chr)
{

  if ((*io_dyn).m_pos == (*io_dyn).m_size)
  {
    dynstring_realloc(io_dyn, 1);
  }

  (*io_dyn).m_text[(*io_dyn).m_pos]= i_chr;
  (*io_dyn).m_pos++;

  return;
}

static void
dynstring_reset(
  struct dynstring*                     io_dyn)
{
  memset((*io_dyn).m_text, 0, (*io_dyn).m_size);
  (*io_dyn).m_pos= 0;
  return;
}

extern ptrdiff_t
bson_fread(
  bson_handle_t const                   i_handle,
  void*                                 o_block,
  size_t                                i_count)
{
  ptrdiff_t                             l_exit;
  FILE*                                 l_fp;
  int                                   l_rc;

  l_exit= 0;

  do
  {

    if (0 == i_handle)
    {
      break;
    }

    if (0 == i_count)
    {
      break;
    }

    if (0 == o_block)
    {
      break;
    }

    l_fp= (FILE*)i_handle;
    memset(o_block, 0, i_count);

    l_rc= feof(l_fp);

    if (l_rc)
    {
      break;
    }

    l_exit= (ptrdiff_t)fread(o_block, 1, i_count, l_fp);

    if (l_exit == (ptrdiff_t)i_count)
    {
      break;
    }

    l_rc= ferror(l_fp);

    if (l_rc)
    {
      l_exit= -1;
    }

  }while(0);

  return l_exit;
}

static ptrdiff_t
bson_parse_cstring(
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  char                                  l_chr;
  ptrdiff_t                             l_rc;

  dynstring_reset(io_dyn);

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_chr, 1);

    if (1 != l_rc)
    {
      /* unexpected end of string */
      l_rc= -1;
      break;
    }

    dynstring_putc(io_dyn, l_chr);

    if (0 == l_chr)
    {
      l_rc= 0;
      break;
    }

  }while(1);

  return l_rc;
}

static ptrdiff_t
bson_parse_int64(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  int64_t                               l_int64;
  ptrdiff_t                             l_rc;

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_int64, sizeof(l_int64));

    if (sizeof(l_int64) != l_rc)
    {
      l_rc= -1;
      break;
    }

    (*io_node).m_object.m_int64= l_int64;
    l_rc= 0;

  }while(0);

  return l_rc;
}

static ptrdiff_t
bson_parse_float(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  double                                l_double;
  ptrdiff_t                             l_rc;

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_double, sizeof(l_double));

    if (sizeof(l_double) != l_rc)
    {
      l_rc= -1;
      break;
    }

    (*io_node).m_object.m_float= l_double;
    l_rc= 0;

  }while(0);

  return l_rc;
}

static ptrdiff_t
bson_parse_string(
  struct bson_node*                     io_node,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;
  int32_t                               l_int32;

  dynstring_reset(io_dyn);

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_int32, sizeof(l_int32));

    if (sizeof(l_int32) != l_rc)
    {
      l_rc= -1;
      break;
    }
  
    dynstring_realloc(io_dyn, l_int32);

    l_rc= (*i_readfn)(i_handle, (*io_dyn).m_text, l_int32);

    if (l_int32 != l_rc)
    {
      l_rc= -1;
      break;
    }

    (*io_node).m_object.m_string.m_size= l_int32;
    (*io_node).m_object.m_string.m_text= (char*)malloc(l_int32);
    memcpy(
      (*io_node).m_object.m_string.m_text,
      (*io_dyn).m_text,
      l_int32);

    l_rc= 0;

  }while(0);

  return l_rc;
}

static ptrdiff_t
bson_parse_document(
  struct bson_node*                     io_node,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;

  bson_document_assign(&(*io_node).m_object.m_document); 

  l_rc= bson_document_parse_internal(
    &(*io_node).m_object.m_document,
    io_dyn,
    i_readfn,
    i_handle);

  return l_rc;
}

static ptrdiff_t
bson_parse_array(
  struct bson_node*                     io_node,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;
  
  l_rc= bson_parse_document(io_node, io_dyn, i_readfn, i_handle);

  return l_rc;
}

static ptrdiff_t
bson_parse_binary(
  struct bson_node*                     io_node,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;
  int32_t                               l_int32;
  char                                  l_octet;

  dynstring_reset(io_dyn);

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_int32, sizeof(l_int32));

    if (sizeof(l_int32) != l_rc)
    {
      l_rc= -1;
      break;
    }

    dynstring_realloc(io_dyn, l_int32);

    l_rc= (*i_readfn)(i_handle, &l_octet, sizeof(l_octet));

    if (sizeof(l_octet) != l_rc)
    {
      l_rc= -1;
      break;
    }

    (*io_node).m_object.m_binary.m_type= l_octet;

    l_rc= (*i_readfn)(i_handle, (*io_dyn).m_text, l_int32);

    if (l_int32 != l_rc)
    {
      l_rc= -1;
      break;
    }

    (*io_node).m_object.m_binary.m_size= l_int32;
    (*io_node).m_object.m_binary.m_byte= (char*)malloc(l_int32);
    memcpy(
      (*io_node).m_object.m_binary.m_byte,
      (*io_dyn).m_text,
      l_int32);

    l_rc= 0;

  }while(0);

  return l_rc;
}

static ptrdiff_t
bson_parse_objectid(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;

  do
  {

    l_rc= (*i_readfn)(
      i_handle, 
      &(*io_node).m_object.m_objectid[0],
      sizeof((*io_node).m_object.m_objectid));

    if (sizeof((*io_node).m_object.m_objectid) != l_rc)
    {
      l_rc= -1;
    }

    l_rc= 0;

  }while(0);

  return l_rc;
}

static ptrdiff_t
bson_parse_boolean(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  char                                  l_chr;
  ptrdiff_t                             l_rc;

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_chr, sizeof(l_chr));

    if (sizeof(l_chr) != l_rc)
    {
      l_rc= -1;
      break;
    }

    (*io_node).m_object.m_boolean= ((l_chr) ? 1 : 0);
    l_rc= 0;

  }while(0);

  return l_rc;
}

static ptrdiff_t
bson_parse_utc_datetime(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;

  l_rc= bson_parse_int64(io_node, i_readfn, i_handle);
  
  return l_rc;
}

static ptrdiff_t
bson_parse_null(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  return 0;
}

static ptrdiff_t
bson_parse_regex(
  struct bson_node*                     io_node,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  /* todo */
  return -1;
}

static ptrdiff_t
bson_parse_js_code(
  struct bson_node*                     io_node,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  /* todo */
  return -1;
}

static ptrdiff_t
bson_parse_js_code_w_s(
  struct bson_node*                     io_node,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  /* todo */
  return -1;
}

static ptrdiff_t
bson_parse_int32(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  int32_t                               l_int32;
  ptrdiff_t                             l_rc;

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_int32, sizeof(l_int32));

    if (sizeof(l_int32) != l_rc)
    {
      l_rc= -1;
      break;
    }

    (*io_node).m_object.m_int32= l_int32;
    l_rc= 0;

  }while(0);

  return l_rc;
}

static ptrdiff_t
bson_parse_timestamp(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;

  l_rc= bson_parse_int64(io_node, i_readfn, i_handle);
  
  return l_rc;
}


static ptrdiff_t
bson_parse_min_key(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  return 0;
}

static ptrdiff_t
bson_parse_max_key(
  struct bson_node*                     io_node,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  return 0;
}

static int
bson_document_parse_internal(
  struct bson_document*                 io_doc,
  struct dynstring*                     io_dyn,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  int32_t                               l_int32;
  ptrdiff_t                             l_rc;
  enum bson_type                        l_op;
  struct bson_node*                     l_node;

  l_rc= 0;

  do
  {

    l_rc= (*i_readfn)(i_handle, &l_int32, sizeof(l_int32));

    if (sizeof(l_int32) != l_rc)
    {
      /* unable to get length of document */
      l_rc= -1;
      break;
    }

    (*io_doc).m_size= l_int32;

    do
    {

      l_op= bson_type_end;
      l_rc= (*i_readfn)(i_handle, &l_op, 1);

      if (l_rc != 1)
      {
        /* unable to read op code */
        l_rc= -1;
        break;
      }

      if (bson_type_end == l_op)
      {
        /* end of document maker */
        l_rc= 0;
        break;
      }

      l_node= bson_document_node_new(io_doc);

      l_rc= bson_parse_cstring(io_dyn, i_readfn, i_handle);

      if (l_rc)
      {
        l_rc= -1;
        break;
      }

      (*l_node).m_type= l_op;
      (*l_node).m_ename= (char*)malloc(1+(*io_dyn).m_size);
      memset((*l_node).m_ename, 0, 1+(*io_dyn).m_size);
      memcpy((*l_node).m_ename, (*io_dyn).m_text, (*io_dyn).m_size);
      
      switch(l_op)
      {
        case bson_type_float:
          l_rc= bson_parse_float(l_node, i_readfn, i_handle); 
          break;
        case bson_type_string:
          l_rc= bson_parse_string(l_node, io_dyn, i_readfn, i_handle); 
          break;
        case bson_type_document:
          l_rc= bson_parse_document(l_node, io_dyn, i_readfn, i_handle); 
          break;
        case bson_type_array:
          l_rc= bson_parse_array(l_node, io_dyn, i_readfn, i_handle); 
          break;
        case bson_type_binary:
          l_rc= bson_parse_binary(l_node, io_dyn, i_readfn, i_handle); 
          break;
        case bson_type_objectid:
          l_rc= bson_parse_objectid(l_node, i_readfn, i_handle); 
          break;
        case bson_type_boolean:
          l_rc= bson_parse_boolean(l_node, i_readfn, i_handle); 
          break;
        case bson_type_utc_datetime:
          l_rc= bson_parse_utc_datetime(l_node, i_readfn, i_handle); 
          break;
        case bson_type_null:
          l_rc= bson_parse_null(l_node, i_readfn, i_handle); 
          break;
        case bson_type_regex:
          l_rc= bson_parse_regex(l_node, io_dyn, i_readfn, i_handle); 
          break;
        case bson_type_js_code:
          l_rc= bson_parse_js_code(l_node, io_dyn, i_readfn, i_handle); 
          break;
        case bson_type_js_code_w_s:
          l_rc= bson_parse_js_code_w_s(l_node, io_dyn, i_readfn, i_handle); 
          break;
        case bson_type_int32:
          l_rc= bson_parse_int32(l_node, i_readfn, i_handle); 
          break;
        case bson_type_timestamp:
          l_rc= bson_parse_timestamp(l_node, i_readfn, i_handle); 
          break;
        case bson_type_int64:
          l_rc= bson_parse_int64(l_node, i_readfn, i_handle); 
          break;
        case bson_type_min_key:
          l_rc= bson_parse_min_key(l_node, i_readfn, i_handle); 
          break;
        case bson_type_max_key:
          l_rc= bson_parse_max_key(l_node, i_readfn, i_handle); 
          break;
        default:
          /* unkown or deprecated/unsupported */
          l_rc= -1;
          break;
      }

      if (l_rc)
      {
        break;
      }
  
    }while(1);

  }while(0);

  return l_rc;;
}

extern int
bson_document_parse(
  struct bson_document*                 io_doc,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle)
{
  ptrdiff_t                             l_rc;
  struct dynstring                      l_dyn;

  dynstring_assign(&l_dyn);
  l_rc= bson_document_parse_internal(io_doc, &l_dyn, i_readfn, i_handle);
  dynstring_discharge(&l_dyn);

  return l_rc;;
}
