/*
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
  Copyright (c) 2015 Dante University Foundation and Contributors
  
  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:
  
     1. Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
  
     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.
  
  THIS SOFTWARE IS PROVIDED BY Dante University Foundation ``AS IS'' AND ANY
  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL Dante University Foundation OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.
  
  The views and conclusions contained in the software and documentation are
  those of the authors and should not be interpreted as representing official
  policies, either expressed or implied, of Dante University Foundation.
  
  Dante University Foundation
  P.O. Box 812158
  Wellesley, MA 02482
  USA
  www.danteuniversity.org
*/
#ifndef __bson_h__

#include <string.h>
#include <stdlib.h>
#include <stdint.h>
#include <stddef.h>
#include <stdio.h>
#include <time.h>

enum bson_type
{
  bson_type_end=                        0x00,
  bson_type_float=                      0x01, /* double Floating point */
  bson_type_string=                     0x02, /* string UTF-8 string */
  bson_type_document=                   0x03, /* Embedded document*/
  bson_type_array=                      0x04, /* Array */
  bson_type_binary=                     0x05, /* binary Binary data */
  bson_type_deprecated1=                0x06, /* Undefined — Deprecated */
  bson_type_objectid=                   0x07, /* ObjectId */
  bson_type_boolean=                    0x08, /* Boolean "false"/"true" */ 
  bson_type_utc_datetime=               0x09, /* UTC datetime */
  bson_type_null=                       0x0A, /* Null value */
  bson_type_regex=                      0x0B, /* Regular expression */
  bson_type_deprecated2=                0x0C, /* DBPointer — Deprecated */
  bson_type_js_code=                    0x0D, /* JavaScript code */
  bson_type_deprecated3=                0x0E, /* Deprecated */
  bson_type_js_code_w_s=                0x0F, /* JavaScript code w/ scope */ 
  bson_type_int32=                      0x10, /* 32-bit Integer */
  bson_type_timestamp=                  0x11, /* Timestamp */
  bson_type_int64=                      0x12, /* 64-bit integer*/
  bson_type_min_key=                    0xFF, /* Min key */
  bson_type_max_key=                    0x7F  /* Max key */
};

enum bson_subtype
{
  bson_subtype_binary=                  0x00, /* Generic binary subtype */
  bson_subtype_function=                0x01, /* Function */
  bson_subtype_binold=                  0x02, /* Binary (Old) */
  bson_subtype_uuidold=                 0x03, /* UUID (Old) */
  bson_subtype_uuid=                    0x04, /* UUID */
  bson_subtype_md5=                     0x05, /* MD5 */
  bson_subtype_user=                    0x80 /* User defined */
};

struct bson_node;

struct bson_binary
{
  int32_t                               m_size;
  enum bson_subtype                     m_type;
  char*                                 m_byte;
};

struct bson_document
{
  int32_t                               m_size;
  struct bson_node*                     m_head;
  struct bson_node*                     m_tail;
};

struct bson_string
{
  int32_t                               m_size;
  char*                                 m_text;
  char                                  m_null[1];
};

struct bson_regex
{
  char*                                 m_pattern;
  char*                                 m_options;
};

struct bson_code_w_s
{
  int32_t                               m_size;
  struct bson_string                    m_string;
  struct bson_document                  m_document;
};

struct bson_node
{
  enum bson_type                        m_type;
  char*                                 m_ename;
  union
  {
    double                              m_float;
    struct bson_string                  m_string;
    struct bson_document                m_document;
    struct bson_document                m_array;
    struct bson_binary                  m_binary;
    char                                m_boolean;
    char                                m_objectid[12];
    int64_t                             m_utc;
    struct bson_regex                   m_regex; 
    struct bson_string                  m_js_code;
    struct bson_code_w_s                m_js_code_w_s;
    int32_t                             m_int32;
    int64_t                             m_int64;
  }                                     m_object;
  struct bson_node*                     m_next;
  struct bson_node*                     m_prev;
};

typedef void*                           bson_handle_t;

typedef ptrdiff_t                       (*bson_read_t)(
  bson_handle_t const                   i_handle,
  void*                                 o_block,
  size_t                                i_count);

extern void
bson_document_assign(
  struct bson_document*                 o_doc);

extern void
bson_document_discharge(
  struct bson_document*                 io_doc);

extern struct bson_node*
bson_document_node_new(
  struct bson_document*                 io_doc);

extern int
bson_document_parse(
  struct bson_document*                 io_doc,
  bson_read_t                           i_readfn,
  bson_handle_t                         i_handle);

extern ptrdiff_t
bson_fread(
  bson_handle_t                         i_handle,
  void*                                 o_block,
  size_t                                i_count);

extern void
bson_node_assign(
  struct bson_node*                     o_node);

extern void
bson_node_discharge(
  struct bson_node*                     io_node);

#define  __bson_h__
#endif
