{- This source file is part of mmisar - a tool for converting
Metamath proofs to Isabelle/Isar.
Copyright 2006 Slawomir Kolodynski

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
-}


module Utils where

import Data.List

-- takes a list and returns nothing if the list does not contain a separator, or
-- a pair (list up to a separator, list after the separator) otherwise.
-- The separator is defined by a predicate
splitGroup :: (a -> Bool) -> [a] -> Maybe ([a],[a])
splitGroup pred x = 
   let z = break pred x
   in if null x then Nothing 
      else if null (snd z) then Just (fst z, [])
         else Just (fst z, tail (snd z))


--checks if a string is prefixed by any from a list of prefixes
-- best used infix
isNotPrefixedByAnyOf :: Eq a => [a] -> [[a]] -> Bool
str `isNotPrefixedByAnyOf` prefs = all (str `isNotPrefixedBy` ) prefs
   where
      str `isNotPrefixedBy` pref = not (isPrefixOf pref str)

-- splits a list into a pair - (before a separator, after a separator), 
-- dropping the separator. The list after the separator has to be nonempty

breakAtDrop :: (a -> Bool) -> [a] -> ([a], [a])
breakAtDrop pred x = (fst y, tail (snd y) ) 
   where y = break pred x 

-- do I really have to define this?
-- isADigit :: Char -> Bool
-- isADigit c = any (c==) ['0','1','2','3','4','5','6','7','8','9']

-- for readibility
sndEl :: [a] -> a
sndEl x = (!!) x 1

thirdEl :: [a] -> a
thirdEl x = x !! 2

-- prepend if not null
pne :: [a] -> [[a]] -> [[a]]
[] `pne` b = b
a `pne` b = a:b 

-- takes a union of a list of lists
unionAll :: Eq a => [[a]] -> [a]
-- unionAll x = foldr union [] x
--unionAll x = foldr union (nub (last x)) (init x)
unionAll x = foldl union (nub (head x)) (tail x)

-- from Henning Thielemann at Haskell-cafe
replace :: Eq a => [a] -> [a] -> [a] -> [a]
replace src dst =
    foldr (\x xs -> let y=x:xs
                    in  if isPrefixOf src y
                          then dst ++ drop (length src) y
                          else y) []

-- multiple replacement, last replacement in the list is performed first!
replaceAll :: Eq a => [ ([a],[a]) ] -> [a] ->[a]
replaceAll srcdest x = if (null srcdest) then x 
                       else replace (fst h) (snd h) $ replaceAll (tail srcdest) x
                          where h = head srcdest


-- inserts an element at a given position in a list and applies a function to the reminder
insertAtApply :: Int -> a -> ([a] -> [a]) -> [a] -> [a]
insertAtApply n e f x = (fst y) ++ e:(f $ snd y)
                 where y = splitAt n x

-- like takeWhile, but from the end
takeWhileBack :: (a -> Bool) -> [a] -> [a]
takeWhileBack p = reverse . takeWhile p . reverse

-- traverses a list and finds the element that satisfies a predicate 
-- while the nesting level satisfies a different predicate. It always breaks 
-- when the nesting level goes below 0.
-- Nesting is defined by an up element (typically an
-- open paranthesis and down element (typically a close paranthesis).
-- Examples 
-- spanNestLev '(' ')' (=='+') (==0) (0,"","ab(+)cd+ah") = (0,"ab(+)cd","+ah")

spanNestLev :: Eq a => a -> a -> (a -> Bool) -> (Int -> Bool) -> (Int, [a],[a]) -> (Int, [a],[a])
spanNestLev up down p pn (n, xs, xe) | null xe                 = (n, xs, [])
                                      | head xe == up   = spanNestLev up down p pn (n+1, xs ++ [head xe], tail xe)
                                      | head xe == down = spanNestLev up down p pn (n-1, xs ++ [head xe], tail xe)
                                      | n < 0 = (n,xs,xe) -- experimental: always span when the nesting level drops below zero 
                                      | (pn n) && (p (head xe)) = (0, xs, xe)
                                      | otherwise       = spanNestLev up down p pn (n, xs ++ [head xe], tail xe)

-- for extracting the second element of a triple
snd3 :: (a,b,c) -> b
snd3 (n,k,l) = k

-- for extracting the third element of a triple
thrd3 :: (a,b,c) -> c
thrd3 (n,k,l) = l
