(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{EquivClass1.thy}*}

theory EquivClass1 imports EquivClass func_ZF ZF1 

begin

text{*In this theory file we extend the  work on equivalence relations 
  done in the standard Isabelle's EquivClass.thy file. The problem that 
  we have with the EquivClass.thy is that the notions congruent and congruent2
  are defined for meta-functions rather then ZF - functions (subsets of 
  Cartesian products). This causes inflexibility (that is typical for typed 
  set theories) in making the notions depend on additional parameters
  For example the congruent2 there takes $[i,[i,i]=>i]$ as parameters, that is 
  the second parameter is a meta-function that takes two sets and results in 
  a set. So, when our function depends on additional parameters, 
  (as for example in Real\_ZF1.thy, where the function we want to be 
  congruent depends on a group and we want to show that for all groups 
  the function is congruent) there is no easy way to use that notion.
  The ZF functions are sets and there is no problem if in actual 
  application this set depends on some parameters. *}

section{*Congruent functions and projections on the quotient*}

text{*First we define the notion of function that maps equivalent 
  elements to equivalent values. We use similar names as
  in the original EquivClass.thy file to indicate the conceptual 
  correspondence of the notions. Then we define the projection of
  a function onto the quotient space. We will show that if the function is
  congruent the projection is 
  a mapping from the quotient space into itself. In standard math the 
  condion that the function is congruent allows to show that the value
  of the projection does not depend on the choice of elements that represent
  the equivalence classes. We set up things a little differently to avoid
  making choices.*}

constdefs
  "Congruent(r,f) \<equiv>
  (\<forall>x y. <x,y> \<in> r  \<longrightarrow> <f`(x),f`(y)> \<in> r)"

  "ProjFun(A,r,f) \<equiv>
  {<c,d> \<in> (A//r)\<times>(A//r). (\<Union>x\<in>c. r``{f`(x)}) = d}";

text{*Elements of equivalence classes belong to the set.*}

lemma EquivClass_1_L1: 
  assumes A1: "equiv(A,r)" and A2: "C \<in> A//r" and A3: "x\<in>C"
  shows "x\<in>A"
proof -;
  from A2 have "C \<subseteq> \<Union> (A//r)" by auto;
  with A1 A3 show "x\<in>A"
    using Union_quotient by auto;
qed;

text{*If an element belongs to an equivalence class, then its image
  under relation is this equivalence class.*}

lemma EquivClass_1_L2: 
  assumes A1:"equiv(A,r)" "C \<in> A//r" and A2:"x\<in>C"
  shows "r``{x} = C"
proof -;
  from A1 A2 have "x \<in> r``{x}" 
    using EquivClass_1_L1  equiv_class_self by simp;
  with A2 have T1:"r``{x}\<inter>C \<noteq> 0" by auto;
  from A1 A2 have "r``{x} \<in> A//r"
    using EquivClass_1_L1 quotientI by simp;
  with A1 T1 show ?thesis
    using quotient_disj by blast;
qed;
      
text{*Elements that belong to the same equivalence class are equivalent.*}

lemma EquivClass_1_L2A:
  assumes "equiv(A,r)" "C \<in> A//r" "x\<in>C" "y\<in>C"
  shows "<x,y> \<in> r" 
  using prems EquivClass_1_L2 EquivClass_1_L1 equiv_class_eq_iff
  by simp;

text{*Every $x$ is in the class of $y$, then they are equivalent.*}

lemma EquivClass_1_L2B: 
  assumes A1:"equiv(A,r)" and A2:"y\<in>A" and A3:"x \<in> r``{y}"
  shows "<x,y> \<in> r"
proof -;
  from A2 have  "r``{y} \<in> A//r"
    using quotientI by simp;
  with A1 A3 show ?thesis using
    EquivClass_1_L1 equiv_class_self equiv_class_nondisjoint by blast;
qed;
  
text{*If a function is congruent then the equivalence classes of the values
  that come from the arguments from the same class are the same.*}

lemma EquivClass_1_L3: 
  assumes A1:"equiv(A,r)" and A2:"Congruent(r,f)" 
  and A3:"C \<in> A//r" "x\<in>C" "y\<in>C" 
  shows "r``{f`(x)} = r``{f`(y)}"
proof -;
  from A1 A3 have "<x,y> \<in> r"
    using EquivClass_1_L2A by simp;
  with A2 have  "<f`(x),f`(y)> \<in> r"
    using Congruent_def by simp;
  with A1 show ?thesis using equiv_class_eq by simp
qed;

text{*The values of congruent functions are in the space.*}

lemma EquivClass_1_L4:
  assumes A1:"equiv(A,r)" and A2:"C \<in> A//r" "x\<in>C"
  and A3:"Congruent(r,f)"
  shows "f`(x) \<in> A"
proof -;
  from A1 A2 have "x\<in>A";
    using EquivClass_1_L1 by simp;
  with A1 have "<x,x> \<in> r"
    using equiv_def refl_def by simp;
  with A3 have  "<f`(x),f`(x)> \<in> r"
    using Congruent_def by simp;
  with A1 show ?thesis using equiv_type by auto;
qed;

text{*Equivalence classes are not empty.*}

lemma EquivClass_1_L5: 
  assumes A1:"refl(A,r)" and A2: "C \<in> A//r"
  shows "C\<noteq>0"
proof -;
  from A2 obtain x where D1: "C = r``{x}" and D2: "x\<in>A"
    using quotient_def by auto;
  from D2 A1 have "x \<in> r``{x}" using refl_def by auto;
  with D1 show ?thesis by auto;
qed;
  
text{*To avoid using an axiom of choice, we define the projection using 
  the expression $\bigcup _{x\in C} r(\{f(x)\})$. 
  The next lemma shows that for
  congruent function this is in the quotient space $A/r$. *}

lemma EquivClass_1_L6:
  assumes A1:"equiv(A,r)" and A2:"Congruent(r,f)" 
  and A3:"C \<in> A//r"
  shows "(\<Union>x\<in>C. r``{f`(x)}) \<in> A//r"
proof -;
  from A1 A3 have "C\<noteq>0"
    using equiv_def EquivClass_1_L5 by auto;
  moreover from A2 A3 A1 have "\<forall>x\<in>C. r``{f`(x)} \<in> A//r"
    using EquivClass_1_L4 quotientI by auto;
  moreover from A1 A2 A3 have 
    "\<forall>x y. x\<in>C \<and> y\<in>C \<longrightarrow> r``{f`(x)} = r``{f`(y)}" 
    using EquivClass_1_L3 by blast;
  ultimately show ?thesis by (rule ZF1_1_L2);
qed;

text{*Congruent functions can be projected.*}

lemma EquivClass_1_T1: 
  assumes "equiv(A,r)" "Congruent(r,f)"
  shows "ProjFun(A,r,f) \<in> A//r \<rightarrow> A//r"
  using prems EquivClass_1_L6 ProjFun_def func1_1_L11A
  by simp;
  
text{*We now define congruent functions of two variables. Congruent2 
  corresponds to congruent2 in EquivClass.thy, but uses ZF-functions rather
  than meta-functions.*}

constdefs
  "Congruent2(r,f) \<equiv>
  (\<forall>x1 x2 y1 y2. <x1,x2> \<in> r \<and> <y1,y2> \<in> r  \<longrightarrow> 
  <f`<x1,y1>,f`<x2,y2> > \<in> r)"

  "ProjFun2(A,r,f) \<equiv>
  {<p,d> \<in> ((A//r)\<times>(A//r))\<times>(A//r) . 
  (\<Union> z \<in> fst(p)\<times>snd(p). r``{f`(z)}) = d}";


text{*The following lemma is a two-variables equivalent 
  of EquivClass\_1\_L3.*}

lemma EquivClass_1_L7: 
  assumes A1:"equiv(A,r)" and A2:"Congruent2(r,f)"
  and A3:"C1 \<in> A//r" "C2 \<in> A//r" 
  and A4:"z1 \<in> C1\<times>C2" "z2 \<in> C1\<times>C2"
  shows "r``{f`(z1)} = r``{f`(z2)}"
proof -;
  from A4 obtain x1 y1 x2 y2 where 
    "x1\<in>C1" and "y1\<in>C2" and D1:"z1 = <x1,y1>" and 
    "x2\<in>C1" and "y2\<in>C2" and D2:"z2 = <x2,y2>" 
    by auto;
  with A1 A3 have "<x1,x2> \<in> r" and "<y1,y2> \<in> r"
    using EquivClass_1_L2A by auto;
  with A2 have "<f`<x1,y1>,f`<x2,y2> > \<in> r"
    using Congruent2_def by simp;
  with A1 D1 D2 show ?thesis using equiv_class_eq by simp;
qed;

text{*The values of congruent functionsof two variables are in the space.*}

lemma EquivClass_1_L8:
  assumes A1:"equiv(A,r)" and A2:"C1 \<in> A//r" and A3:"C2 \<in> A//r"
  and A4:"z \<in> C1\<times>C2" and A5: "Congruent2(r,f)"
  shows "f`(z) \<in> A"
proof -;
  from A4 obtain x y where "x\<in>C1" and "y\<in>C2" and D1:"z = <x,y>"  
    by auto;
  with A1 A2 A3 have "x\<in>A" and "y\<in>A" 
    using EquivClass_1_L1 by auto;
  with A1 A4 have "<x,x> \<in> r" and "<y,y> \<in> r"
    using equiv_def refl_def by auto;
  with A5 have "<f`<x,y>, f`<x,y> > \<in> r"
    using Congruent2_def by simp;
  with A1 D1 show ?thesis using equiv_type by auto;
qed;

text{*The values of congruent functions are in the space. Note that
  although this lemma is intended to be used with functions, we don't
  need to assume that we $f$ is a function.*}

lemma EquivClass_1_L8A:
  assumes A1:"equiv(A,r)" and A2:"x\<in>A" "y\<in>A"
  and A3: "Congruent2(r,f)"
  shows "f`<x,y> \<in> A"
proof -
  from A1 A2 have "r``{x} \<in> A//r" "r``{y} \<in> A//r" 
    "<x,y> \<in> r``{x}\<times>r``{y}"
    using equiv_class_self quotientI by auto;
  with A1 A3 show ?thesis using EquivClass_1_L8 by simp;
qed;
  
text{*The following lemma is a two-variables equivalent of 
  EquivClass\_1\_L6.*}

lemma EquivClass_1_L9:
  assumes A1:"equiv(A,r)" and A2:"Congruent2(r,f)" 
  and A3:"p \<in> (A//r)\<times>(A//r)"
  shows "(\<Union> z \<in> fst(p)\<times>snd(p). r``{f`(z)}) \<in> A//r"
proof -;
  from A3 have D1:"fst(p) \<in> A//r" and D2:"snd(p) \<in> A//r"
    by auto;
  with A1 A2 have 
    T1:"\<forall>z \<in> fst(p)\<times>snd(p). f`(z) \<in> A"
    using EquivClass_1_L8 by simp;
  from A3 A1 have "fst(p)\<times>snd(p) \<noteq> 0" 
    using equiv_def EquivClass_1_L5 Sigma_empty_iff
    by auto;
  moreover from A1 T1 have 
    "\<forall>z \<in> fst(p)\<times>snd(p). r``{f`(z)} \<in> A//r"
    using quotientI by simp;
  moreover from A1 A2 D1 D2 have
    "\<forall>z1 z2. z1 \<in> fst(p)\<times>snd(p) \<and> z2 \<in> fst(p)\<times>snd(p) \<longrightarrow> 
    r``{f`(z1)} = r``{f`(z2)}"
    using EquivClass_1_L7 by blast;
   ultimately show ?thesis by (rule ZF1_1_L2);
qed;

text{*Congruent functions of two variables can be projected.*}

theorem EquivClass_1_T1: 
  assumes "equiv(A,r)" "Congruent2(r,f)"
  shows "ProjFun2(A,r,f) \<in> (A//r)\<times>(A//r) \<rightarrow> A//r"
  using prems EquivClass_1_L9 ProjFun2_def func1_1_L11A by simp;

text{*We define the projection on the quotient space as a function that takes
  an element of $A$ and assigns its equivalence class in $A/r$.*}

constdefs
  "Proj(A,r) \<equiv> {<x,c> \<in> A\<times>(A//r). r``{x} = c}"

text{*The projection diagram commutes. I wish I knew how to draw this diagram
  in \LaTeX. *}

lemma EquivClass_1_L10: assumes A1:"equiv(A,r)" and A2:"Congruent2(r,f)" 
  and A3:"x\<in>A" "y\<in>A"
  shows "ProjFun2(A,r,f)`<r``{x},r``{y}> = r``{f`<x,y>}"
proof -;
  from A3 A1 have "r``{x} \<times> r``{y} \<noteq> 0"
    using quotientI equiv_def EquivClass_1_L5 Sigma_empty_iff
    by auto;
  moreover have 
    "\<forall>z \<in> r``{x}\<times>r``{y}.  r``{f`(z)} = r``{f`<x,y>}"
  proof;
    fix z assume A4:"z \<in> r``{x}\<times>r``{y}"
    from A1 A3 have 
      "r``{x} \<in> A//r" "r``{y} \<in> A//r"
      "<x,y> \<in> r``{x}\<times>r``{y}"
      using quotientI equiv_class_self by auto;
    with A1 A2 A4 show
      "r``{f`(z)} = r``{f`<x,y>}"
      using EquivClass_1_L7 by blast;
  qed;
  ultimately have 
    "(\<Union>z \<in> r``{x}\<times>r``{y}. r``{f`(z)}) =  r``{f`<x,y>}"
    by (rule ZF1_1_L1);
  moreover from A3 A1 A2 have 
    "ProjFun2(A,r,f)`<r``{x},r``{y}> = 
    (\<Union>z \<in> r``{x}\<times>r``{y}. r``{f`(z)})"
    using quotientI EquivClass_1_T1 ProjFun2_def func1_1_L11B 
    by simp;
  ultimately show ?thesis by simp;
qed;


section{*Projecting commutative, associative and distributive operations.*}

text{*In this section we show that if the operations are congruent with
  respect to an equivalence relation then the projection to the quotient 
  space preserves commutativity, associativity and distributivity.*}

text{*The projection of commutative operation is commutative.*}

lemma EquivClass_2_L1: assumes 
  A1:"equiv(A,r)" and A2:"Congruent2(r,f)"
  and A3:"f {is commutative on} A"
  and A4:"c1 \<in> A//r" "c2 \<in> A//r"
  shows "ProjFun2(A,r,f)` <c1,c2> =  ProjFun2(A,r,f)`<c2,c1>"
proof -;
  from A4 obtain x y where D1:
    "c1 = r``{x}" "c2 = r``{y}"
    "x\<in>A" "y\<in>A"
    using quotient_def by auto;
  with A1 A2 have "ProjFun2(A,r,f)` <c1,c2> =  r``{f`<x,y>}"
    using EquivClass_1_L10 by simp;
  also from A3 D1 have
    "r``{f`<x,y>} = r``{f`<y,x>}"
    using IsCommutative_def by simp;
  also from A1 A2 D1 have
    "r``{f`<y,x>} = ProjFun2(A,r,f)` <c2,c1>"
    using EquivClass_1_L10 by simp;
  finally show ?thesis by simp;
qed;

text{*The projection of commutative operation is commutative.*}

theorem EquivClass_2_T1:
  assumes "equiv(A,r)" and "Congruent2(r,f)"
  and "f {is commutative on} A"
  shows "ProjFun2(A,r,f) {is commutative on} A//r"
  using prems IsCommutative_def EquivClass_2_L1 by simp;
    
text{*The projection of an associative operation is associative.*}

lemma EquivClass_2_L2: 
  assumes A1: "equiv(A,r)" and A2: "Congruent2(r,f)"
  and A3: "f {is associative on} A"
  and A4: "c1 \<in> A//r"  "c2 \<in> A//r"  "c3 \<in> A//r"
  and A5: "g = ProjFun2(A,r,f)"
  shows "g`\<langle>g`<c1,c2>,c3\<rangle> = g`\<langle>c1,g`<c2,c3>\<rangle>"
proof -
  from A4 obtain x y z where D1:
    "c1 = r``{x}" "c2 = r``{y}" "c3 = r``{z}"
    "x\<in>A" "y\<in>A" "z\<in>A"
    using quotient_def by auto;
  with A3 have T1:"f`<x,y> \<in> A" "f`<y,z> \<in> A"
    using IsAssociative_def apply_type by auto;
  with A1 A2 D1 A5 have 
    "g`\<langle>g`<c1,c2>,c3\<rangle> =  r``{f`<f`<x,y>,z>}"
    using EquivClass_1_L10 by simp;
  also from D1 A3 have 
    "\<dots> = r``{f`<x,f`<y,z> >}"
    using IsAssociative_def by simp;
  also from T1 A1 A2 D1 A5 have
    "\<dots> = g`\<langle>c1,g`<c2,c3>\<rangle>"
    using EquivClass_1_L10 by simp;
  finally show ?thesis by simp;
qed;

text{*The projection of an associative operation is associative on the
  quotient.*}

theorem EquivClass_2_T2:
  assumes A1:"equiv(A,r)" and A2:"Congruent2(r,f)"
  and A3:"f {is associative on} A"
  shows "ProjFun2(A,r,f) {is associative on} A//r"
proof -;
  let ?g = "ProjFun2(A,r,f)"
  from A1 A2 have 
    "?g \<in> (A//r)\<times>(A//r) \<rightarrow> A//r"
    using EquivClass_1_T1 by simp;
  moreover from A1 A2 A3 have
    "\<forall>c1 \<in> A//r.\<forall>c2 \<in> A//r.\<forall>c3 \<in> A//r.
    ?g`<?g`<c1,c2>,c3> = ?g`< c1,?g`<c2,c3> >"
    using EquivClass_2_L2 by simp;
  ultimately show ?thesis
    using IsAssociative_def by simp;
qed;

text{*The essential condition to show that distributivity is preserved by 
  projections to quotient spaces, provided both operations are congruent
  with respect to the equivalence relation.*}

lemma EquivClass_2_L3: 
  assumes A1: "IsDistributive(X,A,M)"
  and A2: "equiv(X,r)" 
  and A3: "Congruent2(r,A)" "Congruent2(r,M)"
  and A4: "a \<in> X//r" "b \<in> X//r" "c \<in> X//r"
  and A5: "Ap = ProjFun2(X,r,A)" "Mp = ProjFun2(X,r,M)"
  shows "Mp`\<langle>a,Ap`<b,c>\<rangle> = Ap`\<langle> Mp`<a,b>,Mp`<a,c>\<rangle> \<and> 
  Mp`\<langle> Ap`<b,c>,a \<rangle> = Ap`\<langle> Mp`<b,a>,Mp`<c,a>\<rangle>"
proof;
  from A4 obtain x y z where "x\<in>X" "y\<in>X" "z\<in>X"
    "a = r``{x}" "b = r``{y}" "c = r``{z}"   
    using quotient_def by auto;
  with A1 A2 A3 A5 show 
    "Mp`\<langle>a,Ap`<b,c>\<rangle> = Ap`\<langle> Mp`<a,b>,Mp`<a,c>\<rangle>"
    "Mp`\<langle> Ap`<b,c>,a \<rangle> = Ap`\<langle> Mp`<b,a>,Mp`<c,a>\<rangle>"
    using EquivClass_1_L8A EquivClass_1_L10 IsDistributive_def
    by auto;
qed;

text{*Distributivity is preserved by 
  projections to quotient spaces, provided both operations are congruent
  with respect to the equivalence relation.*}

lemma EquivClass_2_L4: assumes A1: "IsDistributive(X,A,M)"
  and A2: "equiv(X,r)" 
  and A3: "Congruent2(r,A)" "Congruent2(r,M)"
  shows "IsDistributive(X//r,ProjFun2(X,r,A),ProjFun2(X,r,M))"
proof-;
 let ?Ap = "ProjFun2(X,r,A)" 
 let ?Mp = "ProjFun2(X,r,M)"
 from A1 A2 A3 have
   "\<forall>a\<in>X//r.\<forall>b\<in>X//r.\<forall>c\<in>X//r.
   ?Mp`< a,?Ap`<b,c> > = ?Ap`< ?Mp`<a,b>,?Mp`<a,c> > \<and> 
   ?Mp`< ?Ap`<b,c>,a > = ?Ap`< ?Mp`<b,a>,?Mp`<c,a> >"
   using EquivClass_2_L3 by simp;
 then show ?thesis using IsDistributive_def by simp;
qed;
    
end