(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Int\_ZF.thy}*}

theory Int_ZF = OrderedGroup_ZF + Finite1 + Int:

text{*This theory file is an interface between the old-style Isabelle 
  (ZF logic) material on integers and the IsarMathLib project. Here we
  redefine the meta-level operations on integers 
  (addition and multiplication) to convert them to ZF-functions and show
  that integers form a commutative group with respect to addition and 
  commutative monid with respect to multiplication. Similarly, we redefine the
  order on integers as a relation, that is a subset of $Z\times Z$.*}

section{*Addition and multiplication as ZF-functions.*}

text{*In this section we provide definitions of addition and multiplication
  as subsets of $(Z\times Z)\times Z$.*}

constdefs

  IntegerAddition :: "i"
  "IntegerAddition == { <x,c> \<in> (int\<times>int)\<times>int. fst(x) $+ snd(x) = c}"

  IntegerMultiplication :: "i"
  "IntegerMultiplication == 
    { <x,c> \<in> (int\<times>int)\<times>int. fst(x) $\<times> snd(x) = c}";


text{*IntegerAddition and IntegerMultiplication are functions on 
  int$\times$int. *}

lemma Int_ZF_1_L1: 
  shows "IntegerAddition \<in> int\<times>int \<rightarrow> int"
  "IntegerMultiplication \<in> int\<times>int \<rightarrow> int"
proof -
  have
    "{<x,c> \<in> (int\<times>int)\<times>int. fst(x) $+ snd(x) = c} \<in> int\<times>int\<rightarrow>int" 
    "{<x,c> \<in> (int\<times>int)\<times>int. fst(x) $\<times> snd(x) = c} \<in> int\<times>int\<rightarrow>int"
    using func1_1_L11A by auto
  then show "IntegerAddition \<in> int\<times>int \<rightarrow> int" 
    "IntegerMultiplication \<in> int\<times>int \<rightarrow> int"
    using IntegerAddition_def IntegerMultiplication_def by auto
qed;

text{*IntegerAddition adds integers and IntegerMultiplication multiplies
  integers.*}

lemma Int_ZF_1_L2: assumes A1:"a \<in> int" "b \<in> int"
  shows "IntegerAddition`(<a,b>) = a $+ b"
  "IntegerMultiplication`(<a,b>) = a $\<times> b"
proof -;
  let ?x = "<a,b>"
  let ?c = "a $+ b"
  let ?d = "a $\<times> b"
  from A1 have "fst(?x) $+ snd(?x) = ?c" "fst(?x) $\<times> snd(?x) = ?d"
    by auto;
  with A1 have 
    "<?x,?c> \<in> {<x,c> \<in> (int\<times>int)\<times>int. fst(x) $+ snd(x) = c}"
    "<?x,?d> \<in> {<x,d> \<in> (int\<times>int)\<times>int. fst(x) $\<times> snd(x) = d}"
    by auto;
  then show "IntegerAddition`(?x) = ?c"
    "IntegerMultiplication`(?x) = ?d"
    using IntegerAddition_def IntegerMultiplication_def 
      Int_ZF_1_L1 apply_iff by auto
qed;
 
text{*Integer addition and multiplication are associative.*}

lemma Int_ZF_1_L3: 
  assumes A1:"x \<in> int" "y\<in>int" "z\<in>int"
  shows "IntegerAddition`(<IntegerAddition`(<x,y>),z>) = 
  IntegerAddition`( < x,IntegerAddition`(<y,z>)> )"
   "IntegerMultiplication`(<IntegerMultiplication`(<x,y>),z>) = 
  IntegerMultiplication`( < x,IntegerMultiplication`(<y,z>)> )"
proof -;
  from A1 show  "IntegerAddition`(<IntegerAddition`(<x,y>),z>) = 
    IntegerAddition`( < x,IntegerAddition`(<y,z>)> )"
    "IntegerMultiplication`(<IntegerMultiplication`(<x,y>),z>) = 
    IntegerMultiplication`( < x,IntegerMultiplication`(<y,z>)> )"
    using Int_ZF_1_L2 zadd_assoc zmult_assoc by auto;
qed;

text{*Integer addition and multiplication are commutative.*}

lemma Int_ZF_1_L4:
  assumes A1:"x \<in> int" "y\<in>int"
  shows "IntegerAddition` <x,y> = IntegerAddition` <y,x>"
  "IntegerMultiplication` <x,y> = IntegerMultiplication` <y,x>"
proof -;
  from A1 show "IntegerAddition` <x,y> = IntegerAddition` <y,x>"
    "IntegerMultiplication` <x,y> = IntegerMultiplication` <y,x>"
    using Int_ZF_1_L2 zadd_commute zmult_commute by auto;
qed;

text{*Zero is neutral for addition and one for multiplication.*}

lemma Int_ZF_1_L5: assumes A1:"x \<in> int"
  shows "IntegerAddition`< ($# 0), x > = x \<and>
   IntegerAddition`< x ,($# 0) > = x"
  "IntegerMultiplication` < ($# 1), x > = x \<and> 
  IntegerMultiplication`< x ,($# 1) > = x"
proof -;
  from A1 show "IntegerAddition`< ($# 0), x > = x \<and>
    IntegerAddition`< x ,($# 0) > = x"
    using Int_ZF_1_L2 zadd_int0 Int_ZF_1_L4 by simp;
  from A1 have T1: "IntegerMultiplication` < ($# 1), x > = x"
    using Int_ZF_1_L2 zmult_int1 by simp;
  moreover from A1 T1 have "IntegerMultiplication`< x ,($# 1) > = x"
    using Int_ZF_1_L4 by simp;
  ultimately show  "IntegerMultiplication` < ($# 1), x > = x \<and> 
    IntegerMultiplication`< x ,($# 1) > = x"
    by simp;
qed;
    

text{*Zero is neutral for addition and one for multiplication.*}

lemma Int_ZF_1_L6: "($# 0) \<in> int \<and> 
  (\<forall>x \<in> int. IntegerAddition`< ($# 0), x > = x \<and>
   IntegerAddition`< x ,($# 0) > = x)"
  "($# 1) \<in> int \<and> 
  (\<forall>x \<in> int. IntegerMultiplication` < ($# 1), x > = x \<and> 
  IntegerMultiplication`< x ,($# 1) > = x)"
proof -
  show "($# 0) \<in> int \<and> 
    (\<forall>x \<in> int. IntegerAddition`< ($# 0), x > = x \<and>
    IntegerAddition`< x ,($# 0) > = x)"
    "($# 1) \<in> int \<and> 
    (\<forall>x \<in> int. IntegerMultiplication` < ($# 1), x > = x \<and> 
    IntegerMultiplication`< x ,($# 1) > = x)"
    using Int_ZF_1_L5 by auto;
qed;

text{*Integers with addition and integers with multiplication
  form monoids.*}
 
theorem Int_ZF_1_T1: "IsAmonoid(int,IntegerAddition)"
  "IsAmonoid(int,IntegerMultiplication)"
proof -
   have  
    "\<exists>e \<in> int. (\<forall>x \<in> int.(IntegerAddition` (< e, x >) = x) &
    (IntegerAddition` (< x ,e >) = x))"
     "\<exists>e \<in> int. (\<forall>x \<in> int.(IntegerMultiplication` (< e, x >) = x) &
    (IntegerMultiplication` (< x ,e >) = x))"
     using Int_ZF_1_L6 by auto;
   then show "IsAmonoid(int,IntegerAddition)"
     "IsAmonoid(int,IntegerMultiplication)" using 
     IsAmonoid_def IsAssociative_def Int_ZF_1_L1 Int_ZF_1_L3 
     by auto;
qed;

(*text{*We can use theorems proven in the monoid0 context.*}

lemma Int_ZF_1_L7: shows "monoid0(int,IntegerAddition)"
proof -; 
  show ?thesis using Int_ZF_1_T1 monoid0_def by simp;
qed;*)

text{*Zero is the neutral element of the integers with addition
  and one is the neutral element of the integers with multiplication.*}

lemma Int_ZF_1_L8: "($# 0) = TheNeutralElement(int,IntegerAddition)"
  "($# 1) = TheNeutralElement(int,IntegerMultiplication)"
proof -;
  have "monoid0(int,IntegerAddition)"
    using Int_ZF_1_T1 monoid0_def by simp;
  moreover have 
    "($# 0) \<in> int \<and>
    (\<forall>x \<in> int.(IntegerAddition` (< ($# 0), x >) = x) \<and> 
    (IntegerAddition` (< x ,($# 0) >) = x))"
    using Int_ZF_1_L6 by auto;
  ultimately show "($# 0) = TheNeutralElement(int,IntegerAddition)"
    by (rule monoid0.group0_1_L4);
  have "monoid0(int,IntegerMultiplication)"
    using Int_ZF_1_T1 monoid0_def by simp;
  moreover have "($# 1) \<in> int \<and> 
    (\<forall>x \<in> int. IntegerMultiplication` < ($# 1), x > = x \<and> 
    IntegerMultiplication`< x ,($# 1) > = x)"
    using Int_ZF_1_L6 by auto;
  ultimately show
    "($# 1) = TheNeutralElement(int,IntegerMultiplication)"
    by (rule monoid0.group0_1_L4);
qed;

text{*Each integer has an inverse (in the addition sense).*}

lemma Int_ZF_1_L9: assumes A1:"g \<in> int"
  shows 
  "\<exists> b \<in> int. IntegerAddition`<g,b> = 
  TheNeutralElement(int,IntegerAddition)"
proof -;
  from A1 have "IntegerAddition`<g,$-g> = 
    TheNeutralElement(int,IntegerAddition)"
    using Int_ZF_1_L2 Int_ZF_1_L8 by simp;
  thus ?thesis by auto;
qed;

text{*Integers with addition form an abelian group. This also shows
  that we can apply all theorems proven in the proof contexts (locales) 
  that require the assumpion that some pair of sets form a group like 
  locale group0.*}
 
theorem Int_ZF_1_T2: 
  "IsAgroup(int,IntegerAddition)"
  "IntegerAddition {is commutative on} int"
  "group0(int,IntegerAddition)"
proof -;
  show "IsAgroup(int,IntegerAddition)"
    using Int_ZF_1_T1 Int_ZF_1_L9 IsAgroup_def
    by simp;
  then show "group0(int,IntegerAddition)"
    using group0_def by simp;
  show "IntegerAddition {is commutative on} int"
    using Int_ZF_1_L4 IsCommutative_def by simp;
qed;

section{*Integers as an ordered group*}

text{*In this section we define order on integers as a relation, that is a 
  subset of $Z\times Z$ and show that integers form an ordered group.*}

text{*We use the $\leq $ (higher order) relation defined in Int.thy to 
  define a subset of $Z\times Z$ that constitutes the ZF order relation 
  corresponding to it.*}

constdefs
  IntegerOrder :: "i" 
  "IntegerOrder == {p\<in>int\<times>int. fst(p) $\<le> snd(p)}"

text{*The next lemma interprets the definition one way. *}

lemma Int_ZF_2_L1: assumes A1: "m\<in>int" "n\<in>int" and A2: "m $\<le> n"
  shows "<m,n> \<in> IntegerOrder"
proof -;
  from A1 A2 have "<m,n> \<in> {x\<in>int\<times>int. fst(x) $\<le> snd(x)}" 
    by simp;
  then show ?thesis using IntegerOrder_def by simp;
qed;

text{*The next lemma interprets the definition the second way. *}

lemma Int_ZF_2_L1A: assumes A1: "<m,n> \<in> IntegerOrder"
  shows "m $\<le> n" "m\<in>int" "n\<in>int"
proof -;
  from A1 have  "<m,n> \<in> {p\<in>int\<times>int. fst(p) $\<le> snd(p)}"
    using IntegerOrder_def by simp;
  thus "m $\<le> n" "m\<in>int" "n\<in>int" by auto;
qed;

text{*Integer order is a relation on integers.*}

lemma Int_ZF_2_L1B: "IntegerOrder \<subseteq> int\<times>int"
proof;
  fix x assume "x\<in>IntegerOrder" 
  then have "x \<in> {p\<in>int\<times>int. fst(p) $\<le> snd(p)}"
    using IntegerOrder_def by simp;
  then show "x\<in>int\<times>int" by simp;
qed;

text{* The order on integers is reflexive.*}

lemma Int_ZF_2_L2: "refl(int,IntegerOrder)"
proof -;
  show ?thesis using Int_ZF_2_L1 zle_refl refl_def
    by simp;
qed;

text{*The essential condition to show antisymmetry of the order on integers.*}

lemma Int_ZF_2_L3: 
  assumes A1:"<m,n> \<in> IntegerOrder" "<n,m> \<in> IntegerOrder"
  shows "m=n"
proof -;
  from A1 have "m $\<le> n" "n $\<le> m" "m\<in>int" "n\<in>int"
    using Int_ZF_2_L1A by auto;
  then show "m=n" using zle_anti_sym by auto;
qed;
  
text{* A The order on integers is antisumetric.*}

lemma Int_ZF_2_L4: "antisym(IntegerOrder)"
proof -
  have "\<forall>m n. 
    <m,n> \<in> IntegerOrder \<and> <n,m> \<in> IntegerOrder \<longrightarrow> m=n"
    using Int_ZF_2_L3 by auto;
  then show ?thesis using imp_conj antisym_def by simp;
qed;

text{*The essential condition to show that the order on integers is 
  transitive.*}

lemma Int_ZF_2_L5: 
  assumes A1: "<m,n> \<in> IntegerOrder" "<n,k> \<in> IntegerOrder"
  shows "<m,k> \<in> IntegerOrder"
proof -;
  from A1 have T1: "m $\<le> n" "n $\<le> k" and T2: "m\<in>int" "k\<in>int"
    using Int_ZF_2_L1A by auto;
  from T1 have "m $\<le> k" by (rule zle_trans);
  with T2 show ?thesis using Int_ZF_2_L1 by simp;
qed;

text{*The order on integers is transitive.*}

lemma Int_ZF_2_L6: "trans(IntegerOrder)"
proof -;
  have "\<forall> m n k. 
    (\<langle>m, n\<rangle>\<in>IntegerOrder \<and> \<langle>n, k\<rangle>\<in>IntegerOrder \<longrightarrow> \<langle>m, k\<rangle>\<in> IntegerOrder)"
    using Int_ZF_2_L5 by blast;
  then show ?thesis by (rule Fol1_L2);
qed;

text{*The order on integers is a partial order.*}

lemma Int_ZF_2_L7: "IsPartOrder(int,IntegerOrder)"
proof -;
  show ?thesis using Int_ZF_2_L2 Int_ZF_2_L4 Int_ZF_2_L6 IsPartOrder_def
    by simp;
qed;

text{*The essential condition to show that the order on integers is 
  preserved by translations. *}

lemma Int_ZF_2_L8: assumes A1: "k \<in> int" 
  and A2: "<m,n> \<in> IntegerOrder"
  shows 
  "< IntegerAddition`<m,k>,IntegerAddition`<n,k> > \<in> IntegerOrder"
  "< IntegerAddition`<k,m>,IntegerAddition`<k,n> > \<in> IntegerOrder"
proof -;  
  from A2 have "m $\<le> n" and T1: "m\<in>int" "n\<in>int" 
    using Int_ZF_2_L1A by auto;
  then have "m $+ k $\<le> n $+ k" "k $+ m $\<le> k $+ n"
    using zadd_right_cancel_zle zadd_left_cancel_zle by auto;
  moreover from T1 A1 have 
    "m $+ k \<in> int" "n $+ k \<in> int"
    "k $+ m \<in> int" "k $+ n \<in> int"
    using Int_ZF_1_L2 Int_ZF_1_L1 apply_funtype by auto;
  ultimately have
    "<m $+ k,n $+ k > \<in> IntegerOrder"
    "< k $+ m, k $+ n > \<in> IntegerOrder"
    using Int_ZF_1_L2 Int_ZF_2_L1 by auto;
  moreover from A1 T1 have
    "m $+ k = IntegerAddition`<m,k>" "n $+ k = IntegerAddition`<n,k>"
    "k $+ m = IntegerAddition`<k,m>" "k $+ n = IntegerAddition`<k,n>"
    using Int_ZF_1_L2 by auto;
  ultimately show 
    "< IntegerAddition`<m,k>,IntegerAddition`<n,k> > \<in> IntegerOrder"
    "< IntegerAddition`<k,m>,IntegerAddition`<k,n> > \<in> IntegerOrder"
    by auto;
qed;

text{*Integers form a linearly ordered group.*}

theorem Int_ZF_2_T1: 
  "IsAnOrdGroup(int,IntegerAddition,IntegerOrder)"
  "IntegerOrder {is total on} int"
proof -;
  have "\<forall>k\<in>int. \<forall>m n. <m,n> \<in> IntegerOrder \<longrightarrow> 
    < IntegerAddition`<m,k>,IntegerAddition`<n,k> > \<in> IntegerOrder \<and>
    < IntegerAddition`<k,m>,IntegerAddition`<k,n> > \<in> IntegerOrder"
    using Int_ZF_2_L8 by simp;
  then show "IsAnOrdGroup(int,IntegerAddition,IntegerOrder)" using
    Int_ZF_1_T2 Int_ZF_2_L1B Int_ZF_2_L7 IsAnOrdGroup_def
    by simp;
  show "IntegerOrder {is total on} int"
    using IsTotal_def zle_linear Int_ZF_2_L1 by auto;  
qed;
end






 
