/*
* ext_exec.c

Copyright (C) 2011-2013 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <ctype.h>
#include <errno.h>

#define __USE_BSD 1
// for u_int in db.h
#include <sys/types.h>

#include <sys/stat.h>
#include <stdlib.h>
#include <syslog.h>

#include "config_names.h"
#include "dbstruct.h"
#include "ext_exec.h"

#include <assert.h>

#if defined TEST_MAIN
static char *fname;
#define MAX_CMD_LENGTH 64
#else
static char fname[] = IPQBDB_CONNKILL_CMD;
#define MAX_CMD_LENGTH 8192
#endif

static connkill_cmd *allocate_cmd(int argc, size_t stringsize)
{
	size_t alloc =
		sizeof (connkill_cmd) +
		(argc + 1) * sizeof (char*);      // terminating NULL argument

	// extra space for strings, with a terminating 0 for each
	size_t const size = alloc + stringsize + argc;

	connkill_cmd *cmd = (connkill_cmd*)malloc(size);
	if (cmd)
	{
		memset(cmd, 0, size);
		cmd->argc = argc;
		cmd->argv[0] = (char*)cmd + alloc;     // first of stringspace
		cmd->argv[1] = (char*)cmd + size  - 1; // last of stringspace
	}

	return cmd;
}

static int
parse_arguments(char *s, connkill_cmd *cmd, size_t line_no, app_private *ap)
{
	assert(s && *s);
	assert(cmd == NULL || (cmd->argv[0] != NULL && cmd->argv[1] != NULL));

	char *first = cmd? cmd->argv[0]: NULL;
	char const *const last = cmd? cmd->argv[1]: NULL;

	size_t count = 0;
	int in_quote = 0, escaped = 0, quote = 0, seenspace = 0, ch;

	if (cmd) cmd->argv[count] = first;

	while ((ch = *(unsigned char*)s++) != 0)
	{
		(void)last; // only used in assertion, n.u. with NDEBUG
		assert(cmd == NULL || first <= last);
		assert(cmd == NULL || count <= cmd->argc);

		if (ch == '\\')
		{
			escaped = !escaped;
			if (escaped)
				continue;
		}

		if (escaped)
		{
			if (cmd) *first++ = ch;
			escaped = 0;
			continue;
		}

		if (in_quote)
		{
			if (ch == quote)
			{
				in_quote = 0;
				continue;
			}
		}
		else
		{
			if (isspace(ch))
			{
				if (!seenspace)
				{
					seenspace = 1;
					++count;
					if (cmd)
					{
						*first++ = 0;
						cmd->argv[count] = first;
					}
				}
				continue;
			}

			seenspace = 0;
			if ((ch == '\'' || ch == '\"'))
			{
				quote = ch;
				in_quote = 1;
				continue;
			}
		}

		if (cmd) *first++ = ch;
	}
	
	if (ap && (in_quote || escaped))
	{
		report_error(ap, LOG_WARNING,
			"dangling escape or unterminated string at line %zu in %s\n",
			line_no, fname);
	}	

	return count + 1;  // end of string terminates arg (trailing 0 in place)
}

static connkill_cmd *
read_conkill_line(FILE *fp, size_t *line_no, app_private *ap)
{
	char buf[MAX_CMD_LENGTH], *rbuf = &buf[0];
	char *const ebuf = &buf[MAX_CMD_LENGTH];

	while (fgets(rbuf, ebuf - rbuf, fp) != NULL)
	{
		int ch;
		char *s = &buf[0];       // parse from the beginning of buf
		char *z = s + strlen(s); // check line fits in buffer

		++*line_no;

		while ((ch = *(unsigned char*)s) != 0 && isspace(ch)) // left trim
			++s;
		
		if (z + 1 < ebuf || z[-1] == '\n')
		{
			while (--z > s && isspace(*(unsigned char*)z))  // right trim
				*z = 0;

			if (z >= rbuf && *z == '\\') // escaped newline
			{
				*z = '\n';
				rbuf = z + 1;
				assert(rbuf + 1 < ebuf);
				continue;
			}
		}
		else
		{
			report_error(ap, LOG_CRIT,
				"line %zu too long in %s (max line length=%u)\n",
				*line_no, fname, MAX_CMD_LENGTH);
			return NULL;
		}

		rbuf = &buf[0];
		if (s > z || *s == '#') // comment or empty line
			continue;

		int argc = parse_arguments(s, NULL, *line_no, ap);
		// add one argument for default arg, and reserve string space as needed
		connkill_cmd *cmd = allocate_cmd(argc + 1, z - s + 1);
		if (cmd)
			parse_arguments(s, cmd, 0, NULL);

		assert(cmd == NULL || (cmd->argv != NULL && cmd->argv[0] != NULL));
		return cmd;
	}

	return NULL;
}

connkill_cmd* read_connkill_cmd(app_private *ap)
/*
* the command has to be written on a file owned by root.
* this is only called if the --exec option was found in ibd-parse or ibd-ban.
*/
{
	static const char wont_kill[] = ", established connections won't be killed!";
	struct stat buf;
	if (stat(fname, &buf))
	{
		report_error(ap, LOG_CRIT, "cannot stat %s (%s)%s\n",
			fname, strerror(errno), wont_kill);
		return NULL;
	}

#if defined TEST_MAIN
#define EXPECTED_UID getuid()
#define EXPECTED_GID getgid()
#else
#define EXPECTED_UID 0
#define EXPECTED_GID 0
#endif

	if (buf.st_uid != (EXPECTED_UID) ||
		((buf.st_mode & S_IWGRP) != 0 && buf.st_gid != (EXPECTED_GID)) ||
		(buf.st_mode & S_IWOTH) != 0 ||
		!S_ISREG(buf.st_mode))
	{
		report_error(ap, LOG_CRIT,
			"%s must be a regular file, owned by root, and not world writable%s\n",
			fname, wont_kill);
		return NULL;
	}

	if (((buf.st_mode & S_IWUSR) != 0 && (buf.st_mode & S_IRUSR) == 0) ||
		((buf.st_mode & S_IWGRP) != 0 && (buf.st_mode & S_IRGRP) == 0) /* ||
		((buf.st_mode & S_IWOTH) != 0 && (buf.st_mode & S_IROTH) == 0) */)
	/*
	* someone can write it but cannot read it
	*/
	{
		report_error(ap, LOG_CRIT,
			"%s must be readable by whoever can write it%s\n",
			fname, wont_kill);
		return NULL;
	}

	if ((buf.st_mode & S_IXUSR) != 0 &&
		((buf.st_mode & S_IRGRP) == 0 || (buf.st_mode & S_IXGRP) != 0) &&
		((buf.st_mode & S_IROTH) == 0 || (buf.st_mode & S_IXOTH) != 0))
	/*
	* executable by owner (and by users with read access)
	* it will be called with IP argument
	*/
	{
		connkill_cmd *cmd = allocate_cmd(2, 0);
		if (cmd)
		{
			cmd->argv[0] = fname;
			cmd->ip_ndx = 1;
		}

		return cmd;
	}
	else if ((buf.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)) != 0)
	/*
	* ambiguos
	*/
	{
		report_error(ap, LOG_CRIT,
			"%s must be executable by root and whoever can read it,"
			" or not executable at all%s\n",
			fname, wont_kill);
		return NULL;
	}	

	/*
	* not executable
	* it must contain exactly one line which is a possible command,
	* no line must be longer than MAX_CMD_LENGTH
	*/

	connkill_cmd *cmd = NULL;
	size_t line_no = 0;
	FILE *fp = fopen(fname, "r");
	if (fp == NULL)
	{
		report_error(ap, LOG_CRIT, "cannot read %s: %s\n",
			fname, strerror(errno));
		return NULL;
	}
	
	while (!feof(fp) && !ferror(fp))
	{
		connkill_cmd *tcmd = read_conkill_line(fp, &line_no, ap);
		if (tcmd == NULL)
		/*
		* possibly eof, line too long, alloc failure, ...
		*/
			break;

		if (cmd == NULL)
		{
			static const char cmd_ignored[] = ", command ignored";
			for (size_t i = 1; i < tcmd->argc; ++i)
			{
				char const *const a = tcmd->argv[i];
				if (a && strcmp(a, "{}") == 0)
				{
					if (tcmd->ip_ndx == 0)
						tcmd->ip_ndx = i;
					else
						report_error(ap, LOG_WARNING,
							"ignoring extra argument \"{}\""
							" at position %zu in %s\n",
							i, fname);
				}
			}
			if (tcmd->ip_ndx == 0) // default arg
				tcmd->ip_ndx = tcmd->argc - 1;

			if (stat(tcmd->argv[0], &buf))
			{
				report_error(ap, LOG_WARNING, "cannot stat %s (%s)%s\n",
					tcmd->argv[0], strerror(errno), cmd_ignored);
				free(tcmd);
			}
			else if (!S_ISREG(buf.st_mode) || (buf.st_mode & S_IXUSR) == 0)
			{
				report_error(ap, LOG_WARNING,
					"%s is not a regular executable file%s\n",
					tcmd->argv[0], cmd_ignored);
				free(tcmd);
			}
			else
				cmd = tcmd;
		}
		else
		{
			report_error(ap, LOG_WARNING,
				"ignoring extraneous text after command, at line %zu of %s\n",
				line_no, fname);
			free(tcmd);
		}
	}

	if (ferror(fp))
		report_error(ap, LOG_ERR, "error reading %s: %s\n",
			fname, strerror(errno));

	fclose(fp);
	if (cmd == NULL)
		report_error(ap, LOG_CRIT, "no valid command found in %s%s\n",
			fname, wont_kill);

	return cmd;
}

static inline char *discard_const(char const* ip) { return (char*)ip;}

int system_exec(connkill_cmd *cmd, char const *ip, app_private *ap)
{
	assert(cmd);
	assert(cmd->argv);
	assert(ip);

	cmd->argv[cmd->ip_ndx] = discard_const(ip);
	unsigned needed = cmd->argc + 1;  // spaces + term. 0
	for (unsigned i = 0; i < cmd->argc; ++i)
		if (cmd->argv[i])
			needed += strlen(cmd->argv[i]);

	char command[needed];
	char *p = &command[0];
	for (unsigned i = 0; i < cmd->argc; ++i)
		if (cmd->argv[i])
			p += sprintf(p, "%s ", cmd->argv[i]);
	*p = 0;

#if defined TEST_MAIN
	printf("running \"%s\"\n", command);
#endif

	int rtc = system(command);
	if (rtc == -1)
		report_error(ap, LOG_CRIT, "cannot run \"%s\": %s\n",
			command, strerror(errno));

	return rtc;
}

void run_exec(connkill_cmd *cmd, char const *ip, app_private *ap)
{
	assert(cmd);
	assert(ip);

	cmd->argv[cmd->ip_ndx] = discard_const(ip);
	execv(cmd->argv[0], &cmd->argv[0]);
	report_error(ap, LOG_CRIT, "cannot run %s: %s\n",
		cmd->argv[0], strerror(errno));
	_exit(1);
}


#if defined TEST_MAIN

void display_result(connkill_cmd *cmd, int rtc)
{
	char const *result;
	char buf[40];
	if (rtc == -1)
	{
		result = "couldn't run";
	}
	else if (rtc)
	{
		if (WIFSIGNALED(rtc))
		{
			result = "was signaled";
		}
		else
		{
			int r = WEXITSTATUS(rtc);
			if (r == 127)
			{
				result = "exited with status 127, no shell?";
			}
			else
			{
				sprintf(buf, "exited with status %d", r);
				result = buf;
			}
		}
	}
	else
		result = "exited successfully";

	printf("%s %s\n", cmd->argv[0]? cmd->argv[0]: "NULL", result);
}

int main(int argc, char *argv[])
{
	int verbose = 0, run4 = 0, run6 = 0;
	app_private ap;
	ap.mode = error_report_stderr;
	ap.err_prefix = "TESTexec";

	for (int i = 1; i < argc; ++i)
	{
		char *a = argv[i];

		if (strcmp(a, "--version") == 0)
		{
			printf("%s: version " PACKAGE_VERSION "\n", ap.err_prefix);
			return 0;
		}
		
		if (strcmp(a, "--help") == 0)
		{
			printf("%s: [-v] file...\n"
			"Displays how file is parsed and which argument (ip_ndx) gets the IP.\n"
			"File(s) must be owned by the current user/group rather than root\n"
			"Option -v also displays argument numbers; --version works\n"
			"option -r runs the command with argument 127.0.0.1,\n"
			"option -R runs the command with argument ::1.\n",
				argv[0]);
			return 0;
		}

		if (strcmp(a, "-v") == 0)
		{
			verbose = 1;
			continue;
		}

		if (strcmp(a, "-r") == 0)
		{
			run4 = 1;
			continue;
		}

		if (strcmp(a, "-R") == 0)
		{
			run6 = 1;
			continue;
		}

		fname = a;
		connkill_cmd *cmd = read_connkill_cmd(&ap);
		puts(fname);
		if (cmd)
		{
			printf("ip_ndx: %u\nargc: %u\n", cmd->ip_ndx, cmd->argc);
			for (unsigned j = 0; j <= cmd->argc; ++j)
			{
				if (verbose)
					printf("%u: ", j);
				puts(cmd->argv[j]? cmd->argv[j]: "(NULL)");
			}

			if (run4)
			{
				int rtc = system_exec(cmd, "127.0.0.1", &ap);
				display_result(cmd, rtc);
			}

			if (run6)
			{
				int rtc = system_exec(cmd, "::1", &ap);
				display_result(cmd, rtc);
			}

			free(cmd);
		}
	}
	return 0;
}
#endif

