/*
* opendb.c - written by vesely in milano on 13 sep 2008.

Copyright (C) 2008-2019 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#if ! defined _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>

#include <db.h>
#include <limits.h>
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "config_names.h"
#include "dbstruct.h"
#include <string.h>

#if !defined PATH_MAX
#define PATH_MAX 1024
#endif

#include <assert.h>

static int isdir(char const *p)
{
	struct stat st;
	int rtc = stat(p, &st);

	if (rtc == 0)
	{
		if (!(S_ISDIR(st.st_mode)))
		{
			rtc = -1;
			errno = ENOTDIR;
		}
	}

	return rtc;
}

static int createdir (char *fname)
{
	int rtc = 0;
	char *p = strrchr(fname, '/');
	if (p)
	{
		if (p == fname)
			++p;
		*p = 0;
		if ((rtc = isdir(fname)) != 0 &&
			(errno == ENOENT || errno == ENOTDIR) &&
			(rtc = createdir(fname)) == 0)
				rtc = mkdir(fname, 0700);
		if (rtc == 0)
			*p = '/';
	}
	return rtc;
}

static void
db_errcall_fcn(const DB_ENV* env, char const *err_prefix, char const *msg)
{
	app_private*ap = NULL;
	if (env)
		ap = (app_private*) env->app_private;
	report_error(ap, LOG_ERR, "db error: %s\n", msg);
	(void)err_prefix;
}

void fputs_database_help()
{
	fputs("\nDatabase filenames are prefixed with \""
		IPQBDB_DATABASE_PREFIX
		"\" (unless they start with '/'),\nand suffixed with \""
		IPQBDB_DATABASE_SUFFIX
		"\" (unless they already end with it).\n", stdout);
}

char *database_fname(char const *name, app_private *ap)
{
	return database_fname_with_home(name, ap, NULL, 0);
}

char *database_fname_with_home(char const *name, app_private *ap,
	char const *home, int add_slash_if)
{
	static const char db_prefix[] = IPQBDB_DATABASE_PREFIX;
	static const char db_suffix[] = IPQBDB_DATABASE_SUFFIX;
	char fname[PATH_MAX];
	char const *prefix = home? home: db_prefix;

	// check correct version
	int major = 0, minor = 0;
	char const *libv = db_version(&major, &minor, NULL);
	if (major != DB_VERSION_MAJOR || minor != DB_VERSION_MINOR)
	{
		report_error(ap, LOG_CRIT, "running %s, while app compiled with "
			DB_VERSION_STRING "\n", libv);
		return NULL;
	}

	// check lengths of names
	unsigned lname = strlen(name), lsuf = strlen(db_suffix);
	unsigned lpre = (name[0] != '/')? strlen(prefix): 0;
	char const *p = name + lname - lsuf;

	if (p >= name && strcmp(p, db_suffix) == 0)
		lsuf = 0;

	if (lpre + lname + lsuf + 1 >= sizeof fname)
	{
		report_error(ap, LOG_CRIT, "name too long: %s%s%s\n",
			name[0] != '/'? prefix: "", name, lsuf? db_suffix: "");
		return NULL;
	}

	// build fname
	if (name[0] != '/')
	{
		if (prefix[0] != '/')
		{
			report_error(ap, LOG_CRIT, "invalid name: %s%s%s\n",
				prefix, name, lsuf? db_suffix: "");
			return NULL;
		}
		assert(lpre); // at least "/"
		strcpy(fname, prefix);
		if (fname[lpre - 1] != '/' && add_slash_if)
			fname[lpre++] = '/';
	}
	strcpy(&fname[lpre], name);
	if (lsuf)
		strcat(&fname[lpre + lname], db_suffix);

	// make sure the dir exists, possibly make it
	errno = 0;
	if (createdir(fname))
	{
		report_error(ap, LOG_CRIT, "cannot createdir %s: %s\n",
			fname, strerror(errno));
		return NULL;
	}
	return strdup(fname);
}

int open_database(char *fname, app_private *ap, DB_ENV **db_env, DB** db)
{
	DB_ENV *de = *db_env;
	char const *what = NULL;
	int rtc;

	if (de == NULL)
	{
		char * const p = strrchr(fname, '/'); // full path MUST by caller
		rtc = db_env_create(&de, 0);
		if (rtc)
		{
			report_error(ap, LOG_CRIT,
				"cannot create DB: %s\n", db_strerror(rtc));
			return 1;
		}

		de->set_errpfx(de, ap->err_prefix);
		de->set_errcall(de, db_errcall_fcn);
		de->app_private = ap;

		// set a timeout of 4 seconds. this can be overridden by setting, e.g.,
		// "set_lock_timeout 5000000" in DB_CONFIG.
		de->set_timeout(de, 4000000U, DB_SET_LOCK_TIMEOUT);

		assert(p);
		if (p)
			*p = 0;

		rtc = de->open(de, p? fname: ".",
			/* DB_SYSTEM_MEM for shmget rather than mmap?
			* it requires the set_shm_key method, that can be done in DB_CONFIG
			*
			* CDB is the Concurrent Datastore (not transactioned) environment.
			*/
			DB_CREATE | DB_INIT_MPOOL | DB_INIT_CDB,
			/*
			* mode == 0 implies using the default rw by user and group
			* otherwise see chmod(2)
			*/
			0);
		if (p)
			*p = '/';

		if (rtc)
		{
			de->close(de, 0);
			de = NULL;
			what = "DB_ENV->open";
			goto error_exit;
		}
	}

	if (db)
	{
		rtc = db_create(db, de, 0);
		if (rtc)
		{
			what = "db_create";
			*db = NULL;
			goto error_exit;
		}

		rtc = (*db)->open(*db, NULL, fname, NULL, DB_BTREE, DB_CREATE, 0);
		if (rtc)
		{
			(*db)->close(*db, 0);
			*db = NULL;
			what = "DB->open";
			goto error_exit;
		}
	}

	*db_env = de;
	return 0;

	error_exit:
	{
		report_error(ap, LOG_CRIT, "cannot %s for %s: %s\n",
			what, fname, db_strerror(rtc));
		return 2;
	}
}

int close_db(DB *db)
{
	int rtc = 0;
	if (db)
	{
		DB_ENV *db_env = db->get_env(db);
		rtc = db->close(db, 0);
		if (rtc && db_env)
			db_env->err(db_env, rtc, "close");
	}
	return rtc;
}

/*
* if cleanup is non-zero, this will remove __db.00?
*/
int close_dbenv(DB_ENV *db_env, int cleanup)
{
	int rtc = 0, rtc2 = 0;
	if (db_env)
	{
		char const *home = NULL;
		char *home2 = NULL;
		app_private *ap = (app_private*) db_env->app_private;
		if (cleanup && (rtc2 = db_env->get_home(db_env, &home)) == 0 && home)
			home2 = strdup(home);
		rtc = db_env->close(db_env, 0);
		if (rtc && ap)
			report_error(ap, LOG_ERR, "error closing environment: %s\n",
				db_strerror(rtc));
		if (home2)
		{
			DB_ENV *de = NULL;
			if ((rtc2 = db_env_create(&de, 0)) == 0 && de)
				rtc2 = de->remove(de, home2, 0);
			if (rtc2 && home2 && ap)
				report_error(ap, LOG_INFO, "not cleaning up %s: %s\n",
					home2, db_strerror(rtc2));
			free(home2);
		}
	}
	return rtc;
}

#if defined TEST_MAIN

int main(int argc, char *argv[])
{
	int i;
	DB_ENV *db_env;
	DB *db = NULL;
	for (i = 1; i < argc; i += 1)
	{
		char *a = argv[i];
		char tst = *a++;
		switch (tst)
		{
			case 'c': createdir(a); break;
			case 'o': // open
			{
				app_private ap;

				ap.mode = 0;
				if (db || db_env)
					printf("already have %s%s%s open\n",
						db? "db": "",
						db && db_env? " and ": "",
						db_env? "db_env": "");
				else
					open_database(a, "test", &ap, &db_env, &db);
				break;
			}

			default: printf("invalid '%c' in %s\n", tst, argv[i]); break;
		}
	}
	if (db)
		db->close(db, 0);
	if (db_env)
		db_env->close(db_env, 0);
	return 0;
}

#endif // TEST_MAIN
