/*
* ipv4_util.c

Copyright (C) 2008-2010 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "ipv4_util.h"
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <arpa/inet.h> // for htonl (or is it <netinet/in.h>?)

#include <assert.h>

char const *parse_ip_invalid_what(int err)
{
	switch (err)
	{
		case parse_ip_ok: return "no error";
		case parse_ip_invalid_too_high: return "IP address with high octet";
		case parse_ip_invalid_too_long: return "IP address with many octets";
		case parse_ip_invalid_separator: return "separator after IP address";
		case parse_ip_invalid_range: return "IP range";
		case parse_ip_invalid_cidr: return "IP CIDR";
		default: return "internal error";
	}
}

int parse_ip_address(char *ip_address, ip4_range *ip, char **term)
{
	int i, sep, mode_flag = 0;
	union bytewise
	{
		uint32_t ip;
		unsigned char b[4];
	} uf, ul, *un = &uf;
	char *p = ip_address;

	memset(&ul, 0, sizeof ul);
	memset(&uf, 0, sizeof uf);

	/*
	* parse any of "1", "1.2", "1.2.3.4", "1.2.3-1.2.3.7", "1.2.3.0/29", ecc
	* store first ip in uf, other stuff in ul
	*/
	for (i = 0;;)
	{
		char *t = NULL;
		unsigned long l = strtoul(p, &t, 10);

		if (l > UCHAR_MAX)
			return parse_ip_invalid_too_high;

		else
		{
			un->b[i] = (unsigned char)l;
			p = t;
			sep = *p;
			if (sep == '.' && mode_flag != '/')
			{
				if (++i >= 4)
					return parse_ip_invalid_too_long;
			}
			else if (mode_flag == 0 && (sep == '-' || sep == '/'))
			{
				mode_flag = sep;
				i = 0;
				un = &ul;
			}
			else
				break;

			p += 1;
		}
	}

/*
* if the terminator is taken, set it to p later,
* otherwise if the string is not 0-terminated fail
*/
	if (sep != 0 && term == NULL)
		return parse_ip_invalid_separator;

	/*
	* set first and last ip according to parsed values
	* (ip is untouched on error)
	*/
	if (mode_flag == '-')
	{
		if (memcmp(&uf, &ul, sizeof uf) > 0)
			return parse_ip_invalid_range;

		ip->last = ul.ip;
	}
	else if (mode_flag == '/')
	{
		if (ul.b[0] > 32)
			return parse_ip_invalid_cidr;

		if (ul.b[0])
		{
			ip->last = uf.ip;
			if (ul.b[0] < 32)
				ip->last |= htonl((1U << (32 - ul.b[0])) - 1);
		}
		else
			ip->last = -1;
	}
	else
		ip->last = uf.ip;

	assert(sep == *p);
	if (term)
		*term = p;

	ip->first = uf.ip;

	return 0;
}
