/* pqueue.c -- Hybrid priority queue with some search functions
   Copyright (C) 2004 - 2005 Matas Aguirre - Gonzalo Saavedra

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

/*
 * Contributions and ideas
 *      2005 - Maximiliano Pin
 */

/* TODO
 * 1. best return values
 * 2. re-priority functions
 * 3. a lot of things
 * 4. cleanups, there are so many unused static functions
 */

#include "pqueue.h"

static int is_extern(int,pqueue *);
static int min_son_pos(int);
static int max_son_pos(int);
static void change(int,int,pqueue *);
static int is_root(int);
static int father_pos(int);
static pq_node ** new_heap(int);
static void sink(int, pqueue *);
static int pos(prio_t,pqueue *);
static int pq_requeue(pqueue *);
static void float_node(int,pqueue *);
static void positionate(int,pqueue *);


/* Functions */

/* true if pos is an extern queue node */
int is_extern(int pos,pqueue *q){
    return pos>=q->csize;
}

/* minor son position */
int min_son_pos(int pos){
    return 2*pos;
}

/* major son position */
int max_son_pos(int pos){
    return 2*pos+1;
}

/* switch two heap positions */
void change(int p1,int p2,pqueue *q){
    pq_node * aux=q->heap[p1];
    q->heap[p1]=q->heap[p2];
    q->heap[p2]=aux;
}

/* true if pos is root's position */
int is_root(int pos){
    return pos==1;
}

/* father position of node in pos */
int father_pos(int pos){
    return pos/2;
}

/*
 * build the queue
 * with a heap of sz or DEFAULT_SZ size
 */
int pq_new(pqueue *q,int sz,int(*eq_f)(void*,void*)){
    int aux_sz;
    if(!q)
        return PQERR;
    if(q->csize)
        return NEPTYERR;
    aux_sz=MAX(sz,DEFAULT_SZ);
    q->heap=new_heap(aux_sz);
    q->size=aux_sz;
    q->csize=0;
    q->equals_f=eq_f?eq_f:NULL;
    return POK;
}

/* returns a heap pointer with sz size */
pq_node ** new_heap(int sz){
    return (pq_node**)calloc(sz+1,sizeof(pq_node*));
}

/*
 * adds the data d in the queue with priority p
 * negative priorities are ok
 */
prio_t pq_add(void * d,pqueue *q){
    int pos;
    prio_t p;
    if(!q)
        return PQERR;
    if(q->csize==q->size)
        pq_requeue(q);

    pos=++q->csize;

    q->heap[pos] = (pq_node *)d;
    p = ((pq_node *)d)->pri;
    float_node(pos,q);

    return p;
}

/* resize the queue q */
int pq_requeue(pqueue *q){
    int i;
    pq_node **old_heap;
    if(!q)
        return PQERR;

    old_heap=q->heap;
    if(!(q->heap=(pq_node**)realloc(q->heap,((q->size+1)<<1)*sizeof(pq_node*)))){
        q->heap=old_heap;
        return PQERR;
    }

    q->size=(q->size<<1)+1;
    i=q->csize+1;
    for(;i<(q->size+1);i++)
        q->heap[i]=NULL;

    return POK;
}

/* return the minor priority element */
void * pq_min(pqueue *q){
    if(q->csize==0)
        return NULL;
    return (void *)q->heap[1];
}

/* delete the minor priority element */
void * pq_del_min(pqueue *q){
    pq_node *pqn;
    if(q->csize==0)
        return NULL;

    pqn=q->heap[1];
    q->heap[1]=q->heap[q->csize--];

    sink(1,q);
    q->heap[q->csize+1]=NULL;

    return (void *)pqn;
}

/* sink the node to its correct position */
void sink(int pos, pqueue *q){
    int son;
    pq_node *tmp;
    if(!q || pos<=0 || pos > q->csize)
        return;
    tmp=q->heap[pos];
    for(; pos*2<(q->csize+1); pos=son){
        son=pos*2;
        if(son!=q->csize && PRI_GREATER(q->heap[son+1]->pri,q->heap[son]->pri))
            son++;
        if(PRI_GREATER(q->heap[son]->pri,tmp->pri))
            q->heap[pos]=q->heap[son];
        else
            break;
    }
    q->heap[pos]=tmp;
}

/* empty the queue */
int pq_empty(pqueue *q){
    if(!q)
        return PQERR;
    for(;q->csize;q->csize--)
        free(q->heap[q->csize]);
    return POK;
}

/* return the element with priority p */
void * pq_get(prio_t p,pqueue *q){
    int i;
    if(!q || !q->heap || (i=pos(p,q))==PRIERR)
        return NULL;
    return (void *)q->heap[i];
}

/* return the position in the heap of the element with priority p */
int pos(prio_t p,pqueue *q){
    int i;
    if(!q || !q->heap)
        return PQERR;
    for(i=1; i<(q->csize+1) && !PRI_EQ(p,q->heap[i]->pri) ;i++)
        ;
    return i>q->csize ? PRIERR : i;
}

/* delete the element with priority p */
void * pq_del(prio_t p,pqueue *q){
    int i;
    pq_node *aux;
    if(!q || !q->heap || (i=pos(p,q))==PRIERR)
        return NULL;
    aux=q->heap[i];
    q->heap[i]=q->heap[q->csize];
    q->csize--;

    positionate(i,q);

    return (void *)aux;
}


/*
 * execs pq->equals_f over each element in the queue
 * if it returns a true value, then the data
 * will be deleted
 */
void * pq_del_func(void *data2, pqueue *pq){
    int i;
    void *data;
    if(!pq || !pq->equals_f || !data2)
        return NULL;

    for(i=1;i<(pq->csize+1);i++){
        if(pq->equals_f((void *)pq->heap[i],data2)){
	    data=(void *)pq->heap[i];

            pq->heap[i]=pq->heap[pq->csize];
            pq->csize--;
            positionate(i,pq);
            return data;
        }
    }
    return NULL;
}

/*
 * execs pq->equals_f over each element in the queue
 * if it returns a true value, then the data will be returned
 */
void * pq_get_func(void *data2, pqueue *pq){
    int i;
    if(!pq || !pq->equals_f || !data2)
        return NULL;

    for(i=1;i<(pq->csize+1);i++){
        if(pq->equals_f((void *)pq->heap[i],data2))
            return (void *)pq->heap[i];
    }
    return NULL;
}

/*
 * removes element pointed by 'node'
 * returns 'node' if found, NULL if not
 */
void * pq_del_ptr(void *node, pqueue *pq)
{
    int i;
    if(!pq || !node)
        return NULL;

    for(i=1;i<(pq->csize+1);i++){
        if((void *)pq->heap[i] == node){
            pq->heap[i]=pq->heap[pq->csize];
            pq->csize--;
            positionate(i,pq);
            return node;
        }
    }
    return NULL;
}

/*
 * call pq_empty and free the heap
 */
int pq_delete(pqueue *q){
    if(!q)
        return PQERR;
    if(pq_empty(q)==PQERR)
        return PQERR;
    free(q->heap);
    return POK;
}

/*
 * change the priority to p, of the first node which
 * data is equal to data
 */
int pq_chpri_func(void *data, prio_t p, pqueue *q){
    int i;
    if(!data || !q)
        return PQERR;

    for(i=1;i<(q->csize+1);i++){
        if(q->equals_f((void *)q->heap[i],data))
            break;
    }

    if(i>q->csize)
        return NFOUNDERR;

    q->heap[i]->pri=p;

    positionate(i,q);
    return p;
}

/*
 * change the priority to p, of element pointed by 'node'
 */
int pq_chpri_ptr(void *node, prio_t p, pqueue *q){
    int i;
    if(!node || !q)
        return PQERR;

    for(i=1;i<(q->csize+1);i++){
        if((void *)q->heap[i] == node)
            break;
    }

    if(i>q->csize)
        return NFOUNDERR;

    q->heap[i]->pri=p;

    positionate(i,q);
    return p;
}

void float_node(int i,pqueue *q){
    pq_node *node=q->heap[i];
    prio_t p=node->pri;
    for(; i>1 && PRI_GREATER(p,q->heap[i/2]->pri); i/=2)
        q->heap[i]=q->heap[i/2];
    q->heap[i]=node;
}

void positionate(int son_pos,pqueue *q){
    prio_t son_p, dad_p;
    if(!(son_pos/2))
        return sink(son_pos,q);

    son_p=q->heap[son_pos]->pri;
    dad_p=q->heap[son_pos/2]->pri;

    if(PRI_GREATER(son_p,dad_p))
        float_node(son_pos,q);
    else
        sink(son_pos,q);
}
