/* demux.c -- I/O demultiplexer, timer and secure signal handler
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#define _POSIX_SOURCE

#if HAVE_CONFIG_H
#include <config.h>
#else
#define HAVE_SYSINFO 1
#define RETSIGTYPE void
#endif

#include <sys/time.h>		/* select */
#include <sys/types.h>		/* select */
#include <unistd.h>		/* select */
#include <time.h>		/* time, localtime, mktime */
#include <stdio.h>		/* stdlib.h (on some systems), printf, perror */
#include <stdlib.h>		/* malloc, free */
#include <errno.h>		/* errno */
#include <signal.h>		/* signal */
#if (HAVE_SYSINFO && HAVE_STRUCT_SYSINFO_UPTIME)
#include <sys/sysinfo.h>	/* sysinfo */
#endif
#include "demux.h"
#include "pqueue.h"		/* priority queue */

/*
#define DEMUX_DEBUG 1
*/

#ifdef DEMUX_DEBUG
# define DEB(x) x
# define DPRINTF(fmt, args...) printf( fmt, ## args )
# define DPERROR(x) perror( x )
#else
# define DEB(x)
# define DPRINTF(fmt, args...)
# define DPERROR(x)
#endif

extern int errno;

/* Node in file descriptor lists. */
struct fd_list_node {
	int     fd;
	void    (*callback) (int, void *);
	void   *data;
	int     remove;
	struct fd_list_node *next;
};

/* Node in timer queue. */
struct timer_pq_node {
	prio_t next_alarm;             /* must be the first field for pqueue */
	unsigned long int time_incr;
	long int time_shift;
	void (*callback) (void *);
	void *data;
};

/* List of file descriptors. */
static struct fd_list_node *input_list = 0, *output_list = 0;

/* Timer queue (all fields automatically initialized to 0). */
static pqueue timer_q;

static volatile unsigned int recvd_signals = 0;

#define MAX_SIGNALS (8 * sizeof( recvd_signals ))
static RETSIGTYPE (*sig_handlers[MAX_SIGNALS]) (int);

static int demux_enable;

/* Prototypes */
static long get_uptime ();
static void fd_list_add (struct fd_list_node **list, int fd,
			 void (*callback) (int, void *), void *data);
static void * fd_list_remove (struct fd_list_node **list, int fd);
static void fd_list_purge (struct fd_list_node **list);
static void fd_list_cleanup (struct fd_list_node **list);
static timer_id timer_pq_add (time_t next_alarm,
                              unsigned long int time_incr,
                              long int time_shift,
                              void (*callback) (void *),
                              void *data);
static void timer_pq_cleanup ();
static struct timer_pq_node *find_next_timer ();
static time_t time_quantize (unsigned long int seconds,
			     unsigned long int shift);
static void proc_timer (struct timer_pq_node *timer);
static RETSIGTYPE dmx_sighandler (int sig);
static void dispatch_signals ();

/* Begin demultiplexing. Doesn't return until dmx_stop() is called. */
void
dmx_main_loop ()
{
	struct timer_pq_node *timer;
	struct fd_list_node *node;
	struct timeval tv, *ptv;
	int     retval;
	time_t  now;
	fd_set  rfds, wfds, *prfds, *pwfds;
	int     fd_roof;

	demux_enable = 1;
	do {
		timer = find_next_timer ();
		if (timer) {
			now = get_uptime (NULL);
			if (now >= timer->next_alarm) {
				proc_timer (timer);
				continue;
			}
			tv.tv_sec = timer->next_alarm - now;
			tv.tv_usec = 0;
			ptv = &tv;
		}
		else {
			ptv = (void *)0;
		}

		fd_list_purge (&input_list);
		fd_list_purge (&output_list);
		fd_roof = -1;

		if (input_list) {
			prfds = &rfds;
			FD_ZERO (prfds);
			for (node = input_list; node; node = node->next) {
				FD_SET (node->fd, prfds);
				if (node->fd > fd_roof)
					fd_roof = node->fd;
			}
		}
		else {
			prfds = (void *)0;
		}

		if (output_list) {
			pwfds = &wfds;
			FD_ZERO (pwfds);
			for (node = output_list; node; node = node->next) {
				FD_SET (node->fd, pwfds);
				if (node->fd > fd_roof)
					fd_roof = node->fd;
			}
		}
		else {
			pwfds = (void *)0;
		}

		fd_roof++;

		if (recvd_signals) {
			dispatch_signals ();
			continue;	/* things may have changed */
		}

		/* TODO: Some signal could arrive between last 'if' and next
		   'select', and we wouldn't react to it. This is a difficult
		   race condition. We can fix it using pselect, but it's
		   still not implemented in Linux. */

		retval = select (fd_roof, prfds, pwfds, (void *)0, ptv);
		if (retval > 0) {
			/* Availability */
			if (prfds) {
				DPRINTF ("demux: data available\n");
				for (node = input_list; node;
				     node = node->next) {
					if (FD_ISSET (node->fd, prfds)
					    && !node->remove) {
						node->callback (node->fd,
						                node->data);
					}
				}
			}
			if (pwfds) {
				for (node = output_list; node;
				     node = node->next) {
					if (FD_ISSET (node->fd, pwfds)
					    && !node->remove) {
						node->callback (node->fd,
						                node->data);
					}
				}
			}
		}
		else if (retval == 0) {
			/* Timeout */
			DPRINTF ("demux: timer\n");
			proc_timer (timer);
		}
		else {
			/* Error */
			if (errno != EINTR) {
				perror ("select");
				demux_enable = 0;
			}
		}
		dispatch_signals ();
	} while (demux_enable);

	fd_list_purge (&input_list);
	fd_list_purge (&output_list);
}

/* Clean up all registered fd's and timers. You must free all data
   associated to fd's and timers, e.g. by calling dmx_remove_timer(),
   etc, before calling this, and free()ing the returned pointer. */
void
dmx_cleanup ()
{
	fd_list_cleanup (&input_list);
	fd_list_cleanup (&output_list);
	timer_pq_cleanup();
}

/* Exit dmx_main_loop(). Registered fd's and timers remain. */
void
dmx_stop ()
{
	demux_enable = 0;
}

/* Call 'callback' with 'fd' and 'data' as parameters when some data is
   available for reading in 'fd'. */
void
dmx_add_input_fd (int fd, void (*callback) (int, void *), void *data)
{
	fd_list_add (&input_list, fd, callback, data);
}

/* Call 'callback' with 'fd' and 'data' as parameters when data can be
   sent to 'fd'. */
void
dmx_add_output_fd (int fd, void (*callback) (int, void *), void *data)
{
	fd_list_add (&output_list, fd, callback, data);
}

/* Check if data is available in 'fd'. Returns boolean. */
int
dmx_chk_input_fd (int fd)
{
	fd_set  rfds;
	struct timeval tv;

	FD_ZERO (&rfds);
	FD_SET (fd, &rfds);
	tv.tv_sec = 0;
	tv.tv_usec = 0;

	return select (fd + 1, &rfds, (void *)0, (void *)0, &tv) > 0;
}

/* Check if data can be sent to 'fd'. Returns boolean. */
int
dmx_chk_output_fd (int fd)
{
	fd_set  wfds;
	struct timeval tv;

	FD_ZERO (&wfds);
	FD_SET (fd, &wfds);
	tv.tv_sec = 0;
	tv.tv_usec = 0;

	return select (fd + 1, (void *)0, &wfds, (void *)0, &tv) > 0;
}

/* Call 'callback' with 'data' as parameter every 'seconds' seconds,
   taking last midnight plus 'shift' seconds as reference.
   If shift is -1, no reference is taken, and 'callback' is not
   called until 'seconds' seconds have happened. */
timer_id
dmx_add_periodic_alarm (unsigned long int seconds, long int shift,
                        void (*callback) (void *), void *data)
{
	return timer_pq_add (shift < 0 ? get_uptime() + seconds
	                               : time_quantize(seconds, shift),
	                     seconds, shift, callback, data);
}

/* Call 'callback' with 'data' as parameter in 'seconds' seconds from now. */
timer_id
dmx_add_timer (unsigned long int seconds, void (*callback) (void *),
               void *data)
{
	return timer_pq_add (get_uptime() + seconds, 0, 0, callback, data);
}

/* Check if timer needs to be dispatched and do it. Use when you are
   spending too much time in a callback. */
void
dmx_dispatch_timer (timer_id id)
{
	if (get_uptime() >= id->next_alarm)
		proc_timer (id);
}

/* Call 'callback' when signal 'signum' arrives. This function must be used
   instead of signal() if the handler calls demux functions. You may use it
   for other handlers as well, for increased security and stability thru
   synchronous processing of signals. Works like BSD/GNU signal() even in
   POSIX environments, ie. the handler keeps assigned to the signal after
   that signal arrives. */
void
dmx_signal (int signum, void (*callback) (int))
{
	if (signum >= 0 && signum < MAX_SIGNALS) {
		sig_handlers[signum] = callback;
		signal (signum, dmx_sighandler);
	}
	DEB (
		    else
		    printf ("demux: signal number too big!\n");)
}

/* Stop reporting data availability in 'fd'. Returns 'data' as passed
   on creation. */
void *
dmx_remove_input_fd (int fd)
{
	return fd_list_remove (&input_list, fd);
}

/* Stop reporting availability to send data to 'fd'. Return 'data' as
   passed on creation. */
void *
dmx_remove_output_fd (int fd)
{
	return fd_list_remove (&output_list, fd);
}

/* Remove a timeout or periodic alarm. Returns 'data' as passed on creation. */
void *
dmx_remove_timer (timer_id id)
{
	struct timer_pq_node* node;
	void *data = NULL;

	node = (struct timer_pq_node *)pq_del_ptr (id, &timer_q);
	if (node) {
		data = node->data;
		free (node);
	}

	return data;
}

/* Stop reporting signal 'signum'. Restores default action for the signal. */
void
dmx_remove_signal (int signum)
{
	if (signum >= 0 && signum < MAX_SIGNALS) {
		signal (signum, SIG_DFL);
		recvd_signals &= ~(1 << signum);
	}
}

/* Get seconds since last boot. */
static long
get_uptime ()
{
#if (HAVE_SYSINFO && HAVE_STRUCT_SYSINFO_UPTIME)
	struct sysinfo info;

	sysinfo (&info);
	return info.uptime;
#else
	/* TODO: This gives problems if system time changes.
	 * timer_create() seems to have the same problem.
	 * setitimer() maybe too.
	 * sysinfo() is linux-only.
	 * There is some good alternative??? */
	return time (NULL);
#endif
}

/* Add a node to a fd list. */
static void
fd_list_add (struct fd_list_node **list, int fd,
             void (*callback) (int, void *), void *data)
{
	struct fd_list_node *new_node;

	new_node = malloc (sizeof (struct fd_list_node));
	new_node->fd = fd;
	new_node->callback = callback;
	new_node->data = data;
	new_node->remove = 0;
	new_node->next = *list;
	*list = new_node;
}

/* Mark a node of a fd list to be removed. */
static void *
fd_list_remove (struct fd_list_node **list, int fd)
{
	struct fd_list_node *node;

	for (node = *list; node; node = node->next) {
		if (node->fd == fd) {
			node->remove = 1;
			break;
		}
	}

	return node ? node->data : NULL;
}

/* Remove all nodes marked to be removed. */
static void
fd_list_purge (struct fd_list_node **list)
{
	struct fd_list_node *node, *prev = (void *)0, *next;

	for (node = *list; node; node = next) {
		next = node->next;
		if (node->remove) {
			if (prev) {
				prev->next = next;
			}
			else {
				*list = next;
			}
			free (node);
		}
		else
			prev = node;
	}
}

/* Remove all nodes from a fd list. */
static void
fd_list_cleanup (struct fd_list_node **list)
{
	struct fd_list_node *node, *next;

	for (node = *list; node; node = next) {
		next = node->next;
		free (node);
	}
	*list = (void *)0;
}

/* Add a node to the timer queue. */
static timer_id
timer_pq_add (time_t next_alarm, unsigned long int time_incr,
              long int time_shift, void (*callback) (void *),
              void *data)
{
	struct timer_pq_node *new_node;

	if (timer_q.size == 0) {
		/* not yet initialized */
		if (pq_new (&timer_q, 127, NULL) < 0)
			return NULL;
	}

	new_node = (struct timer_pq_node*)malloc (sizeof(struct timer_pq_node));
	new_node->next_alarm = (prio_t)next_alarm;
	new_node->time_incr = time_incr;
	new_node->time_shift = time_shift;
	new_node->callback = callback;
	new_node->data = data;

	pq_add (new_node, &timer_q);

	return new_node;
}

/* Remove all nodes from the timer queue. */
static void
timer_pq_cleanup ()
{
	struct timer_pq_node *node;

	while ((node = (struct timer_pq_node *)pq_del_min (&timer_q)))
		free (node);

	pq_delete (&timer_q);
}

/* Find next timer. */
static struct timer_pq_node *
find_next_timer ()
{
	return (struct timer_pq_node*)pq_min (&timer_q);
}

/* Quantize time in 'seconds' and return the time of the next quantization. */
static time_t
time_quantize (unsigned long int seconds, unsigned long int shift)
{
	time_t  now = time (NULL);
	time_t  upnow = get_uptime ();
	time_t  time_0, time_diff;
	struct tm *timeptr;

	timeptr = localtime (&now);
	timeptr->tm_hour = timeptr->tm_min = timeptr->tm_sec = 0;
	time_0 = mktime (timeptr) + shift;
	time_diff = now - time_0;
	seconds--;

	return upnow + seconds - (time_diff + seconds) % (seconds + 1);
}

/* Process a timer. Call its callback and remove it or update it. */
static void
proc_timer (struct timer_pq_node *timer)
{
	time_t  last_alarm, next_alarm, now;
	void   *data = timer->data;
	void    (*callback) (void *) = timer->callback;
	struct timer_pq_node tmp_node;

	if (timer->time_incr) {
		last_alarm = next_alarm = timer->next_alarm;
		if (timer->time_shift >= 0) {
			next_alarm = time_quantize (timer->time_incr,
			                            timer->time_shift);
		}
		/* TODO: We've got a problem here. Sometimes a periodic
		   alarm raises twice (eg. at 10:59:59 and 11:00:00).
		   By now I fix it avoiding an alarm raising twice in less
		   than two seconds (Max). */
#ifdef DMX_HI_RESOLUTION
		if (next_alarm == last_alarm) {
#else
		if (next_alarm <= (last_alarm + 2)) {
#endif
			/* quantize gave the same time, or shift was
			   -1, meaning quantization is not to be used */
			next_alarm += timer->time_incr;
			now = get_uptime ();
			if (now > next_alarm) {
				/* maybe system time changed forward */
				next_alarm = now + timer->time_incr;
			}
		}
		/* this updates timer->next_alarm */
		pq_chpri_ptr (timer, (prio_t)next_alarm, &timer_q);
	}
	else {
		dmx_remove_timer (timer);
	}

	callback (data);
}

/* Handler for all signals. */
static RETSIGTYPE
dmx_sighandler (int sig)
{
	recvd_signals |= (1 << sig);
}

/* Dispatch all pending signals. */
static void
dispatch_signals ()
{
	unsigned int s;
	int          i;

	while (recvd_signals) {
		DPRINTF ("demux: signal catched\n");
		for (s = 1, i = 0; s; s <<= 1, i++) {
			if (recvd_signals & s) {
				recvd_signals &= ~s;
				/* this is needed for POSIX signal(), and
				   does not hurt for BSD signal() */
				signal (i, dmx_sighandler);
				sig_handlers[i] (i);
			}
		}
	}
}
