/* misc.c -- miscellaneous functions
   Copyright (C) 2004 Julio A. Becerra

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#define _POSIX_SOURCE
#define _ISOC99_SOURCE
#define _XOPEN_SOURCE		/* TODO we should not need this... */
#define _XOPEN_SOURCE_EXTENDED	/* ... bug in GNU libc6? */

#include <ctype.h>  
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "common.h"
#include "misc.h"
#include "user_iface.h"

#ifdef HAVE_ICONV
#include <iconv.h>
#include <langinfo.h>
#endif

#define BUF_SIZE 512

#ifdef HAVE_ICONV
static iconv_t cur_to_utf8 = (iconv_t) -1;
static iconv_t utf8_to_cur = (iconv_t) -1;
#endif

char *
home_path (char *file)
{
	struct passwd *pass;
	size_t size;
	char *buf;
	
	pass = getpwuid (getuid());
	CHECK (pass != NULL);
				
	size = strlen (pass->pw_dir);
	buf = malloc ((size + strlen (file) + 2) * sizeof (char));
	CHECK (buf != NULL);

	strcpy (buf, pass->pw_dir);
	
	buf [size] = '/';
	buf [size + 1] = '\0';

	strcat (buf, file);

	return buf;

error:
	return NULL;
}

	
char *
home_path2 (char *dir, char *file)
{
	struct passwd *pass;
	size_t size;
	char *buf;
	
	pass = getpwuid (getuid());
	CHECK (pass != NULL);
				
	size = strlen (pass->pw_dir);
	buf = malloc ((size + strlen(dir) + strlen(file) + 2) * sizeof (char));
	CHECK (buf != NULL);

	strcpy (buf, pass->pw_dir);
	
	buf [size] = '/';
	buf [size + 1] = '\0';

	strcat (buf, dir);
	strcat (buf, file);

	return buf;

error:
	return NULL;
}


int
check_dir (char *dirname)
{
	struct stat str;
	char *dir;	
	int ret;
	
	dir = home_path (dirname);
	CHECK_J (dir != NULL, error_r);
	
	if (stat (dir, &str) == 0) {
		if (S_ISDIR (str.st_mode)) {
			if (str.st_uid != getuid())
				ui_output_info ("Check %s owner.", dir);

			else if (str.st_mode & S_IRWXG & S_IRWXO)
				ui_output_info ("Insecure %s permissions", dir);
		}

		else {
			ui_output_err ("%s is not a directory.", dir);
			return ERR;
		}
	}
	
	else if (errno == ENOENT) {
		ret = mkdir (dir, S_IRUSR | S_IWUSR | S_IXUSR);
		if (ret != 0) {
			ui_output_err ("Cannot create %s", dir);
			return ERR;
		}
		ui_output_info ("%s created.", dir);
	}
	
	else
		CHECK (errno != EACCES);
	
	free (dir);
	return OK;
	
error:
	free (dir);
error_r:		
	return ERR;
}
	

int 
read_tag (int fd, char *src, char *name, char *buff, size_t size)
{
	char *b;
	int i=0, j=0, k=0, t=0, c=0, nk=0, cl=0, s=0, w=0, co=0;
	ssize_t r;
	size_t name_l;
	int ret=0;

	name_l = strlen (name);
	
	if (src == NULL) {
		b = malloc (BUF_SIZE * sizeof (char));
		if (b == NULL) 
			return ERR;
		r=read (fd, b, BUF_SIZE);
		CHECK (r != -1);
	}
	else {
		b = src;
		r = strlen (src);
	}
	
	do {
		while (i < r)  {
			if (b [i] == '{' && !c) {
				nk++;
				if (nk > 1 && name_l == k && j < size) {
					buff [j] = '{';
					j++;
					ret++;
				}
			}
			else if (b [i] == '}' && !c) {
				t=0;
				s=0;
				if (nk > 0)
					nk--;
				if (name_l == k)  {
					if (nk > 0) {
						if (j < size) {
							buff [j] = '}';
							j++;
						}
					}
					else {
						if (src == NULL) {
							t = i-r+1;
							free (b);
							lseek (fd, t, SEEK_CUR);
						}
						if (j >= size) 
							return ERR_SIZE;
						
						buff [j] = '\0';	
						if (ret!=0 ) {
							if (w!=ret)
								ret = ERR_FMT;
							else if (co)
								ret = ERR_CO;
						}
						return ret;
					}
				}
				else
					k=0;
			}
			else if (nk > 0) {
				if (name_l == k && j < size) {
					if (isspace (b[i]) && !c) {
						s=0;
					}
					else if (nk > 1 || b[i] != '"' || cl) {
						buff [j] = b[i];
						j++;

						if (nk == 1) 
							if (s == 0) {
								w++;
								s = 1;
							}
					}
				}
										
				if (b[i] == '"') {
					if (nk == 1) 
						co = 1;
					c=!c;
					cl=1;
				}
				else
					cl=0;
			}
			else if (isspace ((int) b[i])) {
				t=0;
				if (name_l != k) {	
					k = 0; 
				}
			}
			else {
				if (k == name_l)
					k=0;
			
				if (name[k] == b[i] && !t) {
					k++;
					if (k == name_l)
						t=1;
				}	
				 
				else {
					t=1; 
					k=0;
				}
			}
			i++;		
		}
		
		if (src != NULL)
			r=0;
		else {
			r = read (fd, b, BUF_SIZE);
			CHECK (r != -1);
			i=0;
		}
		
	} while (r);

	if (src == NULL)
		free (b);
	
	return ERR_TAG;

error:
	free (b);
	return ERR;	
}


int 
write_tag (int fd, char *name, char *buff, size_t tabs)
{
	ssize_t r;
	int i = 0, j = 0, c = 0, d = 0;
	size_t s=0;
	char *b, *t;
	char str[] = "{\n";
	char str2[] = " { ";
	char str3[] = " }\n";
	char str4[] = "\"";
	char str6[] = "}\n\n";	
	char str7[] = "\n";
	
	t = malloc ((tabs + 1) * sizeof (char));
	if (t == NULL)
		return ERR;
	
	while (i < tabs) {
		t [i] = '\t';
		i++;
	}
	t [i] = '\0';
			
	if (name != NULL) {
		s = strlen (name) + tabs + 1;

		if (buff == NULL) {
			s += strlen (str7) + tabs + strlen (str);	
			b = malloc (s * sizeof (char));
			if (b == NULL) {
				free (t);
				return ERR;
			}
			strcpy (b, t);
			strcat (b, name);
			strcat (b, str7);
			strcat (b, t);
			strcat (b, str);
		}
		else {	
			s+=strlen (str2) + strlen (str3) + 2*strlen (buff) + 2;
			b = malloc (s * sizeof (char));
			if (b == NULL) {
				free (t);
				return ERR;
			}
			
			i = 0;
			while (buff [i] != '\0')
			{
				if (buff [i]=='{' || buff [i]=='}' || 
				    buff [i]==' ')
					c = 1;

				else if (buff [i] == '"')
					d = 1;
				
				i++;	
			}
	
			strcpy (b, t);
			strcat (b, name);
			strcat (b, str2);
	
			if (c) 
				strcat (b, str4);
	
			if (d) {
				i = 0;
				j = strlen (b);
				while (buff [i] != '\0') {
					if (buff [i] == '"') {
						b[j] = '"';
						j++;
						b[j] = '"';
						j++;
					}
					else {
						b[j] = buff [i];
						j++;
					}
					
					i++;
				}
				b [j] = '\0';
			}

			else 
				strcat (b, buff);
				
			if (c)
				strcat (b, str4);
						
			strcat (b, str3);
		}
	}
	else {
		s = strlen (str6) + tabs + 1;
		b = malloc (s * sizeof (char));
		if (b == NULL) {
			free (t);
			return ERR;
		}
		strcpy (b, t);
		strcat (b, str6); 
	}
	
	free (t);

	r = write (fd, b, strlen (b));
	if (r != strlen (b)) {
		free (b);
		return ERR;
	}
	
	free (b);	
	return OK;
}

char *
ipv4_to_string (ip_t ip)
{
	struct in_addr in;

	in.s_addr = htonl (ip);
	return inet_ntoa (in);
}

int
utf8_encode (const char *src, size_t mlen, char *utf8s, size_t utf8smlen) 
{
	size_t utf8l, r;

	utf8l = utf8smlen;

#ifdef HAVE_ICONV
	if (cur_to_utf8 != (iconv_t) -1)  {
		while (mlen > 0 && utf8smlen > 0) {
			r = iconv(cur_to_utf8, (char **)&src, &mlen, &utf8s,
			          &utf8smlen);
			if (r==(size_t)-1 && errno==EILSEQ && utf8smlen>0) {
				*utf8s++ = '?'; 
				src++;
				mlen--; 
				utf8smlen--;
				ui_output_err ("Illegal sequence in charset "
				               "conversion.");
			} 
			else 
				break;
			
			iconv(cur_to_utf8, NULL, NULL, NULL, NULL);
		}
		
		utf8l-=utf8smlen;
	}
	else {
#endif
		if (mlen < utf8smlen)
			utf8l=mlen;
		else
			utf8l=utf8smlen;
		strncpy (utf8s, src, utf8l);
#ifdef HAVE_ICONV
	}
#endif
	
	return utf8l;
}

char *
utf8_wencode (const wchar_t *src)
{
	char *dest;
	size_t n;

	n = wcstombs (NULL, src, 0) + 1;
	dest = malloc (n);
	wcstombs (dest, src, n);

	if (cur_to_utf8 != (iconv_t) -1)
	{
		char *local = dest;
		size_t len = strlen (local);
		size_t sz = 4 * len + 1;
		dest = malloc (sz);
		len = utf8_encode (local, len, dest, sz);
		free (local);
		dest[len] = '\0';
	}

	return dest;
}

int
utf8_decode (const char *src, size_t mlen, char *dest, size_t destmlen)
{
	size_t destl, r;

	destl = destmlen;

#ifdef HAVE_ICONV
	if (utf8_to_cur != (iconv_t) -1) {
		while (mlen > 0 && destmlen > 0) {
			r = iconv(utf8_to_cur, (char **)&src, &mlen, &dest,
			          &destmlen);
			if (r==(size_t)-1 && errno==EILSEQ && destmlen>0) {
				*dest++ = '?';
				src++;
				mlen--;
				destmlen--;
			}
			else
				break;
			
			iconv(utf8_to_cur, NULL, NULL, NULL, NULL);
	        }
	
		destl-=destmlen;
	}
	else {
#endif
		if (mlen < destmlen)  
			destl=mlen; 
		else
			destl=destmlen;
		strncpy (dest, src, destl);
#ifdef HAVE_ICONV
	}
#endif
	
	return destl;
}

wchar_t *
utf8_wdecode (const char *src)
{
	wchar_t *dest;
	size_t n;

	if (utf8_to_cur == (iconv_t) -1) {
		n = mbstowcs (NULL, src, 0) + 1;
		dest = malloc (n * sizeof(wchar_t));
		mbstowcs (dest, src, n);
	}
	else {
		size_t len = strlen (src);
		char *local = malloc (len + 1);
		len = utf8_decode (src, len, local, len + 1);
		local[len] = '\0';
		n = mbstowcs (NULL, local, 0) + 1;
		dest = malloc (n * sizeof(wchar_t));
		mbstowcs (dest, local, n);
		free (local);
	}

	return dest;
}

int
get_encoding (void)
{
#ifdef HAVE_ICONV
	int rv = OK;
	char *local_charset = nl_langinfo (CODESET);
	if (strcmp (local_charset, "UTF-8") == 0) {
		/* no conversion needed */
		return rv;
	}
	utf8_to_cur = iconv_open (local_charset, "UTF-8");
	if (utf8_to_cur == (iconv_t) -1) {
		ui_output_err ("Converting \"%s\" to \"UTF-8\" "
		               "not supported.", local_charset);
		rv = ERR;
	}
	cur_to_utf8 = iconv_open ("UTF-8", local_charset);
	if (utf8_to_cur == (iconv_t) -1) {
		ui_output_err ("Converting \"UTF-8\" to \"%s\" "
		               "not supported.", local_charset);
		rv = ERR;
	}
#else
	int rv = ERR;
#endif
	return rv;
}

